/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2012 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, August 04, 2012
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.1, August 04, 2012
 * @author        Chittaranjan Tripathy (2007-2012)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */
import java.util.*;
import java.io.Writer;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.FileWriter;
import java.io.File;

/**
 * Description of the class
 */
public class myPhiPsiSolutionTree {

    private myProtein __best_fragment = null;
    //public int __sampled_rdc_set_number = 0;
    private int __counter_for_total_number_of_solutions = 0;
    private int __counter_for_total_number_of_final_loops = 0;
    
    private int __this_phi_psi_solution_tree_id = -1;
       
    private myInputDataAndParameterManager __this_input_data_and_parameter_manager = myInputDataAndParameterManager.getInstance();
    
    private double __min_score = Double.MAX_VALUE;
    private double __min_clash_score = Double.MAX_VALUE; // to be used later    
    private int __max_depth = 0;
    
    private myRedPencil thisRedPencil = new myRedPencil("off"); // Let's default each local red pencil off
    
    /**
     * This enum defines the types of dihedrals exist in the protein backbone
     * chain.
     */
    public static enum myDihedral {

        PHI, PSI, OMEGA
    };

    /**
     * This class represents a degree of freedom in protein backbone chain. An
     * instance of this class represent a dihedral (any of phi, psi and omega)
     * along with the residue number.
     */
    public class myDof {

        private int __residue_number;
        private myDihedral __dof;

        public myDof(int resNum, myDihedral dof) {
            __residue_number = resNum;
            __dof = dof;
        }

        /**
         * Returns the residue number of the residue for which this dihedral object
         * is defined.
         *
         * @return the residue number of the residue for which this dihedral object
         * is defined
         */
        public int getResidueNumber() {
            return __residue_number;
        }

        /**
         * Returns the dihedral type for which this dihedral object is defined.
         *
         * @return the dihedral type for which this dihedral object is defined
         */
        public myDihedral getDof() {
            return __dof;
        }

        /**
         * Returns the string representation of the object.
         * 
         * @return the string representation of the object
         */
        @Override
        public String toString() {
            return "Residue: " + getResidueNumber() + "    DOF: " + getDof();
        }
    }

    /**
     * Returns a vector of backbone dihedrals (degrees of freedom) to be computed.
     *
     * @param thisSseId the secondary structure boundary specification
     * @return a vector of backbone dihedrals (degrees of freedom) to be computed
     */
    private Vector<myDof> getDofsToModify(final mySseInfo thisSseId) {
        Vector<myDof> dofs = new Vector<myDof>();
        for (int i = thisSseId.getBeginResidueNumber(); i <= thisSseId.getEndResidueNumber(); i++) {
            dofs.add(new myDof(i, myDihedral.PHI));
            dofs.add(new myDof(i, myDihedral.PSI));
        }
        return dofs;
    }

    /**
     * Returns a copy of the best fragment over a set of DFS searches one per each
     * set of sampled RDCs.
     *
     * @return a copy of the best fragment
     */
    public myProtein getBestFragment() {
        if (__best_fragment == null)
            return null;
        else return new myProtein(__best_fragment);
    }

    /**
     * Returns the maximum search depth achieved so far while searching different
     * DFS trees (one tree for each set of sampled RDCs).
     *
     * @return the maximum search depth
     */
    public int getMaxSearchDepth() {
        return __max_depth;
    }

    /**
     * Returns the number of solutions found thus far among all searches of the
     * DFS trees (one tree for each set of sampled RDCs).
     * 
     * @return the number of solutions found thus far
     */
    public int getSolutionCounter() {
        return __counter_for_total_number_of_solutions;
    }

    public int getCurrentTreeId() {
        return __this_phi_psi_solution_tree_id;
    }
    
    public void setCurrentTreeId(int thisTreeIdentificationNumber) {
        __this_phi_psi_solution_tree_id = thisTreeIdentificationNumber;
    }    
    
    /**
     * This method invokes the recursive depth-first search based fragment
     * computation from the given set of sampled RDCs and input parameters.
     *
     * @param thisSseId the secondary structure element boundary specification
     * @param phiTypeRdcSampled the sampled phi-defining RDC set
     * @param psiTypeRdcSampled the sampled psi-defining RDC set
     * @param phiTypeRdcVecWithMissingRdcsFilled the sampled phi-defining RDC set with missing RDCs filled
     * @param psiTypeRdcVecWithMissingRdcsFilled the sampled psi-defining RDC set with missing RDCs filled
     * @param Rg the rotation matrix that specifies the ralative rotation of the first peptide plane (of the fragment) with respect to the principal order frame
     * @param Syy the diagonilized alignment tensor component
     * @param Szz the diagonilized alignment tensor component
     * @param firstPeptidePlane the first peptide plane of the fragment
     * @param phiPsiSolutionFile the file which stores the set of computed phi and psi sequences
     * @param weightInScoringFunction the relative weight used in scoring function
     * @param rightHand
     */
    public void phiPsiSolutionTreeRootCaller(final mySseInfo thisSseId,
            final Vector<myDipolarCoupling> phiTypeRdcSampled, final Vector<myDipolarCoupling> psiTypeRdcSampled,
            final Vector<myDipolarCoupling> phiTypeRdcVecWithMissingRdcsFilled, final Vector<myDipolarCoupling> psiTypeRdcVecWithMissingRdcsFilled,
            Matrix Rg, double Syy, double Szz, final myPeptidePlane firstPeptidePlane,
            Writer phiPsiSolutionFile, final double weightInScoringFunction, final boolean rightHand) {

        Vector<myDof> dofsToBeComputed = getDofsToModify(thisSseId);
        // With phiType and psiType RDCs the psi angle of the very last residue
        // cannot be computed as the psiType RDC data is not available for it.
        // Therefore, we remove the last psi from the dofsToBeComputed.
        dofsToBeComputed.remove(dofsToBeComputed.size() - 1);

        double[] ppS = new double[dofsToBeComputed.size()];
        int initialSearchDepth = 0;
        int maxSearchDepth = dofsToBeComputed.size();

//        recursiveDepthFirstSearchOfPhiPsiSolutionTree(thisSseId, phiTypeRdcSampled, psiTypeRdcSampled, phiTypeRdcVecWithMissingRdcsFilled, psiTypeRdcVecWithMissingRdcsFilled,
//            Rg, Syy, Szz, initialSearchDepth, maxSearchDepth, ppS, firstPeptidePlane, phiPsiSolutionFile, weightInScoringFunction, rightHand, dofsToBeComputed);
    }

    /**
     * Analyze the solution wrt. a scoring function. Save the best solution,
     * increment the number of solutions found thus far. Print into files the best
     * solution fragment, the most recent fragment, and the solution phi psi sequence.
     * Currently, the analysis of a solution is implemented to minimize the
     * scoring function, and doing the steric check.
     *
     * @param thisSseId the secondary structure element boundary specification
     * @param phiTypeRdcSampled the sampled phi-defining RDC set
     * @param psiTypeRdcSampled the sampled psi-defining RDC set
     * @param phiTypeRdcVecWithMissingRdcsFilled the sampled phi-defining RDC set with missing RDCs filled
     * @param psiTypeRdcVecWithMissingRdcsFilled the sampled psi-defining RDC set with missing RDCs filled
     * @param ppS an array to store a solution (a sequence of phi and psi DOFs), i.e., a path from the root to a leaf of the depth-first search tree
     * @param firstPeptidePlane the first peptide plane of the fragment
     * @param phiPsiSolutionFile the file which stores the set of computed phi and psi sequences
     * @param weightInScoringFunction the relative weight used in scoring function
     * @param dofsToBeModified the vector of dihedrals (DOFs) to be computed
     * @param Syy the diagonilized alignment tensor component
     * @param Szz the diagonilized alignment tensor component
     */
    public void analyzeThisSolution(final mySseInfo thisSseId, final Vector<myDipolarCoupling> phiTypeRdcSampled, final Vector<myDipolarCoupling> psiTypeRdcSampled,
            final Vector<myDipolarCoupling> phiTypeRdcVecWithMissingRdcsFilled, final Vector<myDipolarCoupling> psiTypeRdcVecWithMissingRdcsFilled,
            double[] ppS, final myPeptidePlane firstPeptidePlane, Writer phiPsiSolutionFile, final double weightInScoringFunction, final Vector<myDof> dofsToBeModified,
            double Syy, double Szz) {
        // Note: (1) The parameter dofsToBeModified is not currently used. Perhaps it can be used afterwards.
        //       (2) ppS contains a solution
        if (ppS.length % 2 != 1) {
            System.out.println("Error: the length of the phi-psi chain should be odd");
            System.exit(1);
        }

        // construct a vector of phi and psi from the solution
        Vector<myPhiPsi> thisSolutionVec = new Vector<myPhiPsi>(ppS.length / 2 + 1); // initial capacity
        for (int i = 0; i < ppS.length - 1; i += 2) {
            thisSolutionVec.add(new myPhiPsi(thisSseId.getBeginResidueNumber() + i / 2, "ALA", ppS[i], ppS[i + 1]));
        }
        if (thisSseId.isHelix()) {
            thisSolutionVec.add(new myPhiPsi(thisSseId.getBeginResidueNumber() + (ppS.length - 1) / 2, "ALA", ppS[ppS.length - 1], Const.psiAveHelix));
        } else { // this SSE is a strand
            thisSolutionVec.add(new myPhiPsi(thisSseId.getBeginResidueNumber() + (ppS.length - 1) / 2, "ALA", ppS[ppS.length - 1], Const.psiAveBeta));
        }

        System.out.println("~~~~~~~~~~~~~ Begin Analysis of this Solution ~~~~~~~~~~~~~");
        try {
            // do all the finalization things
            __counter_for_total_number_of_solutions++;
            phiPsiSolutionFile.write("CounterForTotalNumberOfSolutions: " + __counter_for_total_number_of_solutions + "\n<Phi,Psi>: ");
            System.out.print("CounterForTotalNumberOfSolutions: " + __counter_for_total_number_of_solutions + "\n<Phi,Psi>: ");
            phiPsiSolutionFile.write(Arrays.toString(ppS) + '\n');
            System.out.println(Arrays.toString(ppS));

            myProtein thisFragment = myProtein.buildProteinBackboneFromDihedrals(thisSolutionVec, firstPeptidePlane);
            thisFragment.print(myInputDataAndParameterManager.getInstance().getMostRecentFragmentFileForALeafOfSearchTree(), false); // false indicates that this file is being overwritten instead of being appended
            System.out.println("Printing the solution PDB fragment");
            thisFragment.print();

            double phiRmsd = 0.0, psiRmsd = 0.0;

            // Compute phi rmsd and psi rmsd
            if (thisSseId.isHelix()) {
                for (myPhiPsi thisPP : thisSolutionVec) {
                    phiRmsd += (thisPP.getPhi() - Const.phiAveHelix) * (thisPP.getPhi() - Const.phiAveHelix);
                    psiRmsd += (thisPP.getPsi() - Const.psiAveHelix) * (thisPP.getPsi() - Const.psiAveHelix);
                }
            } else if (thisSseId.isStrand()) {
                for (myPhiPsi thisPP : thisSolutionVec) {
                    phiRmsd += (thisPP.getPhi() - Const.phiAveBeta) * (thisPP.getPhi() - Const.phiAveBeta);
                    psiRmsd += (thisPP.getPsi() - Const.psiAveBeta) * (thisPP.getPsi() - Const.psiAveBeta);
                }
            }

            phiRmsd = Math.sqrt(phiRmsd / thisSolutionVec.size());
            psiRmsd = Math.sqrt(psiRmsd / thisSolutionVec.size());
            System.out.println("phiRmsd: " + phiRmsd + "    psiRmsd: " + psiRmsd);

            /*
            // Compute the phiType rdc rmsd
            System.out.println(phiTypeRdcSampled.elementAt(0).getType() + " RDC: ");
            double phiTypeRdcRmsd = 0.0;
            for (int i = 0; i < thisSolutionVec.size(); i++) {
            //System.out.println((thisSseId.getBeginResidueNumber() + i) + ": " + phiTypeRdcSampled.elementAt(i).getRdc() + "  " + phiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc() + "  " + Math.abs(phiTypeRdcSampled.elementAt(i).getRdc() - phiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc()));
            System.out.println(phiTypeRdcSampled.elementAt(i).getResidueNumber() + ": " + phiTypeRdcSampled.elementAt(i).getRdc() + "  " + phiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc() + "  " + Math.abs(phiTypeRdcSampled.elementAt(i).getRdc() - phiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc()));
            phiTypeRdcRmsd += (phiTypeRdcSampled.elementAt(i).getRdc() - phiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc()) * (phiTypeRdcSampled.elementAt(i).getRdc() - phiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc());
            }
            phiTypeRdcRmsd = Math.sqrt(phiTypeRdcRmsd / thisSolutionVec.size());
            System.out.println(phiTypeRdcSampled.elementAt(0).getType() + " RDC RMSD: " + phiTypeRdcRmsd);

            // Compute the psiType rdc rmsd
            System.out.println(psiTypeRdcSampled.elementAt(0).getType() + " RDC: ");
            double psiTypeRdcRmsd = 0.0;
            for (int i = 0; i < thisSolutionVec.size(); i++) {
            System.out.println(psiTypeRdcSampled.elementAt(i).getResidueNumber() + ": " + psiTypeRdcSampled.elementAt(i).getRdc() + "  " + psiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc() + "  " + Math.abs(psiTypeRdcSampled.elementAt(i).getRdc() - psiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc()));
            psiTypeRdcRmsd += (psiTypeRdcSampled.elementAt(i).getRdc() - psiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc()) * (psiTypeRdcSampled.elementAt(i).getRdc() - psiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc());
            }
            psiTypeRdcRmsd = Math.sqrt(psiTypeRdcRmsd / thisSolutionVec.size());
            System.out.println(psiTypeRdcSampled.elementAt(0).getType() + " RDC RMSD: " + psiTypeRdcRmsd);
             */
            myDipolarCoupling.Type phiRdcType = phiTypeRdcVecWithMissingRdcsFilled.elementAt(0).getType();
            myDipolarCoupling.Type psiRdcType = psiTypeRdcVecWithMissingRdcsFilled.elementAt(0).getType();

            Matrix rotToPOF = Matrix.identity(3, 3);
            System.out.println(phiRdcType + " RDC:");
            double phiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(thisFragment, phiTypeRdcVecWithMissingRdcsFilled,
                    rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(phiRdcType.toString()));
            System.out.println(phiRdcType + " RDC rmsd: " + phiTypeRdcRmsd);

            myProtein thisFragmentFirstRemoved = new myProtein(thisFragment);
            //thisFragmentFirstRemoved.removeResidue(thisSseId.getBeginResidueNumber());

            System.out.println(psiRdcType + " RDC:");
            double psiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(thisFragmentFirstRemoved, psiTypeRdcVecWithMissingRdcsFilled,
                    rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(psiRdcType.toString()));
            System.out.println(psiRdcType + " RDC rmsd: " + psiTypeRdcRmsd);

            System.out.println("Phi/Psi: ");
            for (myPhiPsi thisPP : thisSolutionVec) {
                System.out.println(thisPP.getResidueNumber() + "   " + Math.toDegrees(thisPP.getPhi()) + "  " + Math.toDegrees(thisPP.getPsi()));
            }
/*
            double scoreTermFromRdcs = (phiTypeRdcRmsd / myConstantsForRdcs.DmaxScaled.get(phiRdcType.toString()) +
                    psiTypeRdcRmsd / myConstantsForRdcs.DmaxScaled.get(psiRdcType.toString())) + weightInScoringFunction * (phiRmsd + psiRmsd);
*/
            myPair<Double, Integer> squaredPhiRdcDev = myAlignmentTensorEstimator.computeRdcSumOfSquaredDeviation(thisFragment, phiTypeRdcVecWithMissingRdcsFilled,
                    rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(phiRdcType.toString()));

            myPair<Double, Integer> squaredPsiRdcDev = myAlignmentTensorEstimator.computeRdcSumOfSquaredDeviation(thisFragmentFirstRemoved, psiTypeRdcVecWithMissingRdcsFilled,
                    rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(psiRdcType.toString()));

            double phiPreFactor = myConstantsForRdcs.DmaxScaled.get(phiRdcType.toString());
            double psiPreFactor = myConstantsForRdcs.DmaxScaled.get(psiRdcType.toString());
            double scoreTermFromRdcs = squaredPhiRdcDev.first() / (phiPreFactor * phiPreFactor) + squaredPsiRdcDev.first() / (psiPreFactor * psiPreFactor);
            double totalNumberOfRdcs = squaredPhiRdcDev.second() + squaredPsiRdcDev.second();
            scoreTermFromRdcs = Math.sqrt(scoreTermFromRdcs / totalNumberOfRdcs);
            System.out.println("scoreTermFromRdcs: " + scoreTermFromRdcs);
            double scoreTermFromRdcsSaved = scoreTermFromRdcs;
            scoreTermFromRdcs += weightInScoringFunction * (phiRmsd + psiRmsd);           
            System.out.println("scoreTermFromRdcsAndPhiPsi: " + scoreTermFromRdcs);

            double clashScore = Double.MAX_VALUE;
            // Check for steric clash and return the clash score            
            myStericChecker sc = new myStericChecker(1.0, 0.8, 0.4);
            clashScore = sc.checkStericClash(thisFragment);            
            System.out.println("clashScore: " + clashScore);

            // combine the scores
            double jointScore = scoreTermFromRdcs + clashScore / 20.0;
            System.out.println("jointScore: " + jointScore);

            if (jointScore < __min_score ) { //we got a better fragment save it!                
                __min_score = jointScore;
                //__min_clash_score = clashScore;
                __best_fragment = new myProtein(thisFragment);

                Writer outputBestFragment = null;
                File bestFragmentFile = new File(__this_input_data_and_parameter_manager.getBestFragmentPdbFile());
                if (bestFragmentFile.exists()) {
                    bestFragmentFile.delete();
                }
                outputBestFragment = new BufferedWriter(new FileWriter(bestFragmentFile));//, true)); //true for append mode

                outputBestFragment.write("REMARK 100 phiRmsd: " + phiRmsd + "    psiRmsd: " + psiRmsd + '\n');
                outputBestFragment.write("REMARK 100 " + phiTypeRdcSampled.elementAt(0).getType() + " RDC RMSD (including missing RDCs): " + phiTypeRdcRmsd + '\n');
                outputBestFragment.write("REMARK 100 " + psiTypeRdcSampled.elementAt(0).getType() + " RDC RMSD (including missing RDCs): " + psiTypeRdcRmsd + '\n');                
                outputBestFragment.write("REMARK 100 scoreTermFromRdcs: " + scoreTermFromRdcsSaved + '\n');
                outputBestFragment.write("REMARK 100 scoreTermFromRdcsAndPhiPsiDeviation: " + scoreTermFromRdcs + '\n');
                outputBestFragment.write("REMARK 100 clashScore: " + clashScore + '\n');
                outputBestFragment.write("REMARK 100 jointScore: " + jointScore + '\n');
                outputBestFragment.write("REMARK 100 Printing the phi and psi values below\n");
                for (myPhiPsi thisPP : thisSolutionVec) {
                    outputBestFragment.write("REMARK 100 " + thisPP.getResidueNumber() + "   " + Math.toDegrees(thisPP.getPhi()) + "  " + Math.toDegrees(thisPP.getPsi()) + '\n');
                }
                outputBestFragment.write("REMARK 100 Printing the corresponding PDB fragment\n");
                thisFragment.print(outputBestFragment);
                outputBestFragment.close();
            }
        } catch (IOException e) {
            System.out.println("An IOException is thrown while writing to the outputfile");
            e.printStackTrace();
            System.exit(1);
        }
        System.out.println("~~~~~~~~~~~~~~ End Analysis of this Solution ~~~~~~~~~~~~~~");
    }

    /**
     * This method invokes the recursive depth-first search based fragment
     * computation from the given set of sampled RDCs and input parameters.
     *
     * @param thisSseId the secondary structure element boundary specification
     * @param phiTypeRdcSampled the sampled phi-defining RDC set
     * @param psiTypeRdcSampled the sampled psi-defining RDC set
     * @param phiTypeRdcVecWithMissingRdcsFilled the sampled phi-defining RDC set with missing RDCs filled
     * @param psiTypeRdcVecWithMissingRdcsFilled the sampled psi-defining RDC set with missing RDCs filled
     * @param Rg the rotation matrix that specifies the ralative rotation of the first peptide plane (of the fragment) with respect to the principal order frame
     * @param Syy the diagonilized alignment tensor component
     * @param Szz the diagonilized alignment tensor component
     * @param firstPeptidePlane the first peptide plane of the fragment
     * @param phiPsiSolutionFile the file which stores the set of computed phi and psi sequences
     * @param weightInScoringFunction the relative weight used in scoring function
     * @param rightHand
     * @param hBondWeight
     * @param noeVec the set of NOEs used for packing the beta-sheet
     * @param partialSheetThusFar the partial sheet computed thus far
     */
    public void phiPsiSolutionTreeRootCallerForBetaStrand(final mySseInfo thisSseId,
            final Vector<myDipolarCoupling> phiTypeRdcSampled, final Vector<myDipolarCoupling> psiTypeRdcSampled,
            final Vector<myDipolarCoupling> phiTypeRdcVecWithMissingRdcsFilled, final Vector<myDipolarCoupling> psiTypeRdcVecWithMissingRdcsFilled,
            Matrix Rg, double Syy, double Szz, final myPeptidePlane firstPeptidePlane,
            Writer phiPsiSolutionFile, final double weightInScoringFunction, final boolean rightHand,
            final double hBondWeight, Vector<myNoe> noeVec, myProtein partialSheetThusFar) {

        Vector<myDof> dofsToBeComputed = getDofsToModify(thisSseId);
        // With phiType and psiType RDCs the psi angle of the very last residue
        // cannot be computed as the psiType RDC data is not available for it.
        // Therefore, we remove the last psi from the dofsToBeComputed.
        dofsToBeComputed.remove(dofsToBeComputed.size() - 1);

        double[] ppS = new double[dofsToBeComputed.size()];
        int initialSearchDepth = 0;
        int maxSearchDepth = dofsToBeComputed.size();

        recursiveDepthFirstSearchOfPhiPsiSolutionTreeForBetaStrand(thisSseId, phiTypeRdcSampled, psiTypeRdcSampled, phiTypeRdcVecWithMissingRdcsFilled, psiTypeRdcVecWithMissingRdcsFilled,
            Rg, Syy, Szz, initialSearchDepth, maxSearchDepth, ppS, firstPeptidePlane, phiPsiSolutionFile, weightInScoringFunction, rightHand, dofsToBeComputed,
            hBondWeight, noeVec, partialSheetThusFar);
    }

    /**
     * Do depth-first search of the analytic solution tree to compute the set of all
     * possible phi, psi sequences from which a set of possible fragments can be
     * constructed.
     *
     * @param thisSseId the secondary structure element boundary specification
     * @param phiTypeRdcSampled the sampled phi-defining RDC set
     * @param psiTypeRdcSampled the sampled psi-defining RDC set
     * @param phiTypeRdcVecWithMissingRdcsFilled the sampled phi-defining RDC set with missing RDCs filled
     * @param psiTypeRdcVecWithMissingRdcsFilled the sampled psi-defining RDC set with missing RDCs filled
     * @param Rg the rotation matrix that specifies the ralative rotation of the first peptide plane (of the fragment) with respect to the principal order frame
     * @param Syy the diagonilized alignment tensor component
     * @param Szz the diagonilized alignment tensor component
     * @param currSearchDepth current search depth in the depth-first search tree
     * @param maxSearchDepth maximum depth of the depth-first search tree
     * @param ppS an array to store a solution (a sequence of phi and psi DOFs), i.e., a path from the root to a leaf of the depth-first search tree
     * @param firstPeptidePlane the first peptide plane of the fragment
     * @param phiPsiSolutionFile the file which stores the set of computed phi and psi sequences
     * @param weightInScoringFunction the relative weight used in scoring function
     * @param rightHand
     * @param dofsToBeModified the vector of dihedrals (DOFs) to be computed
     * @param hBondWeight
     * @param noeVec the set of NOEs used for packing the beta-sheet
     * @param partialSheetThusFar the partial sheet computed thus far
     */
    public void recursiveDepthFirstSearchOfPhiPsiSolutionTreeForBetaStrand(final mySseInfo thisSseId, final Vector<myDipolarCoupling> phiTypeRdcSampled, final Vector<myDipolarCoupling> psiTypeRdcSampled,
            final Vector<myDipolarCoupling> phiTypeRdcVecWithMissingRdcsFilled, final Vector<myDipolarCoupling> psiTypeRdcVecWithMissingRdcsFilled,
            Matrix Rg, double Syy, double Szz, final int currSearchDepth, final int maxSearchDepth, double[] ppS, final myPeptidePlane firstPeptidePlane,
            Writer phiPsiSolutionFile, final double weightInScoringFunction, final boolean rightHand, final Vector<myDof> dofsToBeModified,
            final double hBondWeight, Vector<myNoe> noeVec, myProtein partialSheetThusFar) {

        Vector<Double> phiVec = new Vector<Double>();
        Vector<Double> psiVec = new Vector<Double>();

        if (currSearchDepth == maxSearchDepth) { // We hit the basis of the recursion, i.e., we reached a leaf (NIL) node after computing all phi and psi values
            analyzeThisSolutionForBetaStrand(thisSseId, phiTypeRdcSampled, psiTypeRdcSampled, phiTypeRdcVecWithMissingRdcsFilled, psiTypeRdcVecWithMissingRdcsFilled,
                    ppS, firstPeptidePlane, phiPsiSolutionFile, weightInScoringFunction, dofsToBeModified, Syy, Szz,
                    hBondWeight, noeVec, partialSheetThusFar);
        } else { //currSearchDepth < maxSearchDepth. So recurse down the tree.
            if (dofsToBeModified.elementAt(currSearchDepth).getDof() == myDihedral.PHI) { //this dof is PHI
                phiVec = myForKin.computePhi(Rg, Syy, Szz, phiTypeRdcSampled.elementAt(currSearchDepth / 2), thisSseId.isHelix()); // We need phiType RDC for residue i to solve for Phi_i

                if (phiVec == null || phiVec.isEmpty()) {
                    if (thisRedPencil.isOn())
                        System.out.println("There exists no solution for phi for the sampled rdc " + phiTypeRdcSampled.elementAt(currSearchDepth / 2));                    
                }

                for (double thisPhi : phiVec) { //for each phi recurse down the tree searching for the psi
                    ppS[currSearchDepth] = thisPhi;
                    recursiveDepthFirstSearchOfPhiPsiSolutionTreeForBetaStrand(thisSseId, phiTypeRdcSampled, psiTypeRdcSampled, phiTypeRdcVecWithMissingRdcsFilled, psiTypeRdcVecWithMissingRdcsFilled,
                            Rg, Syy, Szz, currSearchDepth + 1, dofsToBeModified.size(), ppS, firstPeptidePlane, phiPsiSolutionFile, weightInScoringFunction, rightHand, dofsToBeModified,
                            hBondWeight, noeVec, partialSheetThusFar);
                }
            } else if (dofsToBeModified.elementAt(currSearchDepth).getDof() == myDihedral.PSI) { //this dof is PSI
                double phiAtThisResidue = ppS[currSearchDepth - 1];
                psiVec = myForKin.computePsi(Rg, Syy, Szz, psiTypeRdcSampled.elementAt(currSearchDepth / 2 + 1), phiAtThisResidue, thisSseId.isHelix()); // We need psiType rdc for residue (i+1) to solve for Psi_i

                if (psiVec == null || psiVec.isEmpty()) {
                    if (thisRedPencil.isOn())
                        System.out.println("There exists no solution for psi for the sampled rdc " + psiTypeRdcSampled.elementAt(currSearchDepth / 2 + 1));                    
                }

                for (double thisPsi : psiVec) { // for each psi recurse down the tree searching for the next phi
                    if (currSearchDepth / 2 + 1 > __max_depth) { // update the the maximum depth (in terms of residues) achieved
                        __max_depth = currSearchDepth / 2 + 1;
                    }
                    ppS[currSearchDepth] = thisPsi;
                    double thisPhi = ppS[currSearchDepth - 1];
                    //myPhiPsi pp = new myPhiPsi();

                    //System.out.println("thisPhi: " + thisPhi + "    thisPsi: " + thisPsi); // TODO: JUL04
                    Matrix newRg = myForKin.newRG__(thisPhi, thisPsi, Rg, rightHand); //compute a new rotation matrix rg
                    recursiveDepthFirstSearchOfPhiPsiSolutionTreeForBetaStrand(thisSseId, phiTypeRdcSampled, psiTypeRdcSampled, phiTypeRdcVecWithMissingRdcsFilled, psiTypeRdcVecWithMissingRdcsFilled,
                            newRg, Syy, Szz, currSearchDepth + 1, dofsToBeModified.size(), ppS, firstPeptidePlane, phiPsiSolutionFile, weightInScoringFunction, rightHand, dofsToBeModified,
                            hBondWeight, noeVec, partialSheetThusFar);
                }
            } else {
                System.out.println("Error: Illegal DOF");
                System.exit(1);
            }
        }
    }

    /**
     * Analyze the solution wrt. a scoring function. Save the best solution,
     * increment the number of solutions found thus far. Print into files the best
     * solution fragment, the most recent fragment, and the solution phi psi sequence.
     * Currently, the analysis of a solution is implemented to minimize the
     * scoring function, and doing the steric check.
     *
     * @param thisSseId the secondary structure element boundary specification
     * @param phiTypeRdcSampled the sampled phi-defining RDC set
     * @param psiTypeRdcSampled the sampled psi-defining RDC set
     * @param phiTypeRdcVecWithMissingRdcsFilled the sampled phi-defining RDC set with missing RDCs filled
     * @param psiTypeRdcVecWithMissingRdcsFilled the sampled psi-defining RDC set with missing RDCs filled
     * @param ppS an array to store a solution (a sequence of phi and psi DOFs), i.e., a path from the root to a leaf of the depth-first search tree
     * @param firstPeptidePlane the first peptide plane of the fragment
     * @param phiPsiSolutionFile the file which stores the set of computed phi and psi sequences
     * @param weightInScoringFunction the relative weight used in scoring function
     * @param dofsToBeModified the vector of dihedrals (DOFs) to be computed
     * @param Syy the diagonilized alignment tensor component
     * @param Szz the diagonilized alignment tensor component
     * @param hBondWeight
     * @param noeVec the set of NOEs used for packing the beta-sheet
     * @param partialSheetThusFar the partial sheet computed thus far
     */
    public void analyzeThisSolutionForBetaStrand(final mySseInfo thisSseId, final Vector<myDipolarCoupling> phiTypeRdcSampled, final Vector<myDipolarCoupling> psiTypeRdcSampled,
            final Vector<myDipolarCoupling> phiTypeRdcVecWithMissingRdcsFilled, final Vector<myDipolarCoupling> psiTypeRdcVecWithMissingRdcsFilled,
            double[] ppS, final myPeptidePlane firstPeptidePlane, Writer phiPsiSolutionFile, final double weightInScoringFunction, final Vector<myDof> dofsToBeModified,
            double Syy, double Szz,
            final double hBondWeight, Vector<myNoe> noeVec, myProtein partialSheetThusFar) {
        // Note: (1) The parameter dofsToBeModified is not currently used. Perhaps it can be used afterwards.
        //       (2) ppS contains a solution
        if (ppS.length % 2 != 1) {
            System.out.println("Error: the length of the phi-psi chain should be odd");
            System.exit(1);
        }

        // construct a vector of phi and psi from the solution
        Vector<myPhiPsi> thisSolutionVec = new Vector<myPhiPsi>(ppS.length / 2 + 1); // initial capacity
        for (int i = 0; i < ppS.length - 1; i += 2) {
            thisSolutionVec.add(new myPhiPsi(thisSseId.getBeginResidueNumber() + i / 2, "ALA", ppS[i], ppS[i + 1]));
        }
        if (thisSseId.isHelix()) {
            thisSolutionVec.add(new myPhiPsi(thisSseId.getBeginResidueNumber() + (ppS.length - 1) / 2, "ALA", ppS[ppS.length - 1], Const.psiAveHelix));
        } else { // this SSE is a strand
            thisSolutionVec.add(new myPhiPsi(thisSseId.getBeginResidueNumber() + (ppS.length - 1) / 2, "ALA", ppS[ppS.length - 1], Const.psiAveBeta));
        }

        System.out.println("~~~~~~~~~~~~~ Begin Analysis of this Solution ~~~~~~~~~~~~~");
        try {
            // do all the finalization things
            __counter_for_total_number_of_solutions++;
            phiPsiSolutionFile.write("CounterForTotalNumberOfSolutions: " + __counter_for_total_number_of_solutions + "\n<Phi,Psi>: ");
            System.out.print("CounterForTotalNumberOfSolutions: " + __counter_for_total_number_of_solutions + "\n<Phi,Psi>: ");
            phiPsiSolutionFile.write(Arrays.toString(ppS) + '\n');
            System.out.println(Arrays.toString(ppS));

            myProtein thisFragment = myProtein.buildProteinBackboneFromDihedrals(thisSolutionVec, firstPeptidePlane);
            thisFragment.print(myInputDataAndParameterManager.getInstance().getMostRecentFragmentFileForALeafOfSearchTree(), false); // false indicates that this file is being overwritten instead of being appended
            System.out.println("Printing the solution PDB fragment");
            thisFragment.print();

            double phiRmsd = 0.0, psiRmsd = 0.0;

            // Compute phi rmsd and psi rmsd
            if (thisSseId.isHelix()) {
                for (myPhiPsi thisPP : thisSolutionVec) {
                    phiRmsd += (thisPP.getPhi() - Const.phiAveHelix) * (thisPP.getPhi() - Const.phiAveHelix);
                    psiRmsd += (thisPP.getPsi() - Const.psiAveHelix) * (thisPP.getPsi() - Const.psiAveHelix);
                }
            } else if (thisSseId.isStrand()) {
                for (myPhiPsi thisPP : thisSolutionVec) {
                    phiRmsd += (thisPP.getPhi() - Const.phiAveBeta) * (thisPP.getPhi() - Const.phiAveBeta);
                    psiRmsd += (thisPP.getPsi() - Const.psiAveBeta) * (thisPP.getPsi() - Const.psiAveBeta);
                }
            }

            phiRmsd = Math.sqrt(phiRmsd / thisSolutionVec.size());
            psiRmsd = Math.sqrt(psiRmsd / thisSolutionVec.size());
            System.out.println("phiRmsd: " + phiRmsd + "    psiRmsd: " + psiRmsd);

            /*
            // Compute the phiType rdc rmsd
            System.out.println(phiTypeRdcSampled.elementAt(0).getType() + " RDC: ");
            double phiTypeRdcRmsd = 0.0;
            for (int i = 0; i < thisSolutionVec.size(); i++) {
            //System.out.println((thisSseId.getBeginResidueNumber() + i) + ": " + phiTypeRdcSampled.elementAt(i).getRdc() + "  " + phiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc() + "  " + Math.abs(phiTypeRdcSampled.elementAt(i).getRdc() - phiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc()));
            System.out.println(phiTypeRdcSampled.elementAt(i).getResidueNumber() + ": " + phiTypeRdcSampled.elementAt(i).getRdc() + "  " + phiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc() + "  " + Math.abs(phiTypeRdcSampled.elementAt(i).getRdc() - phiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc()));
            phiTypeRdcRmsd += (phiTypeRdcSampled.elementAt(i).getRdc() - phiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc()) * (phiTypeRdcSampled.elementAt(i).getRdc() - phiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc());
            }
            phiTypeRdcRmsd = Math.sqrt(phiTypeRdcRmsd / thisSolutionVec.size());
            System.out.println(phiTypeRdcSampled.elementAt(0).getType() + " RDC RMSD: " + phiTypeRdcRmsd);

            // Compute the psiType rdc rmsd
            System.out.println(psiTypeRdcSampled.elementAt(0).getType() + " RDC: ");
            double psiTypeRdcRmsd = 0.0;
            for (int i = 0; i < thisSolutionVec.size(); i++) {
            System.out.println(psiTypeRdcSampled.elementAt(i).getResidueNumber() + ": " + psiTypeRdcSampled.elementAt(i).getRdc() + "  " + psiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc() + "  " + Math.abs(psiTypeRdcSampled.elementAt(i).getRdc() - psiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc()));
            psiTypeRdcRmsd += (psiTypeRdcSampled.elementAt(i).getRdc() - psiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc()) * (psiTypeRdcSampled.elementAt(i).getRdc() - psiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc());
            }
            psiTypeRdcRmsd = Math.sqrt(psiTypeRdcRmsd / thisSolutionVec.size());
            System.out.println(psiTypeRdcSampled.elementAt(0).getType() + " RDC RMSD: " + psiTypeRdcRmsd);
             */
            myDipolarCoupling.Type phiRdcType = phiTypeRdcVecWithMissingRdcsFilled.elementAt(0).getType();
            myDipolarCoupling.Type psiRdcType = psiTypeRdcVecWithMissingRdcsFilled.elementAt(0).getType();

            Matrix rotToPOF = Matrix.identity(3, 3);
            System.out.println(phiRdcType + " RDC:");
            double phiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(thisFragment, phiTypeRdcVecWithMissingRdcsFilled,
                    rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(phiRdcType.toString()));
            System.out.println(phiRdcType + " RDC rmsd: " + phiTypeRdcRmsd);

            myProtein thisFragmentFirstRemoved = new myProtein(thisFragment);
            //thisFragmentFirstRemoved.removeResidue(thisSseId.getBeginResidueNumber());

            System.out.println(psiRdcType + " RDC:");
            double psiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(thisFragmentFirstRemoved, psiTypeRdcVecWithMissingRdcsFilled,
                    rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(psiRdcType.toString()));
            System.out.println(psiRdcType + " RDC rmsd: " + psiTypeRdcRmsd);

            System.out.println("Phi/Psi: ");
            for (myPhiPsi thisPP : thisSolutionVec) {
                System.out.println(thisPP.getResidueNumber() + "   " + Math.toDegrees(thisPP.getPhi()) + "  " + Math.toDegrees(thisPP.getPsi()));
            }
/*
            double scoreTermFromRdcs = (phiTypeRdcRmsd / myConstantsForRdcs.DmaxScaled.get(phiRdcType.toString()) +
                    psiTypeRdcRmsd / myConstantsForRdcs.DmaxScaled.get(psiRdcType.toString())) + weightInScoringFunction * (phiRmsd + psiRmsd);
*/            
            myPair<Double, Integer> squaredPhiRdcDev = myAlignmentTensorEstimator.computeRdcSumOfSquaredDeviation(thisFragment, phiTypeRdcVecWithMissingRdcsFilled,
                    rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(phiRdcType.toString()));

            myPair<Double, Integer> squaredPsiRdcDev = myAlignmentTensorEstimator.computeRdcSumOfSquaredDeviation(thisFragmentFirstRemoved, psiTypeRdcVecWithMissingRdcsFilled,
                    rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(psiRdcType.toString()));

            double phiPreFactor = myConstantsForRdcs.DmaxScaled.get(phiRdcType.toString());
            double psiPreFactor = myConstantsForRdcs.DmaxScaled.get(psiRdcType.toString());
            double scoreTermFromRdcs = squaredPhiRdcDev.first() / (phiPreFactor * phiPreFactor) + squaredPsiRdcDev.first() / (psiPreFactor * psiPreFactor);
            double totalNumberOfRdcs = squaredPhiRdcDev.second() + squaredPsiRdcDev.second();
            scoreTermFromRdcs = Math.sqrt(scoreTermFromRdcs / totalNumberOfRdcs);
            System.out.println("scoreTermFromRdcs: " + scoreTermFromRdcs);
            scoreTermFromRdcs += weightInScoringFunction * (phiRmsd + psiRmsd);
            System.out.println("scoreTermFromRdcsAndPhiPsi: " + scoreTermFromRdcs);            

            myNewPacker pk = new myNewPacker();
            pk.setGlobalBestOrBestFirst(true);
            myTriple<myProtein, myProtein, Double> scoreFromCenterFit = pk.__packUsingNoes__(partialSheetThusFar, thisFragment, noeVec, null);//pk.centerFit__(partialSheetThusFar, thisFragment, noeVec);

            if (scoreFromCenterFit.first() == null || scoreFromCenterFit.second() == null) {
                System.out.println("Warning: In AnalyzeSolutionForBetaStrand() the fragment is null null coz score is >= 5");
                System.out.println("Therefore, discarding this solution and returning from analyzeThisSolutionForBetaStrand");
                System.out.println("~~~~~~~~~~~~~~ End Analysis of this Solution ~~~~~~~~~~~~~~");
                return;
            }

            double scoreTermFromSheetPacking = scoreFromCenterFit.third();
            System.out.println("scoreTermFromSheetPacking: " + scoreTermFromSheetPacking);

            double jointScore = scoreTermFromRdcs + scoreTermFromSheetPacking;
            System.out.println("jointScore: " + jointScore);

            if (jointScore < __min_score) { //we got a better fragment save it!                
                __min_score = jointScore;
                //__min_clash_score = clashScore;
                __best_fragment = new myProtein(scoreFromCenterFit.second());

                Writer outputBestFragment = null;
                File bestFragmentFile = new File(__this_input_data_and_parameter_manager.getBestFragmentPdbFile());
                if (bestFragmentFile.exists()) {
                    bestFragmentFile.delete();
                }
                outputBestFragment = new BufferedWriter(new FileWriter(bestFragmentFile));//, true)); //true for append mode

                outputBestFragment.write("REMARK 100 phiRmsd: " + phiRmsd + "    psiRmsd: " + psiRmsd + '\n');
                outputBestFragment.write("REMARK 100 " + phiTypeRdcSampled.elementAt(0).getType() + " RDC RMSD (including missing RDCs): " + phiTypeRdcRmsd + '\n');
                outputBestFragment.write("REMARK 100 " + psiTypeRdcSampled.elementAt(0).getType() + " RDC RMSD (including missing RDCs): " + psiTypeRdcRmsd + '\n');
                outputBestFragment.write("REMARK 100 scoreTermFromSheetPacking: " + scoreTermFromSheetPacking + '\n');
                outputBestFragment.write("REMARK 100 scoreTermFromRdcsAndPhiPsiDeviation: " + scoreTermFromRdcs + '\n');
                outputBestFragment.write("REMARK 100 jointScore: " + jointScore + '\n');               
                outputBestFragment.write("REMARK 100 Printing the phi and psi values below\n");
                for (myPhiPsi thisPP : thisSolutionVec) {
                    outputBestFragment.write("REMARK 100 " + thisPP.getResidueNumber() + "   " + Math.toDegrees(thisPP.getPhi()) + "  " + Math.toDegrees(thisPP.getPsi()) + '\n');
                }
                outputBestFragment.write("REMARK 100 Printing the corresponding PDB fragment\n");
                //thisFragment.print(outputBestFragment);
                myProtein mergedProtein = myProtein.cloneAndMergeTwoProteinFragments(scoreFromCenterFit.first(), scoreFromCenterFit.second());
                mergedProtein.print(outputBestFragment);
                outputBestFragment.close();
            }
        } catch (IOException e) {
            System.out.println("An IOException is thrown while writing to the outputfile");
            e.printStackTrace();
            System.exit(1);
        }
        System.out.println("~~~~~~~~~~~~~~ End Analysis of this Solution ~~~~~~~~~~~~~~");
    }


/////////////////////////DESIGN SPACE///////////////////////////////////////////

    
    // Begin note: The following code is not in use now.
    private static final Random __rr_for_dihedral = new Random(myMiscConstants.CRTRandomSeed);
    private double __sampling_width = Math.toRadians(10);

    private double getSampledAngle(double angle) {
        return Math.toRadians(angle) + __rr_for_dihedral.nextGaussian() * __sampling_width;
    }
    // End note

    // Get the parameters from the loop configuration manager
    private myLoopConfigurationManager thisLoopConfigurationManager = new myLoopConfigurationManager();

    private myRama ramaGeneral = thisLoopConfigurationManager.ramaGeneral;
    private myRama ramaGly = thisLoopConfigurationManager.ramaGly;
    private myRama ramaPrePro = thisLoopConfigurationManager.ramaPrePro;
    private myRama ramaPro = thisLoopConfigurationManager.ramaPro;

    private myProteinSequence thisProteinSeq = thisLoopConfigurationManager.__this_protein_sequence;

    private myProtein __global_fold_in_pof = thisLoopConfigurationManager.__global_fold_in_pof;

    private final int __begin_residue_number = thisLoopConfigurationManager.getBeginResidueNumber();
    private final int __end_residue_number = thisLoopConfigurationManager.getEndResidueNumber();

    private myResidue __anchor_residue = new myResidue(__global_fold_in_pof.residueAt(__end_residue_number));

    private myProtein __global_fold_in_pof_without_loops_and_anchor = thisLoopConfigurationManager.getGlolabFoldInPofWithoutLoopsAndAnchor();
    private double __clash_score_for_global_fold_in_pof_without_loops_and_anchor = thisLoopConfigurationManager.getClashScoreForGlobalFoldInPofWithoutLoopsAndAnchor();
    private int __number_of_atoms_in_global_fold_in_pof_without_loops_and_anchor = thisLoopConfigurationManager.getNumberOfAtomsInGlobalFoldInPofWithoutLoopsAndAnchor();

    private double __grid_resolution_for_phi_psi = thisLoopConfigurationManager.getGridResolutionForPhiPsi();
    private final int __depth_at_which_steric_checker_turns_on = thisLoopConfigurationManager.getDepthAtWhichStericCheckerTurnsOn();

    private final double __closure_distance_threshold = thisLoopConfigurationManager.getClosureDistanceThreshold();
    private final double __rms_distance_wrt_reference = 0.0; // this parameter is not used //thisLoopConfigurationManager.getRmsDistanceWrtReference();
    private Vector<myNoe> __loop_noes = thisLoopConfigurationManager.__loop_noe_vec;
    private Vector<myDihedralRestraint> __loop_dihedral_restraints = thisLoopConfigurationManager.__talos_dihedral_restraints;
    
    


    // TODO: A note only, nothing to be done... A ditto of the function below is there in the new version of the myNewPacker.    
    /**
     * Compute the individual steric clash contribution to the overall clash score
     * when two proteins are combined.
     *
     * @param p1 the first protein
     * @param p2 the second protein
     * @return the score (after summing and normalizing) contribution from individual
     * proteins without the clash between the two being considered
     */
    private double individualScore(myProtein p1, myProtein p2) {
        int numberOfAtomsInProtein1 = p1.numberOfAtoms();
        int numberOfAtomsInProtein2 = p2.numberOfAtoms();

        myStericChecker sc1 = new myStericChecker(1.0, 0.8, 0.4);
        double clashScore1 = sc1.checkStericClash(p1);
        //System.out.println("clashScore1: " + clashScore1);
        int clashCount1 = (int) Math.ceil(clashScore1 * numberOfAtomsInProtein1 / 1000.0);

        myStericChecker sc2 = new myStericChecker(1.0, 0.8, 0.4);
        double clashScore2 = sc2.checkStericClash(p2);
        //System.out.println("clashScore2: " + clashScore2);
        int clashCount2 = (int) Math.ceil(clashScore2 * numberOfAtomsInProtein2 / 1000.0);

        double combinedClashScoreFromIndividualContribution = (clashCount1 + clashCount2 + 0.0) / (numberOfAtomsInProtein1 + numberOfAtomsInProtein2 + 0.0) * 1000;

        return combinedClashScoreFromIndividualContribution;
    }

    private double individualScorePreComputed(myProtein p2) {
        int numberOfAtomsInProtein1 = __number_of_atoms_in_global_fold_in_pof_without_loops_and_anchor;
        double clashScore1 = __clash_score_for_global_fold_in_pof_without_loops_and_anchor;
        //System.out.println("clashScore1: " + clashScore1);
        int clashCount1 = (int) Math.ceil(clashScore1 * numberOfAtomsInProtein1 / 1000.0);

        int numberOfAtomsInProtein2 = p2.numberOfAtoms();
        myStericChecker sc2 = new myStericChecker(1.0, 0.8, 0.4);
        double clashScore2 = sc2.checkStericClash(p2);
        //System.out.println("clashScore2: " + clashScore2);
        int clashCount2 = (int) Math.ceil(clashScore2 * numberOfAtomsInProtein2 / 1000.0);

        double combinedClashScoreFromIndividualContribution = (clashCount1 + clashCount2 + 0.0) / (numberOfAtomsInProtein1 + numberOfAtomsInProtein2 + 0.0) * 1000;

        return combinedClashScoreFromIndividualContribution;
    }

    private double individualScorePreComputed(myProtein p2, double p2ClashScore) {
        int numberOfAtomsInProtein1 = __number_of_atoms_in_global_fold_in_pof_without_loops_and_anchor;              
        double clashScore1 = __clash_score_for_global_fold_in_pof_without_loops_and_anchor;        
        int clashCount1 = (int) Math.ceil(clashScore1 * numberOfAtomsInProtein1 / 1000.0);

        int numberOfAtomsInProtein2 = p2.numberOfAtoms();
        int clashCount2 = (int) Math.ceil(p2ClashScore * numberOfAtomsInProtein2 / 1000.0);

        double combinedClashScoreFromIndividualContribution = (clashCount1 + clashCount2 + 0.0) / (numberOfAtomsInProtein1 + numberOfAtomsInProtein2 + 0.0) * 1000;
        return combinedClashScoreFromIndividualContribution;
    }

    /**
     * This method invokes the recursive depth-first search based fragment
     * computation from the given set of sampled RDCs and input parameters.
     *
     * @param thisSseId the secondary structure element boundary specification
     * @param phiTypeRdcSampled the sampled phi-defining RDC set
     * @param psiTypeRdcSampled the sampled psi-defining RDC set
     * @param phiTypeRdcVecWithMissingRdcsFilled the sampled phi-defining RDC set with missing RDCs filled
     * @param psiTypeRdcVecWithMissingRdcsFilled the sampled psi-defining RDC set with missing RDCs filled
     * @param Rg the rotation matrix that specifies the ralative rotation of the first peptide plane (of the fragment) with respect to the principal order frame
     * @param Syy the diagonilized alignment tensor component
     * @param Szz the diagonilized alignment tensor component
     * @param firstPeptidePlane the first peptide plane of the fragment
     * @param phiPsiSolutionFile the file which stores the set of computed phi and psi sequences
     * @param weightInScoringFunction the relative weight used in scoring function
     * @param rightHand
     */
    public void phiPsiSolutionTreeRootCallerForLoop(final mySseInfo thisSseId,
            final Vector<myDipolarCoupling> phiTypeRdcSampled, final Vector<myDipolarCoupling> psiTypeRdcSampled,
            final Vector<myDipolarCoupling> phiTypeRdcVecWithMissingRdcsFilled, final Vector<myDipolarCoupling> psiTypeRdcVecWithMissingRdcsFilled,
            Matrix Rg, double Syy, double Szz, final myPeptidePlane firstPeptidePlane,
            Writer phiPsiSolutionFile, final double weightInScoringFunction, final boolean rightHand) {

        Vector<myDof> dofsToBeComputed = getDofsToModify(thisSseId);
        // With phiType and psiType RDCs the psi angle of the very last residue
        // cannot be computed as the psiType RDC data is not available for it.
        // Therefore, we remove the last psi from the dofsToBeComputed.
        dofsToBeComputed.remove(dofsToBeComputed.size() - 1);

        double[] ppS = new double[dofsToBeComputed.size()];
        int initialSearchDepth = 0;
        int maxSearchDepth = dofsToBeComputed.size();

        if (thisSseId.isHelix() || thisSseId.isStrand()) {
            System.out.println("Error: a loop specification must be provided");
            System.exit(1);
        }

        recursiveDepthFirstSearchOfPhiPsiSolutionTreeForLoop(thisSseId, phiTypeRdcSampled, psiTypeRdcSampled, phiTypeRdcVecWithMissingRdcsFilled, psiTypeRdcVecWithMissingRdcsFilled,
            Rg, Syy, Szz, initialSearchDepth, maxSearchDepth, ppS, firstPeptidePlane, phiPsiSolutionFile, weightInScoringFunction, rightHand, dofsToBeComputed);
    }
    
    private double angDelta = Math.toRadians(5); // TODO: set it as a parameter

    private void removeMultiplicity(Vector<Double> v) {
		
		Collections.sort(v);
        
		if (v.size() <= 1)
            return;

        if (v.size() == 2) {
            if (Math.abs(v.elementAt(0) - v.elementAt(1)) < angDelta)
                v.remove(1);
        }

        if (v.size() == 3) {
            double d1 = Math.abs(v.elementAt(0) - v.elementAt(1));
            double d2 = Math.abs(v.elementAt(1) - v.elementAt(2));

            if (d1 < d2) {
                if (d1 < angDelta) {
                    v.set(1, new Double(v.elementAt(0)));
                }
            } else if (d2 < d1) {
                if (d2 < angDelta) {
                    v.set(1, new Double(v.elementAt(2)));
                }
            }
        }

        if (v.size() == 4) {

            double d1 = Math.abs(v.elementAt(0) - v.elementAt(1));
            double d2 = Math.abs(v.elementAt(1) - v.elementAt(2));

            if (d1 < d2) {
                if (d1 < angDelta) {
                    v.set(1, new Double(v.elementAt(0)));
                }
            } else if (d2 < d1) {
                if (d2 < angDelta) {
                    v.set(1, new Double(v.elementAt(2)));
                }
            }

            d1 = Math.abs(v.elementAt(1) - v.elementAt(2));
            d2 = Math.abs(v.elementAt(2) - v.elementAt(3));

            if (d1 < d2) {
                if (d1 < angDelta) {
                    v.set(2, new Double(v.elementAt(1)));
                }
            } else if (d2 < d1) {
                if (d2 < angDelta) {
                    v.set(2, new Double(v.elementAt(3)));
                }
            }
        }

        if (v.size() > 2) {
                //System.out.println("I am here-1");
            for (int i = 1; i < v.size();) {
                        //System.out.println(v.elementAt(i) + ", " + v.elementAt(i-1));
                if (v.elementAt(i).doubleValue() -  v.elementAt(i-1).doubleValue() == 0) {
                          //              System.out.println("I am here-2");
                          //              System.out.println(v.size());
                    v.removeElementAt(i-1);
                          //              System.out.println(v.size());
                } else {
                    i++;
                }
            }
        }
        //System.out.println(v);
    }


    /**
     * Do depth-first search of the analytic solution tree to compute the set of all
     * possible phi, psi sequences from which a set of possible fragments can be
     * constructed.
     *
     * @param thisSseId the loop boundary specification
     * @param phiTypeRdcSampled the sampled phi-defining RDC set
     * @param psiTypeRdcSampled the sampled psi-defining RDC set
     * @param phiTypeRdcVecWithMissingRdcsFilled the sampled phi-defining RDC set with missing RDCs filled
     * @param psiTypeRdcVecWithMissingRdcsFilled the sampled psi-defining RDC set with missing RDCs filled
     * @param Rg the rotation matrix that specifies the relative rotation of the first peptide plane (of the fragment) with respect to the principal order frame
     * @param Syy the diagonilized alignment tensor component
     * @param Szz the diagonilized alignment tensor component
     * @param currSearchDepth current search depth in the depth-first search tree
     * @param maxSearchDepth maximum depth of the depth-first search tree
     * @param ppS an array to store a solution (a sequence of phi and psi DOFs), i.e., a path from the root to a leaf of the depth-first search tree
     * @param firstPeptidePlane the first peptide plane of the fragment
     * @param phiPsiSolutionFile the file which stores the set of computed phi and psi sequences
     * @param weightInScoringFunction the relative weight used in scoring function
     * @param rightHand
     * @param dofsToBeModified the vector of dihedrals (DOFs) to be computed
     */
    public void recursiveDepthFirstSearchOfPhiPsiSolutionTreeForLoop(final mySseInfo thisSseId, final Vector<myDipolarCoupling> phiTypeRdcSampled, final Vector<myDipolarCoupling> psiTypeRdcSampled,
            final Vector<myDipolarCoupling> phiTypeRdcVecWithMissingRdcsFilled, final Vector<myDipolarCoupling> psiTypeRdcVecWithMissingRdcsFilled,
            Matrix Rg, double Syy, double Szz, final int currSearchDepth, final int maxSearchDepth, double[] ppS, final myPeptidePlane firstPeptidePlane,
            Writer phiPsiSolutionFile, final double weightInScoringFunction, final boolean rightHand, final Vector<myDof> dofsToBeModified) {

        //Vector<Double> phiVec = new Vector<Double>();
        //Vector<Double> psiVec = new Vector<Double>();

        if (currSearchDepth == maxSearchDepth) { // We hit the basis of the recursion, i.e., we reached a leaf (NIL) node after computing all phi and psi values
            //System.out.println("I am at a leaf"); System.exit(1);
            analyzeThisSolutionLoop(thisSseId, phiTypeRdcSampled, psiTypeRdcSampled, phiTypeRdcVecWithMissingRdcsFilled, psiTypeRdcVecWithMissingRdcsFilled,
                    ppS, firstPeptidePlane, phiPsiSolutionFile, weightInScoringFunction, dofsToBeModified, Syy, Szz);
        } else { //currSearchDepth < maxSearchDepth. So recurse down the tree.

            // get the residue number and check what type of Rama plot will be applicable to it.
            int thisResNum = thisSseId.getBeginResidueNumber() + currSearchDepth / 2;

            String thisResType = thisProteinSeq.getResidueType(thisResNum);
            String nextResType = thisProteinSeq.getResidueType(thisResNum + 1);

            //String ramaType = null;
            myRama ramaMapToBeUsed = null;

            if (thisResType.equalsIgnoreCase("PRO")) {
                //ramaType = "pro";
                ramaMapToBeUsed = ramaPro;
            } else if (nextResType.equalsIgnoreCase("PRO")) {
                //ramaType = "prepro";
                ramaMapToBeUsed = ramaPrePro;
            } else if (thisResType.equalsIgnoreCase("GLY")) {
                //ramaType = "gly";
                ramaMapToBeUsed = ramaGly;
            } else {
                //ramaType = "general";
                ramaMapToBeUsed = ramaGeneral;
            }

            if (dofsToBeModified.elementAt(currSearchDepth).getDof() == myDihedral.PHI) { //this dof is PHI

                Vector<Double> phiVec = null;

                if (phiTypeRdcSampled.elementAt(currSearchDepth / 2).getRdc() != -999) {
                    phiVec = myForKin.computePhi(Rg, Syy, Szz, phiTypeRdcSampled.elementAt(currSearchDepth / 2)); // We need phiType RDC for residue i to solve for Phi_i

                    /* TODO: This code was for testing for analytic solutions for the Gly residue only. Please remove it afterwards.
                     if (thisSseId.getBeginResidueNumber() + currSearchDepth / 2 == 53) {
                     System.out.print("Printing the gly phi values: ");
                     for (Double d : phiVec) {
                     System.out.print(Math.toDegrees(d.doubleValue()) + ", ");
                     }
                     System.out.println();
                     System.exit(1);
                     }
                     */

                    if (!(phiVec == null || phiVec.isEmpty())) {
                        checkAndPruneThosePhisOutsideRamaRegion:
                        {
                            Vector<Double> phiVecNew = new Vector<Double>();
                            for (Double phi : phiVec) {
                                if (ramaMapToBeUsed.isAllowedPhi(Math.toDegrees(phi))) {
                                    phiVecNew.add(phi);

                                    //System.out.println("residueNumber: " + thisResNum + "    residueType: " + thisResType + "    phi: " + Math.toDegrees(phi));
                                }
                            }

                            //phiVecNew = Collections.sort(phiNew);
                            int sz1 = phiVecNew.size();
                            if (thisRedPencil.isOn()) 
                                System.out.println("Phi vector: " + phiVecNew);                            
                            removeMultiplicity(phiVecNew);
                            int sz2 = phiVecNew.size();
                            if (thisRedPencil.isOn())
                                System.out.println("Phi vector: " + phiVecNew);
                            if (sz1 - sz2 > 0) {
                                if (thisRedPencil.isOn())
                                    System.out.println("Phi vector multiplicity detected and removed");
                            }
                            phiVec = phiVecNew;
                        }
                        //System.out.println(Arrays.toString(phiVec.toArray()));
                    }
                } else { // phiTypeRdc is missing for this DOF so rample from the Ramachandran
                    phiVec = ramaMapToBeUsed.chopPhiIntervals("allowed", __grid_resolution_for_phi_psi);
                    if (thisRedPencil.isOn())
                        ramaMapToBeUsed.printAllProjectedPhiIntervals("allowed");
                    if (thisRedPencil.isOn()) {
                        System.out.println(ramaMapToBeUsed.getType());
                        System.out.println("residueNumber: " + thisResNum + "    residueType: " + thisResType);
                        System.out.print("The phi values: ");
                    }
                    for (int i = 0; i < phiVec.size(); i++) { // Convert degrees to radian
                        phiVec.set(i, Math.toRadians(phiVec.elementAt(i)));
                        if (thisRedPencil.isOn())
                            System.out.print(Math.toDegrees(phiVec.elementAt(i)) + "  ");                        
                    }
                    if (thisRedPencil.isOn())
                        System.out.println();                    
                }                
                
                /* TODO: Please remove this commented-out code. Code used for debugging the module only.                   
                 if (thisSseId.getBeginResidueNumber() + currSearchDepth / 2 == 58) {
                 //phiVec.clear();
                 //phiVec.add(getSampledAngle(-62.066));
                 //phiVec.add(Math.toRadians(-62.066));
                 Vector<Double> phiVecNew = new Vector<Double>();
                 for (int i = 0; i < phiVec.size(); i++) {
                 if (Math.toRadians(-70) <= phiVec.elementAt(i) && phiVec.elementAt(i) <= Math.toRadians(-50)) {
                 phiVecNew.add(phiVec.elementAt(i));
                 }
                 }
                 phiVec = phiVecNew;
                 }
                 */
                    
                /**
                 * *********** Begin Dihedral Restraint Phi Check *************
                 */
                if (phiVec == null || phiVec.isEmpty()) { // Apply the dihedral-restraint filter
                    int trNum = thisSseId.getBeginResidueNumber() + currSearchDepth / 2;
                    myDihedralRestraint dr = myDihedralRestraint.getDihedralRestraint(__loop_dihedral_restraints, trNum);
                    if (dr != null) {
                        if (dr.isGood()) {
                            double phiLow = dr.getPhiLowerBound();
                            double phiHigh = dr.getPhiUpperBound();

                            Vector<Double> phiVecNew = new Vector<Double>();
                            for (int i = 0; i < phiVec.size(); i++) {
                                if (phiLow <= phiVec.elementAt(i) && phiVec.elementAt(i) <= phiHigh) {
                                    phiVecNew.add(phiVec.elementAt(i));
                                }
                            }
                            phiVec = phiVecNew;
                            if (thisRedPencil.isOn())
                                System.out.println("To restain phi dihedral the restrant used is: " + dr.toString());                            
                        }
                    }
                }
                /**
                 * ************ End Dihedral Restraint Phi Check **************
                 */
                

                if (phiVec == null || phiVec.isEmpty()) {
                    if (thisRedPencil.isOn())
                        System.out.println("There exists no solution for phi for the sampled rdc " + phiTypeRdcSampled.elementAt(currSearchDepth / 2) + " at residue number " + thisResNum);
                }

                for (double thisPhi : phiVec) { //for each phi recurse down the tree searching for the psi
                    if (phiVec.size() > 4) {
                        if (thisRedPencil.isOn())
                            System.out.println("For residue " + (thisSseId.getBeginResidueNumber() + currSearchDepth / 2) + "Current phi on grid: " + Math.toDegrees(thisPhi));
                    }

                    ppS[currSearchDepth] = thisPhi;
                    recursiveDepthFirstSearchOfPhiPsiSolutionTreeForLoop(thisSseId, phiTypeRdcSampled, psiTypeRdcSampled, phiTypeRdcVecWithMissingRdcsFilled, psiTypeRdcVecWithMissingRdcsFilled,
                            Rg, Syy, Szz, currSearchDepth + 1, dofsToBeModified.size(), ppS, firstPeptidePlane, phiPsiSolutionFile, weightInScoringFunction, rightHand, dofsToBeModified);
                }
            } else if (dofsToBeModified.elementAt(currSearchDepth).getDof() == myDihedral.PSI) { //this dof is PSI

                /**
                 * **************** Begin Steric Check, and apply other filters
                 * for Loop*******************
                 */
                if (currSearchDepth >= __depth_at_which_steric_checker_turns_on) {
                    if (currSearchDepth % 2 != 1) {                         
                        System.out.println("Error: the length of the phi-psi chain should be odd");                                                    
                        System.exit(1);
                    }

                    //System.out.println(Arrays.toString(ppS));
                    //System.exit(1);

                    // construct a vector of phi and psi from the solution
                    Vector<myPhiPsi> thisSolutionVec = new Vector<myPhiPsi>(currSearchDepth / 2 + 1); // initial capacity
                    for (int i = 0; i < currSearchDepth - 1; i += 2) {
                        thisSolutionVec.add(new myPhiPsi(thisSseId.getBeginResidueNumber() + i / 2, "ALA", ppS[i], ppS[i + 1]));
                    }
                    if (thisSseId.isHelix()) {
                        thisSolutionVec.add(new myPhiPsi(thisSseId.getBeginResidueNumber() + (currSearchDepth - 1) / 2, "ALA", ppS[currSearchDepth - 1], Const.psiAveHelix));
                    } else { // this SSE is a strand or LOOP
                        thisSolutionVec.add(new myPhiPsi(thisSseId.getBeginResidueNumber() + (currSearchDepth - 1) / 2, "ALA", ppS[currSearchDepth - 1], Const.psiAveBeta));
                    }

                    myProtein thisFragment = myProtein.buildProteinBackboneFromDihedrals(thisSolutionVec, firstPeptidePlane);


                    /**
                     * *********** Begin Reachability Check *************
                     */
                    reachabilityCheck:
                    {
                        myResidue thisTip = thisFragment.residueAt(thisFragment.getEndResidueNumber());
                        myPoint ca1 = thisTip.getAtom(myAtomLabel.__CA).getCoordinates();
                        myPoint ca2 = __anchor_residue.getAtom(myAtomLabel.__CA).getCoordinates();
                        double dist = myPoint.distance(ca1, ca2);
                        int numberOfCAs = __anchor_residue.getResidueNumber() - thisTip.getResidueNumber();
                        //System.out.println(thisTip.getResidueNumber() + "--" + __anchor_residue.getResidueNumber());
                        //System.exit(1);
                        if (numberOfCAs > 1) {
                            if (dist > numberOfCAs * 3.5 + 1.0) { // 1.0 here can be user-defined tolerance
                                //System.out.println("Mango haga");
                                //thisFragment.print();
                                //System.out.println(thisTip.getResidueNumber() + "--" + __anchor_residue.getResidueNumber());
                                //System.out.println("Dist: " + dist + "    ideal: " + numberOfCAs * 3.5 + "    @Depth: " + currSearchDepth / 2);
                                //System.exit(1);
                                //inputChar.readChar();
                                if (thisRedPencil.isOn())
                                    System.out.println("Reachability test fails at residue " + thisTip.getResidueNumber() + ". Retrning...");
                                return;
                            }
                        }
                    }
                    /**
                     * ************ End Reachability Check **************
                     */
                    /**
                     * *************** Begin NOE Check ******************
                     */
                    noeCheck:
                    {
                        double __epsilon_noe_threshold = 1.5;

                        if (__loop_noes != null) {
                            for (myNoe ee : __loop_noes) {

                                // TODO: Implement the Ha2 and Ha3 adjustments to Gly

                                if (ee.getResidueNoA() == thisFragment.getEndResidueNumber()) {

                                    //thisFragment.print();
                                    //System.out.println("Current search depth: " + currSearchDepth);
                                    //System.exit(1);

                                    myResidue r1 = thisFragment.residueAt(ee.getResidueNoA());
                                    myResidue r2 = __global_fold_in_pof_without_loops_and_anchor.residueAt(ee.getResidueNoB());
                                    if (r2 == null) {
                                        r2 = thisFragment.residueAt(ee.getResidueNoB());
                                    }
                                    if (r2 != null) {

                                        //System.out.println("Noe check for: " + ee.toString());

                                        myPoint p1 = r1.getAtom(ee.getAtomA()).getCoordinates();
                                        myPoint p2 = r2.getAtom(ee.getAtomB()).getCoordinates();

                                        double dist = myPoint.distance(p1, p2);
                                        if (dist > ee.getUpper() + __epsilon_noe_threshold) {
                                            return;
                                        }
                                    }
                                } else if (ee.getResidueNoB() == thisFragment.getEndResidueNumber()) {
                                    myResidue r1 = thisFragment.residueAt(ee.getResidueNoB());
                                    myResidue r2 = __global_fold_in_pof_without_loops_and_anchor.residueAt(ee.getResidueNoA());
                                    if (r2 == null) {
                                        r2 = thisFragment.residueAt(ee.getResidueNoA());
                                    }
                                    if (r2 != null) {
                                        myPoint p1 = r1.getAtom(ee.getAtomB()).getCoordinates();
                                        myPoint p2 = r2.getAtom(ee.getAtomA()).getCoordinates();

                                        double dist = myPoint.distance(p1, p2);

                                        if (dist > ee.getUpper() + __epsilon_noe_threshold) {
                                            return;
                                        }
                                        //System.out.println("dist: " + dist + " Noe check for: " + ee.toString());
                                    }
                                }
                            }
                        }

                        /* DO: Note the commented-out block of the code was used for testing only. Remove this anytime.                         
                         if (thisFragment.getEndResidueNumber() == 55) {
                         myPoint pointOnLoop1 = thisFragment.residueAt(55).getAtom(myAtomLabel.__CA).getCoordinates();
                         myPoint pointOnSse1 = __global_fold_in_pof.residueAt(22).getAtom(myAtomLabel.__CA).getCoordinates();
                         double dist = myPoint.distance(pointOnLoop1, pointOnSse1);
                         if (dist > 5.7) {//4.19 act
                         //System.out.println("Dist noe prune#1: " + dist + "    @Depth: " + currSearchDepth / 2);
                         //System.exit(1);
                         return;
                         }
                         }
                         */
                    }
                    /**
                     * **************** End NOE Check ******************
                     */
                    /**
                     * *************** Do Steric Check ******************
                     */
                    stericCheck:
                    {
                        // begin self-clash check
                        //thisFragment.print();
                        double clashScore = Double.MAX_VALUE;
                        // Check for steric clash and return the clash score
                        myStericChecker sc = new myStericChecker(1.0, 0.8, 0.4);
                        clashScore = sc.checkStericClash(thisFragment);
                        //System.out.println("At Depth: " + currSearchDepth / 2 +  " clashScore: " + clashScore);

                        if (clashScore > 100) {
                            //System.out.println(Arrays.toString(ppS));
                            //thisFragment.print();
                            //System.out.println("Clashed so returning");
                            return;
                        }
                        // end seld-clash check

                        // begin clash check with the core
                        double individualClashScoreContribution = individualScorePreComputed(thisFragment, clashScore);

                        // compute the overall clash score after merging the fragments
                        // clone, merge and sort
                        myProtein thisP = myProtein.cloneMergeAndSort(__global_fold_in_pof_without_loops_and_anchor, thisFragment);
                        double overAllClashScoreAfterMergingFragments = sc.checkStericClash(thisP);

                        //System.out.println("**--**");
                        //thisP.print();
                        //System.out.println("**--**");

                        if (overAllClashScoreAfterMergingFragments - individualClashScoreContribution < -100) { // sanity check                            
                            System.out.println("overAllClashScoreAfterMergingFragments: " + overAllClashScoreAfterMergingFragments + "    individualClashScoreContribution: " + individualClashScoreContribution);
                            System.out.println("Error: There is something wrong with the clash score comparison");                                
                            System.exit(1);
                        }

                        if (overAllClashScoreAfterMergingFragments - individualClashScoreContribution > 100) {
                            // idea is that towards the end there will be some clash with the end anchor which should be
                            // disregarded.
                            if (thisFragment.numberOfResidues() < (__end_residue_number - __begin_residue_number + 1) - 2) {
                                if (thisRedPencil.isOn())
                                    System.out.println("Clash with the core while at residue " + thisFragment.getEndResidueNumber() + ". returning...");
                                //System.exit(1);
                                return;
                            }
                        }

                        /* TODO: The code below was used for testing once upon a time. Pls. remove this.
                        if (Math.abs(overAllClashScoreAfterMergingFragments - __clash_score_for_global_fold_in_pof_without_loops_and_anchor) > 10) {
                            //thisP.print();
                            System.out.println("Clash with the core... exiting (returning)");
                            return;
                        }*/
                        //end clash check with the core
                    }
                }
                /**
                 * ****************End DO Steric Check for
                 * Loop*******************
                 */
                Vector<Double> psiVec = null;

                double phiAtThisResidue = ppS[currSearchDepth - 1];

                if (psiTypeRdcSampled.elementAt(currSearchDepth / 2 + 1).getRdc() != -999) { // Note -999 is for the missing RDCs, a standard used throughout in our programs
                    psiVec = myForKin.computePsi(Rg, Syy, Szz, psiTypeRdcSampled.elementAt(currSearchDepth / 2 + 1), phiAtThisResidue); // We need psiType rdc for residue (i+1) to solve for Psi_i

                    if (!(psiVec == null || psiVec.isEmpty())) {
                        checkAndPruneThosePsisOutsideRamaRegion:
                        {
                            Vector<Double> psiVecNew = new Vector<Double>();
                            for (Double psi : psiVec) {
                                if (ramaMapToBeUsed.isAllowedPsi(Math.toDegrees(phiAtThisResidue), Math.toDegrees(psi))) {
                                    psiVecNew.add(psi);
                                }
                            }
                            int sz1 = psiVecNew.size();
                            if (thisRedPencil.isOn())
                                System.out.println("Psi vector: " + psiVecNew);
                            removeMultiplicity(psiVecNew);
                            int sz2 = psiVecNew.size();
                            if (thisRedPencil.isOn())
                                System.out.println("Psi vector: " + psiVecNew);
                            if (sz1 - sz2 > 0) {
                                if (thisRedPencil.isOn())
                                    System.out.println("Psi vector multiplicity detected and removed");
                            }

                            psiVec = psiVecNew;
                        }
                    }
                    //System.out.println(Arrays.toString(psiVec.toArray()));
                } else { // psiTypeRdc is missing for this DOF so rample from the Ramachandran
                    psiVec = ramaMapToBeUsed.chopPsiIntervals(Math.toDegrees(phiAtThisResidue), "allowed", __grid_resolution_for_phi_psi);
                    if (thisRedPencil.isOn())
                        ramaMapToBeUsed.printAllPsiIntervalsGivenPhi(Math.toDegrees(phiAtThisResidue), "allowed");
                    if (thisRedPencil.isOn()) {
                        System.out.println(ramaMapToBeUsed.getType());
                        System.out.println("residueNumber: " + thisResNum + "    residueType: " + thisResType + "    phi: " + Math.toDegrees(phiAtThisResidue));
                        System.out.print("The correspongind psi values: ");
                    }
                    for (int i = 0; i < psiVec.size(); i++) { // Convert degrees to radian
                        psiVec.set(i, Math.toRadians(psiVec.elementAt(i)));
                        if (thisRedPencil.isOn())
                            System.out.print(Math.toDegrees(psiVec.elementAt(i)) + "  ");                        
                    }
                    if (thisRedPencil.isOn())
                        System.out.println();
                }
                

                /* TODO: The commented-out code below, was used for testing once upon a time.                    
                 if (thisSseId.getBeginResidueNumber() + currSearchDepth / 2 == 52) { // N_HN RDC
                 //psiVec.clear();
                 //psiVec.add(getSampledAngle(-20.044));
                 //psiVec.add(Math.toRadians(-20.044));
                 Vector<Double> psiVecNew = new Vector<Double>();
                 for (int i = 0; i < psiVec.size(); i++) {
                 if (Math.toRadians(-30) <= psiVec.elementAt(i) && psiVec.elementAt(i) <= Math.toRadians(-10)) {
                 psiVecNew.add(psiVec.elementAt(i));
                 }
                 }
                 psiVec = psiVecNew;
                 }
                 */        

                /**
                 * *********** Begin Dihedral Restraint Psi Check *************
                 */
                if (psiVec == null || psiVec.isEmpty()) { // Apply the dihedral-restraint filter
                    int trNum = thisSseId.getBeginResidueNumber() + currSearchDepth / 2;
                    myDihedralRestraint dr = myDihedralRestraint.getDihedralRestraint(__loop_dihedral_restraints, trNum);
                    if (dr != null) {
                        if (dr.isGood()) {
                            double psiLow = dr.getPsiLowerBound();
                            double psiHigh = dr.getPsiUpperBound();

                            Vector<Double> psiVecNew = new Vector<Double>();
                            for (int i = 0; i < psiVec.size(); i++) {
                                if (psiLow <= psiVec.elementAt(i) && psiVec.elementAt(i) <= psiHigh) {
                                    psiVecNew.add(psiVec.elementAt(i));
                                }
                            }
                            psiVec = psiVecNew;
                            if (thisRedPencil.isOn())
                                System.out.println("To restain psi dihedral the restrant used is: " + dr.toString());
                        }
                    }
                }
                /**
                 * ************ End Dihedral Restraint Psi Check **************
                 */                
                

                if (psiVec == null || psiVec.isEmpty()) {
                    if (thisRedPencil.isOn())
                        System.out.println("There exists no solution for psi for the sampled rdc " + psiTypeRdcSampled.elementAt(currSearchDepth / 2 + 1) + " at residue number " + thisResNum);                    
                }

                for (double thisPsi : psiVec) { // for each psi recurse down the tree searching for the next phi
                    if (psiVec.size() > 4) {
                        if (thisRedPencil.isOn())
                            System.out.println("For residue " + (thisSseId.getBeginResidueNumber() + currSearchDepth / 2) + "Current psi on grid: " + Math.toDegrees(thisPsi));
                    }

                    if (currSearchDepth / 2 + 1 > __max_depth) { // update the the maximum depth (in terms of residues) achieved
                        __max_depth = currSearchDepth / 2 + 1;
                    }
                    ppS[currSearchDepth] = thisPsi;
                    double thisPhi = ppS[currSearchDepth - 1];
                    //myPhiPsi pp = new myPhiPsi();

                    //System.out.println("thisPhi: " + thisPhi + "    thisPsi: " + thisPsi); // TODO: JUL04
                    Matrix newRg = myForKin.newRG__(thisPhi, thisPsi, Rg, rightHand); //compute a new rotation matrix rg
                    recursiveDepthFirstSearchOfPhiPsiSolutionTreeForLoop(thisSseId, phiTypeRdcSampled, psiTypeRdcSampled, phiTypeRdcVecWithMissingRdcsFilled, psiTypeRdcVecWithMissingRdcsFilled,
                            newRg, Syy, Szz, currSearchDepth + 1, dofsToBeModified.size(), ppS, firstPeptidePlane, phiPsiSolutionFile, weightInScoringFunction, rightHand, dofsToBeModified);
                }
            } else {                
                System.out.println("Error: Illegal DOF");
                System.exit(1);
            }
        }
    }


    public void analyzeThisSolutionLoop(final mySseInfo thisSseId, final Vector<myDipolarCoupling> phiTypeRdcSampled, final Vector<myDipolarCoupling> psiTypeRdcSampled,
            final Vector<myDipolarCoupling> phiTypeRdcVecWithMissingRdcsFilled, final Vector<myDipolarCoupling> psiTypeRdcVecWithMissingRdcsFilled,
            double[] ppS, final myPeptidePlane firstPeptidePlane, Writer phiPsiSolutionFile, final double weightInScoringFunction, final Vector<myDof> dofsToBeModified,
            double Syy, double Szz) {
        // Note: (1) The parameter dofsToBeModified is not currently used. Perhaps it can be used afterwards.
        //       (2) ppS contains a solution
        if (ppS.length % 2 != 1) {
            System.out.println("Error: the length of the phi-psi chain should be odd");
            System.exit(1);
        }

        // construct a vector of phi and psi from the solution
        Vector<myPhiPsi> thisSolutionVec = new Vector<myPhiPsi>(ppS.length / 2 + 1); // initial capacity
        for (int i = 0; i < ppS.length - 1; i += 2) {
            thisSolutionVec.add(new myPhiPsi(thisSseId.getBeginResidueNumber() + i / 2, "ALA", ppS[i], ppS[i + 1]));
        }
        if (thisSseId.isHelix()) {
            thisSolutionVec.add(new myPhiPsi(thisSseId.getBeginResidueNumber() + (ppS.length - 1) / 2, "ALA", ppS[ppS.length - 1], Const.psiAveHelix));
        } else { // this SSE is a strand or LOOP
            thisSolutionVec.add(new myPhiPsi(thisSseId.getBeginResidueNumber() + (ppS.length - 1) / 2, "ALA", ppS[ppS.length - 1], Const.psiAveBeta));
        }

        if (thisRedPencil.isOn())
            System.out.println("~~~~~~~~~~~~~ Begin Analysis of this Solution ~~~~~~~~~~~~~");
        try {
            // do all the finalization things
            __counter_for_total_number_of_solutions++;

            // phiPsiSolutionFile.write("CounterForTotalNumberOfSolutions: " + __counter_for_total_number_of_solutions + '\n');
            /*
             phiPsiSolutionFile.write("CounterForTotalNumberOfSolutions: " + __counter_for_total_number_of_solutions + "\n<Phi,Psi>: ");
             System.out.print("CounterForTotalNumberOfSolutions: " + __counter_for_total_number_of_solutions + "\n<Phi,Psi>: ");
             phiPsiSolutionFile.write(Arrays.toString(ppS) + '\n');
             System.out.println(Arrays.toString(ppS));
             */
            myProtein thisFragment = myProtein.buildProteinBackboneFromDihedrals(thisSolutionVec, firstPeptidePlane);

            // mutate to the actual residue types
            for (myResidue r : thisFragment) {
                r.mutate(thisProteinSeq.getResidueType(r.getResidueNumber()));
            }         

            
            thisFragment.print(myInputDataAndParameterManager.getInstance().getMostRecentFragmentFileForALeafOfSearchTree(), false); // false indicates that this file is being overwritten instead of being appended
            
            //System.out.println("Printing the solution PDB fragment");
            //thisFragment.print();

            double phiRmsd = 0.0, psiRmsd = 0.0;

            // Compute phi rmsd and psi rmsd
            if (thisSseId.isHelix()) {
                for (myPhiPsi thisPP : thisSolutionVec) {
                    phiRmsd += (thisPP.getPhi() - Const.phiAveHelix) * (thisPP.getPhi() - Const.phiAveHelix);
                    psiRmsd += (thisPP.getPsi() - Const.psiAveHelix) * (thisPP.getPsi() - Const.psiAveHelix);
                }
            } else if (thisSseId.isStrand()) {
                for (myPhiPsi thisPP : thisSolutionVec) {
                    phiRmsd += (thisPP.getPhi() - Const.phiAveBeta) * (thisPP.getPhi() - Const.phiAveBeta);
                    psiRmsd += (thisPP.getPsi() - Const.psiAveBeta) * (thisPP.getPsi() - Const.psiAveBeta);
                }
            }

            phiRmsd = Math.sqrt(phiRmsd / thisSolutionVec.size());
            psiRmsd = Math.sqrt(psiRmsd / thisSolutionVec.size());
            if (thisRedPencil.isOn())
                System.out.println("phiRmsd: " + phiRmsd + "    psiRmsd: " + psiRmsd);

            /*
             // Compute the phiType rdc rmsd
             System.out.println(phiTypeRdcSampled.elementAt(0).getType() + " RDC: ");
             double phiTypeRdcRmsd = 0.0;
             for (int i = 0; i < thisSolutionVec.size(); i++) {
             //System.out.println((thisSseId.getBeginResidueNumber() + i) + ": " + phiTypeRdcSampled.elementAt(i).getRdc() + "  " + phiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc() + "  " + Math.abs(phiTypeRdcSampled.elementAt(i).getRdc() - phiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc()));
             System.out.println(phiTypeRdcSampled.elementAt(i).getResidueNumber() + ": " + phiTypeRdcSampled.elementAt(i).getRdc() + "  " + phiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc() + "  " + Math.abs(phiTypeRdcSampled.elementAt(i).getRdc() - phiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc()));
             phiTypeRdcRmsd += (phiTypeRdcSampled.elementAt(i).getRdc() - phiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc()) * (phiTypeRdcSampled.elementAt(i).getRdc() - phiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc());
             }
             phiTypeRdcRmsd = Math.sqrt(phiTypeRdcRmsd / thisSolutionVec.size());
             System.out.println(phiTypeRdcSampled.elementAt(0).getType() + " RDC RMSD: " + phiTypeRdcRmsd);

             // Compute the psiType rdc rmsd
             System.out.println(psiTypeRdcSampled.elementAt(0).getType() + " RDC: ");
             double psiTypeRdcRmsd = 0.0;
             for (int i = 0; i < thisSolutionVec.size(); i++) {
             System.out.println(psiTypeRdcSampled.elementAt(i).getResidueNumber() + ": " + psiTypeRdcSampled.elementAt(i).getRdc() + "  " + psiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc() + "  " + Math.abs(psiTypeRdcSampled.elementAt(i).getRdc() - psiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc()));
             psiTypeRdcRmsd += (psiTypeRdcSampled.elementAt(i).getRdc() - psiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc()) * (psiTypeRdcSampled.elementAt(i).getRdc() - psiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc());
             }
             psiTypeRdcRmsd = Math.sqrt(psiTypeRdcRmsd / thisSolutionVec.size());
             System.out.println(psiTypeRdcSampled.elementAt(0).getType() + " RDC RMSD: " + psiTypeRdcRmsd);
             */
            myDipolarCoupling.Type phiRdcType = phiTypeRdcVecWithMissingRdcsFilled.elementAt(0).getType();
            myDipolarCoupling.Type psiRdcType = psiTypeRdcVecWithMissingRdcsFilled.elementAt(0).getType();

            Matrix rotToPOF = Matrix.identity(3, 3);
            double phiTypeRdcRmsd = -999.999;
            double psiTypeRdcRmsd = -999.999;
            //System.out.println(phiRdcType + " RDC:");
            //double phiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(thisFragment, phiTypeRdcVecWithMissingRdcsFilled,
            //        rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(phiRdcType.toString()));
            //System.out.println(phiRdcType + " RDC rmsd: " + phiTypeRdcRmsd);

            myProtein thisFragmentFirstRemoved = new myProtein(thisFragment);
            //thisFragmentFirstRemoved.removeResidue(thisSseId.getBeginResidueNumber());

//            System.out.println(psiRdcType + " RDC:");
//            double psiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(thisFragmentFirstRemoved, psiTypeRdcVecWithMissingRdcsFilled,
//                    rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(psiRdcType.toString()));
//            System.out.println(psiRdcType + " RDC rmsd: " + psiTypeRdcRmsd);

//            System.out.println("Phi/Psi: ");
//            for (myPhiPsi thisPP : thisSolutionVec) {
//                System.out.println(thisPP.getResidueNumber() + "   " + Math.toDegrees(thisPP.getPhi()) + "  " + Math.toDegrees(thisPP.getPsi()));
//            }
/*
             double scoreTermFromRdcs = (phiTypeRdcRmsd / myConstantsForRdcs.DmaxScaled.get(phiRdcType.toString()) +
             psiTypeRdcRmsd / myConstantsForRdcs.DmaxScaled.get(psiRdcType.toString())) + weightInScoringFunction * (phiRmsd + psiRmsd);
             */
            myPair<Double, Integer> squaredPhiRdcDev = myAlignmentTensorEstimator.computeRdcSumOfSquaredDeviation(thisFragment, phiTypeRdcVecWithMissingRdcsFilled,
                    rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(phiRdcType.toString()));

            myPair<Double, Integer> squaredPsiRdcDev = myAlignmentTensorEstimator.computeRdcSumOfSquaredDeviation(thisFragmentFirstRemoved, psiTypeRdcVecWithMissingRdcsFilled,
                    rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(psiRdcType.toString()));

            double phiPreFactor = myConstantsForRdcs.DmaxScaled.get(phiRdcType.toString());
            double psiPreFactor = myConstantsForRdcs.DmaxScaled.get(psiRdcType.toString());
            double scoreTermFromRdcs = squaredPhiRdcDev.first() / (phiPreFactor * phiPreFactor) + squaredPsiRdcDev.first() / (psiPreFactor * psiPreFactor);
            double totalNumberOfRdcs = squaredPhiRdcDev.second() + squaredPsiRdcDev.second();
            scoreTermFromRdcs = Math.sqrt(scoreTermFromRdcs / totalNumberOfRdcs);
            if (thisRedPencil.isOn())
                System.out.println("scoreTermFromRdcs: " + scoreTermFromRdcs);
            double scoreTermFromRdcsSaved = scoreTermFromRdcs;
            scoreTermFromRdcs += weightInScoringFunction * (phiRmsd + psiRmsd);
            if (thisRedPencil.isOn())
                System.out.println("scoreTermFromRdcsAndPhiPsi: " + scoreTermFromRdcs);

            double clashScore = Double.MAX_VALUE;
            // Check for steric clash and return the clash score
            myStericChecker sc = new myStericChecker(1.0, 0.8, 0.4);
            clashScore = sc.checkStericClash(thisFragment);
            if (thisRedPencil.isOn())
                System.out.println("clashScore: " + clashScore);

            // combine the scores
            double jointScore = /*scoreTermFromRdcs +*/ clashScore /*/ 20.0*/;
            jointScore = getClosureDistance(thisFragment.residueAt(thisFragment.getEndResidueNumber()), __anchor_residue);
            if (thisRedPencil.isOn())
                System.out.println("EndResidueNumber: " + thisFragment.getEndResidueNumber());

            if (jointScore < __closure_distance_threshold && clashScore < 200) {
                if (thisRedPencil.isOn()) {
                    System.out.println("jointScore: " + jointScore);
                    System.out.println("Printing the solution PDB fragment");
                    thisFragment.print();
                }
                
                double bbRmsd = getRmsd(thisFragment, __global_fold_in_pof, __begin_residue_number, __end_residue_number);

                if (true/*jointScore < //__min_score//__closure_distance_threshold*/ /*bbRmsd < __rms_distance_wrt_reference*/) { //we got a better fragment save it!
                    __min_score = jointScore;
                    //__min_clash_score = clashScore;
                    __best_fragment = new myProtein(thisFragment);

                    Writer outputBestFragment = null;
                    File bestFragmentFile = new File(__this_input_data_and_parameter_manager.getBestFragmentPdbFile()); //, true)); //true for append mode
                    
                    outputBestFragment = new BufferedWriter(new FileWriter(bestFragmentFile, true));//, true)); //true for append mode

                    __counter_for_total_number_of_final_loops++;
                    System.out.println("loop counter: " + __counter_for_total_number_of_final_loops + " for the tree with id: " + getCurrentTreeId() + '\n'); 
                    
                    phiPsiSolutionFile.write("loop counter: " + __counter_for_total_number_of_final_loops + " for the tree with id: " + getCurrentTreeId() + '\n');
                    
                    outputBestFragment.write("REMARK 100 loop counter: " + __counter_for_total_number_of_final_loops + " for the tree with id: " + getCurrentTreeId() + '\n');
                    //outputBestFragment.write("REMARK 100 phiRmsd: " + phiRmsd + "    psiRmsd: " + psiRmsd + '\n');
                    outputBestFragment.write("REMARK 100 " + phiTypeRdcSampled.elementAt(0).getType() + " RDC" /*RMSD (including missing RDCs): " + phiTypeRdcRmsd*/ + '\n');
                    outputBestFragment.write("REMARK 100 " + psiTypeRdcSampled.elementAt(0).getType() + " RDC" /*RMSD (including missing RDCs): " + psiTypeRdcRmsd*/ + '\n');
                    //outputBestFragment.write("REMARK 100 scoreTermFromRdcs: " + scoreTermFromRdcsSaved + '\n');
                    //outputBestFragment.write("REMARK 100 scoreTermFromRdcsAndPhiPsiDeviation: " + scoreTermFromRdcs + '\n');
                    outputBestFragment.write("REMARK 100 clashScore: " + clashScore + '\n');
                    //outputBestFragment.write("REMARK 100 jointScore: " + jointScore + '\n');
                    outputBestFragment.write("REMARK 100 closureDistance (unoptimized): " + jointScore + '\n');                         
                    outputBestFragment.write("REMARK 100 scoreWrtReference (note: if negative, reference loop is not input for comparison): " + bbRmsd + '\n');                                       
                    outputBestFragment.write("REMARK 100 Printing the phi and psi values below\n");
                    for (myPhiPsi thisPP : thisSolutionVec) {
                        outputBestFragment.write("REMARK 100 " + thisPP.getResidueNumber() + "   " + Math.toDegrees(thisPP.getPhi()) + "  " + Math.toDegrees(thisPP.getPsi()) + '\n');
                    }
                    outputBestFragment.write("REMARK 100 Printing the corresponding PDB fragment\n");
                    thisFragment.print(outputBestFragment);
                    outputBestFragment.close();
                }
            }
        } catch (IOException e) {
            System.out.println("Error: An IOException is thrown while writing to the outputfile");
            e.printStackTrace();
            System.exit(1);
        }
        
        if (thisRedPencil.isOn())
            System.out.println("~~~~~~~~~~~~~~ End Analysis of this Solution ~~~~~~~~~~~~~~");
    }


    
    // ALL are in POF
    public double getRmsd(myProtein p1, myProtein p2, int beginResidueNumber, int endResidueNumber) {
        
        // First, check if all the residues in the range are present, else return a negative number 
        // to represent the RMSD cannot be computed.
        
        for (int i = beginResidueNumber; i <= endResidueNumber; i++) {
            if (p1.residueAt(i) == null || p2.residueAt(i) == null) {
                return -1;
            }

            myResidue r1 = p1.residueAt(i);
            myResidue r2 = p2.residueAt(i);
            
            if (r1.getAtom(myAtomLabel.__N) == null || r2.getAtom(myAtomLabel.__N) == null
                    || r1.getAtom(myAtomLabel.__CA) == null || r2.getAtom(myAtomLabel.__CA) == null
                    || r1.getAtom(myAtomLabel.__C) == null || r2.getAtom(myAtomLabel.__C) == null) {
                return -1;
            }
        }
        
        double rmsd = 0.0;
        int numberOfResidues = endResidueNumber - beginResidueNumber + 1;

        for (int i = beginResidueNumber; i <= endResidueNumber; i++) {
            myResidue r1 = p1.residueAt(i);
            myResidue r2 = p2.residueAt(i);
            myPoint point1 = r1.getAtom(myAtomLabel.__N).getCoordinates();
            myPoint point2 = r2.getAtom(myAtomLabel.__N).getCoordinates();
            rmsd += myPoint.squaredDist(point1, point2);
            point1 = r1.getAtom(myAtomLabel.__CA).getCoordinates();
            point2 = r2.getAtom(myAtomLabel.__CA).getCoordinates();
            rmsd += myPoint.squaredDist(point1, point2);
            point1 = r1.getAtom(myAtomLabel.__C).getCoordinates();
            point2 = r2.getAtom(myAtomLabel.__C).getCoordinates();
            rmsd += myPoint.squaredDist(point1, point2);
        }
        rmsd = Math.sqrt(rmsd / (3 * numberOfResidues));
        return rmsd;
    }

    public double getClosureDistance(myResidue r1, myResidue r2) {

        double rmsd = 0.0;

        rmsd += myPoint.squaredDist(r1.getAtom("N").getCoordinates(), r2.getAtom("N").getCoordinates());
        rmsd += myPoint.squaredDist(r1.getAtom("CA").getCoordinates(), r2.getAtom("CA").getCoordinates());
        rmsd += myPoint.squaredDist(r1.getAtom("C").getCoordinates(), r2.getAtom("C").getCoordinates());

        rmsd = Math.sqrt(rmsd / 3.0);
//        if (thisRedPencil.isOn())
//            System.out.printf("Closure distance: %8.3f    epsilon: %8.3f\n", rmsd, epsilon);
        return rmsd;
    }

}
