/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2012 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, August 04, 2012
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.1, August 04, 2012
 * @author        Chittaranjan Tripathy (2007-2012)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */
import java.util.*;

/**
 * Description of the class
 */
public class myRotamerLibrary {
    private Map<String, Vector<myRotamer>> __rotamer_library = null;
    private String __library_name = null;
    private boolean __read_old_format_pdb = false;

    /**
     * This is the only instance (singleton) of this class. For now, we designed
     * this class to be a singleton. Later the decision can be changed, if needed.
     */
    private static final myRotamerLibrary __instance = initializeAndLoadRotamerLibrary();

    /**
     * Default constructor which creates an instance of an empty rotamer library.
     * This constructor is private to enforce the singleton property of the class.
     */
//    private myRotamerLibrary() {
//    }

    private static myRotamerLibrary initializeAndLoadRotamerLibrary() {
        // The part BELOW can be read from a configuration file.
        String rotaFileDir = System.getProperty("user.dir") + System.getProperty("file.separator") +
                "rotamerLibraryRichardsons" + System.getProperty("file.separator");                

        String[] rotaFiles = new String[]{"ala", "arg", "asn", "asp", "cys", "gln", "glu", /*"gly",*/ "his", "ile", "leu", "lys", "met", "phe", "pro", "ser", "thr", "trp", "tyr", "val"};
        for (int i = 0; i < rotaFiles.length; i++) {
            rotaFiles[i] = rotaFileDir + rotaFiles[i] + "allH";//".pdb";
        }
        boolean setOldFormatPdbRead = true;
        // The part ABOVE can be read from a configuration file.
        
        return new myRotamerLibrary(setOldFormatPdbRead, rotaFiles);
    }

    /**
     * This method returns the only instance of the object of this type.
     *
     * @return the only instance of an object of this type
     */
    public static synchronized myRotamerLibrary getInstance() {
        return __instance;
    }

    /**
     * This is overriden to make sure that a singleton cannot be cloned.
     *
     * @return
     * @throws CloneNotSupportedException
     */
    @Override public Object clone() throws CloneNotSupportedException {
        throw new CloneNotSupportedException();
    }
    
    private myRotamerLibrary(String... rotamerFileNameList) {
        __rotamer_library = parseRotamerLibrary(rotamerFileNameList);
    }

    private myRotamerLibrary(boolean setOldFormatPdbRead, String... rotamerFileNameList) {
        __read_old_format_pdb = setOldFormatPdbRead;
        __rotamer_library = parseRotamerLibrary(rotamerFileNameList);
    }

    private Map<String, Vector<myRotamer>> parseRotamerLibrary(String... rotamerFileNameList) {
        Map<String, Vector<myRotamer>> thisRotamerLibrary = new TreeMap<String, Vector<myRotamer>>();

        for (String rotamerFileName : rotamerFileNameList) {
            myPair<String, Vector<myRotamer>> rotamersOfThisType = parseRotamerFile(rotamerFileName);
            if (rotamersOfThisType != null) {
                thisRotamerLibrary.put(rotamersOfThisType.first(), rotamersOfThisType.second());
            }
        }
        
        return thisRotamerLibrary;
    }

    private myPair<String, Vector<myRotamer>> parseRotamerFile(String rotamerFile) {
        Vector<myProtein> vp = new Vector<myProtein>();
        myPdbParser pParser = new myPdbParser(rotamerFile, __read_old_format_pdb);
        while (pParser.hasNextProtein()) {
            vp.add(pParser.nextProtein());
        }
        myProtein p = vp.elementAt(0);
        
        int rotamerIndex = 0;
        double probability = 0.0;        
        Vector<myRotamer> vr = new Vector<myRotamer>();
        
        for (myResidue r : p) {
            myRotamer rota = new myRotamer(r, rotamerIndex++, probability);
            vr.add(rota);
        }
        
        if (vr.size() == 0) {
            return null;
        } else {
            return new myPair(vr.elementAt(0).getResidueName(), vr);
        }                
    }

    public void print() {
        for (Map.Entry<String, Vector<myRotamer>> entry : __rotamer_library.entrySet()) {
            System.out.println("******************Rotamer Type: " + entry.getKey() + " *********************");
            for (myRotamer r : entry.getValue()) {
                r.print();
            }
        }
    }

    public void mutateResidue(myResidue r, String newResidueType) {
        mutateResidue(r, newResidueType, 0);
    }

    public void mutateResidue(myResidue r, String newResidueType, int rotamerIndex) {
        newResidueType = newResidueType.toUpperCase().trim();
        String oldResidueType = r.getResidueName();
        r.mutate(newResidueType);        

        if (oldResidueType.equalsIgnoreCase("GLY") && newResidueType.equalsIgnoreCase("GLY")) {            
            return;
        } else if (!oldResidueType.equalsIgnoreCase("GLY") && newResidueType.equalsIgnoreCase("GLY")) { // fix the bond lengths for gly
            myVector3D CA_HA2_BondVector = new myVector3D(r.getAtom(myAtomLabel.__CA).getCoordinates(), r.getAtom(myAtomLabel.__HA2).getCoordinates());
            myVector3D CA_HA3_BondVector = new myVector3D(r.getAtom(myAtomLabel.__CA).getCoordinates(), r.getAtom(myAtomLabel.__HA3).getCoordinates());
            double CA_HA2_BondLength = CA_HA2_BondVector.norm();            
                       
            myPoint CA = r.getAtom(myAtomLabel.__CA).getCoordinates();
            myPoint p = new myPoint(CA.getX() + CA_HA3_BondVector.cosAlpha() * CA_HA2_BondLength, 
                    CA.getY() + CA_HA3_BondVector.cosBeta() * CA_HA2_BondLength,
                    CA.getZ() + CA_HA3_BondVector.cosGamma() * CA_HA2_BondLength);
            r.getAtom(myAtomLabel.__HA3).setCoordinates(p);
        } else if (oldResidueType.equalsIgnoreCase("GLY") && !newResidueType.equalsIgnoreCase("GLY")) {
            // We will extract the CA-CB bond length from the rotamer library
            myRotamer rota = __rotamer_library.get(newResidueType).elementAt(0);
            myVector3D CA_CB_BondVector = new myVector3D(r.getAtom(myAtomLabel.__CA).getCoordinates(), r.getAtom(myAtomLabel.__CB).getCoordinates());
            double CA_CB_BondLengthFromRota = myPoint.distance(rota.getAtom(myAtomLabel.__CA).getCoordinates(), rota.getAtom(myAtomLabel.__CB).getCoordinates());
            
            myPoint CA = r.getAtom(myAtomLabel.__CA).getCoordinates();
            myPoint p = new myPoint(CA.getX() + CA_CB_BondVector.cosAlpha() * CA_CB_BondLengthFromRota,
                    CA.getY() + CA_CB_BondVector.cosBeta() * CA_CB_BondLengthFromRota,
                    CA.getZ() + CA_CB_BondVector.cosGamma() * CA_CB_BondLengthFromRota);
            r.getAtom(myAtomLabel.__CB).setCoordinates(p);
        } else { // both the new and old types are non-gly so ca-cb distance is correct
            // do nothing
        }

        if (!r.getResidueName().equalsIgnoreCase("GLY")) {
            graftSideChain(r, __rotamer_library.get(r.getResidueName()).elementAt(rotamerIndex)); // by default we put the first rotamer
        }
    }

    private void graftSideChain(myResidue r, myRotamer thisRotamer) {
        assert r.getResidueName().equalsIgnoreCase(thisRotamer.getResidueName()) == true;
        if (r.getResidueName().equalsIgnoreCase("GLY")) {
            return;
        }

        myRotamer rota = new myRotamer(thisRotamer);
        align(r, rota);

        // Extract the chain id, and use them below.
        char chainId = r.getAtom(myAtomLabel.__CA).getChainId();
        
        for (myAtom a : rota) {
            a.setChainId(chainId);
            if (a.getAtomName().equalsIgnoreCase(myAtomLabel.__N) ||
                    a.getAtomName().equalsIgnoreCase(myAtomLabel.__H) ||
                    a.getAtomName().equalsIgnoreCase(myAtomLabel.__CA) ||
                    a.getAtomName().equalsIgnoreCase(myAtomLabel.__HA) ||
                    a.getAtomName().equalsIgnoreCase(myAtomLabel.__C) ||
                    a.getAtomName().equalsIgnoreCase(myAtomLabel.__O)) {
                // do nothing
            } else if (r.hasAtom(a.getAtomName())) { // just set the coordinates
                r.getAtom(a.getAtomName()).setCoordinates(a.getCoordinates());
            } else if (r.canHaveAtom(a.getAtomName())) {
                r.addAtom(new myAtom(a));
            }
        }        
    }

    private void align(myResidue r, myRotamer rota) {
        // align rota with r so that N, CA, CB are superimposed
        myPoint p1 = r.getAtom(myAtomLabel.__N).getCoordinates();
        myPoint p2 = r.getAtom(myAtomLabel.__CA).getCoordinates();              
        myPoint p3 = r.getResidueName().equalsIgnoreCase("GLY") ? r.getAtom(myAtomLabel.__HA3).getCoordinates() : r.getAtom(myAtomLabel.__CB).getCoordinates();
        Matrix M1 = myForKin.getRotationWrtGlobalFrame(p1, p2, p3);

        myPoint pr1 = rota.getAtom(myAtomLabel.__N).getCoordinates();
        myPoint pr2 = rota.getAtom(myAtomLabel.__CA).getCoordinates();
        myPoint pr3 = rota.getResidueName().equalsIgnoreCase("GLY") ? rota.getAtom(myAtomLabel.__HA3).getCoordinates() : rota.getAtom(myAtomLabel.__CB).getCoordinates();
        Matrix M2 = myForKin.getRotationWrtGlobalFrame(pr1, pr2, pr3);

        Matrix rotMat = M1.times(M2.transpose());

        for (myAtom a : rota) {
            a.setCoordinates(new myPoint(rotMat.times(a.getCoordinates().getXYZ())));
        }

        pr1 = rota.getAtom(myAtomLabel.__N).getCoordinates();

        myVector3D tr = new myVector3D(pr1, p1);
        for (myAtom a : rota) {
            //System.out.println("before: " + a.getAtomName() + "  " + a.getCoordinates());
            a.getCoordinates().translate(tr);
            //System.out.println("after: " + a.getAtomName() + "  " + a.getCoordinates());
        }
    }

    public int numberOfRotamers(String residueType) {
        return residueType.equalsIgnoreCase("GLY") ? 0 : __rotamer_library.get(residueType).size();
    }

    public static void main(String... args) {

       myRotamerLibrary thisLib = myRotamerLibrary.getInstance();
       thisLib.print();

       //System.exit(1);

        System.out.println("----------Testing mutation and rotamer grafting----------");
        
        Vector<myProtein> vp = new Vector<myProtein>();
        myPdbParser pParser = new myPdbParser("crap.pdb"); //("crap.pdb");//("LW1m.pdb");//("1d3zModel1.pdb");//("LW1m.pdb");//("ubqByRdc1005.pdb");//("ubqByRDC1005Refined.pdb");//("LW1m.pdb");//("crap.pdb");//("LW1m.pdb");//("1d3zModel1.pdb");//("SpaSimRdc.pdb");//"1GHHModel1.pdb");//("1Q2NModel1.pdb");//("SpaSimRdc.pdb");//("1UBQSimplified.pdb");//("1d3zModel1.pdb");//("1D3ZModel1.pdb");//("SpaSimRdc.pdb");//("1Q2NModel1.pdb");//("1GHHModel1.pdb");//("1wcrChainA.pdb");//("1Q2NModel1.pdb");////("1Q2NModel1.pdb");//("mostRecentFragment.pdb");//("1GHHModel1.pdb");
        while (pParser.hasNextProtein()) {
            vp.add(pParser.nextProtein());
        }
        myProtein pp = vp.elementAt(0);
        
        myResidue r = pp.residueAt(32);

/*
//        for (myAtom a : r) {
//            a.print();
//            a.printBonds();
//        }
//        System.out.println("number of bonds: " + r.numberOfBonds());

        //Vector<myRotamer> vVal = thisLib.__rotamer_library.get("VAL");
        myRotamer rMer = thisLib.__rotamer_library.get("ARG").elementAt(0);

        System.out.println("Printing the residue and rotamer");
        r.print();
        rMer.print();

        System.out.println("Aligning the residue and rotamer");
        thisLib.align(r, rMer);
        r.print();
        rMer.print();
        
        System.out.println("number of bonds: " + r.numberOfBonds());
        //System.exit(1);
*/
        thisLib.mutateResidue(r, "GLY");
        thisLib.mutateResidue(r, "ARG", 5);
        //thisLib.mutateResidue(r, "VAL");
        thisLib.mutateResidue(r, "GLY");
        thisLib.mutateResidue(r, "VAL",1);
        //thisLib.mutateResidue(r, "ARG");
        //thisLib.mutateResidue(r, "GLY");

        //TODO: pr in number of bonds computation f--it

        System.out.println("Printing all mutations");

        for (int i = 0; i < 28; i++) {
            System.out.println("MODEL      " + i);
            thisLib.mutateResidue(r, "ARG", i);
            r.print();
            System.out.print("TER\nENDMDL\n");
        }

        System.exit(1);

        System.out.println("Printing the mutated protein");
        pp.print();

        System.exit(1);
        
        System.out.println("Printing the bonds");
        int i = 0;
        for (myAtom a : r) {
            a.print();
            System.out.print(a.numberOfBonds() + " :: ");
            a.printBonds();
            i += a.numberOfBonds();
        }
        System.out.println("number of bonds: " + r.numberOfBonds() + "    i = " + i);

        pp.print();

        System.out.println("Printing the rotamers");
        Vector<myRotamer> rotamers = thisLib.__rotamer_library.get("ALA");
        for (myRotamer rota : rotamers) {
            rota.print();
        }
        
        
    }
}