/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2012 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, August 04, 2012
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.1, August 04, 2012
 * @author        Chittaranjan Tripathy (2007-2012)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */
import java.io.*;
import java.util.*;
import java.util.regex.*;

/**
 * Description of the class
 */
public class mySseInfo {

    private int __begin_residue_number = 0; //The number of the starting residue
    private int __end_residue_number = 0; //The number of the ending residue
    private String __sse_type = myMiscConstants.undefined; //The type of the secondary structure element (H or E)
    private int __sse_id = 0; // SSE id given by the program such as Helix 1, 2 etc. // TODO: Do you really need such an ID? If not remove this field.


    /**
     * Default constructor. It does nothing other than initializing the state.
     */
    public mySseInfo() {
    }

    /**
     * Constructs the specification of a secondary structure element given the
     * boundaries, the type of secondary structure element (helix or strand).
     *
     * @param beginResNum
     * @param endResNum
     * @param typeOfSse
     */
    public mySseInfo(int beginResNum, int endResNum, String typeOfSse) {
        __begin_residue_number = beginResNum;
        __end_residue_number = endResNum;
        __sse_type = typeOfSse;
        __sse_id = 0;
    }

    /**
     * Constructs the specification of a secondary structure element given the
     * boundaries, the type of secondary structure element (helix or strand) and 
     * an id number for the secondary structure element.
     *
     * @param beginResNum first residue number
     * @param endResNum last residue number
     * @param typeOfSse secondary structure element type (helix or strand)
     * @param idOfSse an id number for the secondary structure element
     */
    public mySseInfo(int beginResNum, int endResNum, String typeOfSse, int idOfSse) {
        __begin_residue_number = beginResNum;
        __end_residue_number = endResNum;
        __sse_type = typeOfSse;
        __sse_id = idOfSse;
    }

    /**
     * A copy constructor which takes an instance of this class and constructs
     * another instance with all fields copied appropriately.
     *
     * @param s the object which is to be cloned
     */
    public mySseInfo(final mySseInfo s) {
      __begin_residue_number = s.__begin_residue_number;
      __end_residue_number = s.__end_residue_number;
      __sse_type = s.__sse_type;
      __sse_id = s.__sse_id;
    }

    /**
     * Return the first (lowest) residue number of the secondary structure element.
     *
     * @return the first (lowest) residue number of the secondary element
     */
    public int getBeginResidueNumber() {
        return __begin_residue_number;
    }

    /**
     * Return the last (highest) residue number of the secondary structure element.
     *
     * @return the last (highest) residue number of the secondary element
     */
    public int getEndResidueNumber() {
        return __end_residue_number;
    }

    /**
     * Return the type (helix or strand) of the secondary structure element.
     *
     * @return the type (helix or strand) of the secondary structure element
     */
    public String getType() {
        return __sse_type;
    }

    /**
     * Return the id of the secondary structure element.
     *
     * @return the id of the secondary structure element
     */
    public int getId() {
        return __sse_id;
    }

    /**
     * Return true if the secondary structure element is a helix.
     *
     * @return true if the secondary structure element is a helix
     */
    public boolean isHelix() {
        return __sse_type.toUpperCase().charAt(0) == 'H';
    }

    /**
     * Return true if the secondary structure element is a strand.
     *
     * @return true if the secondary structure element is a strand
     */
    public boolean isStrand() {
        return __sse_type.toUpperCase().charAt(0) == 'E';
    }

    /**
     * Return the number of residues in the secondary structure element.
     *
     * @return the number of residues in the secondary structure element
     */
    public int numberOfResidues() {
        return getEndResidueNumber() - getBeginResidueNumber() + 1;
    }

    /**
     * Return the string representation of the secondary structure element specification.
     *
     * @return the string representation of secondary structure element specification
     */
    @Override public String toString() {
        String desc = __begin_residue_number + "  " + __end_residue_number + "  " + __sse_type + "  " + __sse_id;
        return desc;
    }

    /**
     * A reader for secondary structure identification file.
     *
     * @param idFile the file containing secondary structure element specification
     * @return a vector of mySseInfo objects
     */
    public Vector<mySseInfo> parseSseIds(String sseIdFileName) {
        Vector<mySseInfo> sseIdVec = new Vector<mySseInfo>();
        try {
            Scanner scanner = new Scanner(new File(sseIdFileName));
            scanner.useDelimiter(System.getProperty("line.separator"));
            //String patternString = "[0-9]+\\s+[0-9]+\\s+[HE]\\s+[0-9]+";
            String patternString = "\\s*((@helix)|(@strand))\\s*\\([0-9]+\\s*\\,\\s*[0-9]+\\s*\\,\\s*[0-9]+\\s*\\)\\s*";//,\\s*[H,h,E,e]\\s*

            while (scanner.hasNextLine()) {
                String line = myMiscUtilities.stripLineComment(scanner.nextLine()).trim();
                //System.out.println("----: " + line);
                if (line.matches(patternString)) {
                    System.out.println(line);       
                    String sseType = null;
                    if (line.startsWith("@helix")) {
                        sseType = "H";
                    } else {
                        sseType = "E";
                    }

                    String patternString2 = "[0-9]+";
                    Pattern r = Pattern.compile(patternString2);
                    Matcher m = r.matcher(line);

                    int[] A = new int[3];
                    int i = 0;
                    
                    while (m.find()) {
                        int start = m.start(0);
                        int end = m.end(0);
                        String thisMatch = line.substring(start, end);
                        A[i++] = Integer.parseInt(thisMatch);

                    }                    
                    sseIdVec.add(new mySseInfo(A[0], A[1], sseType, A[2]));
                }
            }

            scanner.close();
        } catch (FileNotFoundException e) {
            System.out.println("Error: Input file " + sseIdFileName + " not found");
            e.printStackTrace();
        } 

        return sseIdVec;
    }

    /**
     * Read the secondary structure specification file and get the SSE specification
     * which will be used to compute the alignment tensor.
     *
     * @param sseIdFileName the file containing secondary structure element specification
     * @return mySseInfo object to be used for computation of the alignment tensor
     */
    public mySseInfo parseSseIdFileAndReturnSseForAlignmentTensorComputation(String sseIdFileName) {
        Vector<mySseInfo> sseIdVec = new Vector<mySseInfo>();
        try {
            Scanner scanner = new Scanner(new File(sseIdFileName));
            scanner.useDelimiter(System.getProperty("line.separator"));
            //String patternString = "[0-9]+\\s+[0-9]+\\s+[HE]\\s+[0-9]+";
            String patternString = "\\s*(@computeAlignmentTensorUsing)\\s*\\(\\s*((@helix)|(@strand))\\s*\\([0-9]+\\s*\\,\\s*[0-9]+\\s*\\,\\s*[0-9]+\\s*\\)\\s*\\)\\s*";//,\\s*[H,h,E,e]\\s*

            while (scanner.hasNextLine()) {
                String line = myMiscUtilities.stripLineComment(scanner.nextLine()).trim();
                //System.out.println("----: " + line);
                if (line.matches(patternString)) {
                    System.out.println(line);
                    String sseType = null;
                    if (line.indexOf("@helix") != -1) {
                        sseType = "H";
                    } else if (line.indexOf("@strand") != -1) {
                        sseType = "E";
                    } else {
                        continue;
                    }

                    String patternString2 = "[0-9]+";
                    Pattern r = Pattern.compile(patternString2);
                    Matcher m = r.matcher(line);

                    int[] A = new int[3];
                    int i = 0;

                    while (m.find()) {
                        int start = m.start(0);
                        int end = m.end(0);
                        String thisMatch = line.substring(start, end);
                        A[i++] = Integer.parseInt(thisMatch);

                    }
                    sseIdVec.add(new mySseInfo(A[0], A[1], sseType, A[2]));                    
                }
            }

            scanner.close();
        } catch (FileNotFoundException e) {
            System.out.println("Error: Input file " + sseIdFileName + " not found");
            e.printStackTrace();
        } 

        if (sseIdVec.size() == 0) {
            return null;
        } else if (sseIdVec.size() == 1) {
            return sseIdVec.elementAt(0);
        } else { //(sseIdVec.size() > 1)
            System.out.println("Error: multiple specifications of SSE to be used for alignment tensor computation");
            System.exit(1);
        }
        return null;
    }

    /**
     * Read the secondary structure specification file and get the SSE specifications
     * and relative orientation information for strands in the beta sheets
     * 
     * @param sseIdFileName the file containing secondary structure element specification
     * @return specification of beta sheets
     */
    public Vector<Vector<myTriple<mySseInfo, mySseInfo, Boolean>>> parseSseIdFileAndReturnSheetSpecifications(String sseIdFileName) {
        Vector<Vector<myTriple<Integer, Integer, String>>> sheetSpecs = new Vector<Vector<myTriple<Integer, Integer, String>>>();
        Vector<Vector<myTriple<mySseInfo, mySseInfo, Boolean>>> finalBSSpecs = new Vector<Vector<myTriple<mySseInfo, mySseInfo, Boolean>>>();

        try {
            Scanner scanner = new Scanner(new File((sseIdFileName)));
            scanner.useDelimiter(System.getProperty("line.separator"));

            while (scanner.hasNextLine()) {
                String line = scanner.nextLine().trim();
                Vector<myTriple<Integer, Integer, String>> thisSheetBySseIds = sph(line);

                if (thisSheetBySseIds != null) {
                    sheetSpecs.add(thisSheetBySseIds);
                }
            }

            // From the sheet ids get the sheet specifications
            mySseInfo id2ary = new mySseInfo();
            Vector<mySseInfo> idVec = id2ary.parseSseIds(sseIdFileName);

            for (Vector<myTriple<Integer, Integer, String>> sheetSpecById : sheetSpecs) {
                Vector<myTriple<mySseInfo, mySseInfo, Boolean>> stt = new Vector<myTriple<mySseInfo, mySseInfo, Boolean>>();
                for (myTriple<Integer, Integer, String> thisStrandPair : sheetSpecById) {

                    mySseInfo ss1 = getSseInfoFromIdNumber(idVec, thisStrandPair.first());
                    mySseInfo ss2 = getSseInfoFromIdNumber(idVec, thisStrandPair.second());
                    boolean parallel = false;
                    if (thisStrandPair.third().equalsIgnoreCase("parallel")) {
                        parallel = true;
                    } else if (thisStrandPair.third().equalsIgnoreCase("antiparallel")) {
                        parallel = false;
                    } else {
                        System.out.println("Error: The relative orientations of the strands is specified incorrectly");
                        System.exit(1);
                    }

                    myTriple<mySseInfo, mySseInfo, Boolean> thisStrandPairWithBoundarySpec = new myTriple<mySseInfo, mySseInfo, Boolean>(ss1, ss2, parallel);
                    stt.add(thisStrandPairWithBoundarySpec);
                }
                if (stt.size() != 0) {
                    finalBSSpecs.add(stt);
                }
            }

            scanner.close();
        } catch (FileNotFoundException e) {
            System.out.println("Error: Input file " + sseIdFileName + " not found");
            e.printStackTrace();
        } 
        return finalBSSpecs;
    }

    /**
     * Return the SSE specification from the numeric id.
     *
     * @param sseSpec vector of SSE specification objects
     * @param id numeric id
     * @return the SSE specification with the numeric id if exists, else return null
     */
    private static mySseInfo getSseInfoFromIdNumber(Vector<mySseInfo> sseSpec, int id) {
        for (mySseInfo s : sseSpec) {
            if (s.getId() == id) {
                return s;
            }
        }
        System.out.println("Error: The SSE spec is not found");
        System.exit(1);
        return null;
    }

    /**
     * Parse a line containing a beta sheet specification.
     *
     * @param sheetSpecLine a line containing a beta sheet specification
     * @return the beta sheet specification
     */
    private Vector<myTriple<Integer, Integer, String>> sph(String sheetSpecLine) {
        String patternString = "\\s*@[S,s][H,h][E,e][E,e][T,t]\\s*\\(\\s*[(\\s*[0-9]+\\s*\\,\\s*[0-9]+\\s*,\\s*(anti)?parallel\\s*)\\s*]+\\)\\s*";// "\\[\\s*[[0-9]+\\s+]+[0-9]+\\s*\\]";
        String line =  sheetSpecLine; // " sheet: (2,  4,  parallel  ) (2 , 3  , antiparallel  )   (4, 5, antiparallel) (5, 6, antiparallel)";

        Vector<myTriple<Integer, Integer, String>> strandRelInfo = new Vector<myTriple<Integer, Integer, String>>();

        if (line.matches(patternString)) {
            String patternString2 = "\\(\\s*[0-9]+\\s*\\,\\s*[0-9]+\\s*,\\s*(anti)?parallel\\s*\\)";
            Pattern r = Pattern.compile(patternString2);
            Matcher m = r.matcher(line);

            while (m.find()) {
                int start = m.start(0);
                int end = m.end(0);
                String thisMatch = line.substring(start, end);
                thisMatch = thisMatch.replaceAll("[(),]", " ");
                thisMatch = thisMatch.trim();
                String[] words = thisMatch.split("\\s{1,}");
                //System.out.println("length:" + words.length);
                //for (String s: words)System.out.println(s);
                myTriple<Integer, Integer, String> t = new myTriple<Integer, Integer, String>(Integer.parseInt(words[0]), Integer.parseInt(words[1]), words[2].trim());
                strandRelInfo.add(t);
                //System.out.println(thisMatch);
            }
        }

        if (strandRelInfo.size() > 0) {
            return strandRelInfo;
        } else {
            return null;
        }
    }
}
