/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2012 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, August 04, 2012
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.1, August 04, 2012
 * @author        Chittaranjan Tripathy (2007-2012)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package utilities;

/**
 * Import statement(s)
 */
import java.util.*;

import analytic.myDipolarCoupling;
import analytic.myProtein;
import analytic.Matrix;
import analytic.myAtom;
import analytic.myResidue;
import analytic.myVector3D;
import analytic.myConstantsForRdcs;
import analytic.myBackboneModeler;
import analytic.myPdbParser;
import analytic.myTriple;
import analytic.myMatrix;
import analytic.myPhiPsi;
import analytic.Const;
import analytic.myInputDataAndParameterManager;
import analytic.myAlignmentTensorEstimator;
import analytic.myMiscConstants;
import analytic.EigenvalueDecomposition;
import analytic.myDipolarCouplingTable;

/**
 * Description of the class
 */
public class RDCSimulator {
    private String __universal_key = "universalKey"; // Our map below has only one element. Implemented as a map for future extensions
   
    private Map<String, Matrix> __alignment_tensors = new TreeMap<String, Matrix>();
      
    private String __pdb_file;    
    
    private Vector<myDipolarCoupling.Type> __types_for_pales = new Vector<myDipolarCoupling.Type>();

    private Vector<myDipolarCoupling> backComputeOneRdcSetFromGivenAT(final myProtein p, Matrix AT, myDipolarCoupling.Type type, double rdcRatio) {
        // TODO: Note about the (caveat) Rule-> first atom has to be a heavy atom, second atom can be anything.
        Vector<myDipolarCoupling> rdcBackCalVec = new Vector<myDipolarCoupling>();

        // get the RDC type from dcType object
        myDipolarCoupling dcType = new myDipolarCoupling(type);
        String atom1Type = dcType.getFirstAtomName();
        String atom2Type = dcType.getSecondAtomName();
        int firstAtomResNum = dcType.getFirstAtomResidueNumber();
        int secondAtomResNum = dcType.getSecondAtomResidueNumber();
        int diffRes = secondAtomResNum - firstAtomResNum;

        for (myResidue thisRes : p) {
            myResidue resA = thisRes;
            myResidue resB = p.residueAt(resA.getResidueNumber() + diffRes);
            if (resA == null || resB == null) {
                continue;
            } else { //if (atom1Type.length() == 2 && atom1Type.startsWith("C") && atom2Type.length() == 2) {
                Vector<myAtom> vA = new Vector<myAtom>();
                myAtom atomA1 = null, atomA2 = null;
                atomA1 = resA.getAtom(atom1Type);
                if (atomA1 != null) {
                    vA.add(atomA1);
                } else { // means CG1 CG2 type
                    atomA1 = resA.getAtom(atom1Type + '1');
                    atomA2 = resA.getAtom(atom1Type + '2');
                    if (atomA1 != null && atomA2 != null) {
                        vA.add(atomA1);
                        vA.add(atomA2);
                    }
                }

                for (myAtom A : vA) {
                    Vector<myAtom> vB = new Vector<myAtom>();
                    String Aname = A.getAtomName();
                    String Bname = atom2Type;
                    myAtom atomB = resB.getAtom(Bname);
                    if (atomB != null) { // means HA for CA
                        vB.add(atomB);
                    } else if (resB.getAtom(Bname + '2') != null && resB.getAtom(Bname + '3') != null) { // HA2 and HA2 for CA
                        vB.add(resB.getAtom(Bname + '2'));
                        vB.add(resB.getAtom(Bname + '3'));
                    } else if (Character.isDigit(Aname.charAt(Aname.length() - 1))) { // means CG1 CG2 type
                        myAtom atomB2 = resB.getAtom(Bname + Aname.charAt(Aname.length() - 1) + '2');   // e.g. HG12 type
                        myAtom atomB3 = resB.getAtom(Bname + Aname.charAt(Aname.length() - 1) + '3');   // e.g. HG13 type
                        if (atomB2 != null && atomB3 != null) {
                            vB.add(atomB2);
                            vB.add(atomB3);
                        }
                    }

                    // NOTE AUG 02, 2010. The following block implements the new AT computation.

                    // now we have an atom A and the set of atoms {B} s.t. AB is a
                    // internuclear vector for which RDC has to be back-computed.
                    double rdcBackCal = 0.0;
                    for (myAtom B : vB) {
                        Matrix RotToPOF = Matrix.identity(3, 3); // added AUG 02, 2010
                        myVector3D interNuclearVec = myVector3D.normalize(myVector3D.rotate(new myVector3D(A.getCoordinates(), B.getCoordinates()), RotToPOF)); // this is a unit vector, so the direction cosines are the components of the vector
                        double x = interNuclearVec.cosAlpha();
                        double y = interNuclearVec.cosBeta();
                        double z = interNuclearVec.cosGamma();
                        //System.out.println("cosX: " + cosX + "    cosY: " + cosY + "    cosZ: " + cosZ);
                        //double rdcCalOnThisVector = rdcRatio * (x * x * Sxx + y * y * Syy + z * z * Szz);
                        double Syy = AT.get(1, 1), Szz = AT.get(2, 2), Sxy = AT.get(0, 1), Sxz = AT.get(0, 2), Syz = AT.get(1, 2);
                        double rdcCalOnThisVector = rdcRatio * (x * (Sxy * y + Sxz * z - x * (Syy + Szz)) + y * (Sxy * x + Syy * y + Syz * z) + z * (Sxz * x + Syz * y + Szz * z));
                        rdcBackCal += rdcCalOnThisVector;
                    }

                    if (vB.size() == 1) {
                        rdcBackCalVec.add(new myDipolarCoupling(A.getAtomName(), resA.getResidueNumber(), Bname, resB.getResidueNumber(), rdcBackCal, 0.0));
                    } else if (vB.size() == 2) {
                        String newBname = vB.elementAt(0).getAtomName();
                        assert Character.isDigit(newBname.charAt(newBname.length() - 1)) == true : "Error: incorrect second atom name while back-computing RDC";
                        newBname = newBname.substring(0, newBname.length() - 1) + '#'; // remove last digit and replace with '#'
                        rdcBackCalVec.add(new myDipolarCoupling(A.getAtomName(), resA.getResidueNumber(), newBname, resB.getResidueNumber(), rdcBackCal, 0.0));
                    }
                }
            }
        }
        return rdcBackCalVec;
    }

    private Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> backComputeRdcsFromGivenAT(myProtein p, Matrix AT, Vector<myDipolarCoupling.Type> rdcTypes) {
        Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMediumBackCal = new TreeMap<myDipolarCoupling.Type, Vector<myDipolarCoupling>>();
        Vector<myDipolarCoupling> rdcBackCal = null;
        for (myDipolarCoupling.Type type : rdcTypes) {
            switch (type) {
                case N_HN:
                case CA_HA:
                case CA_C:
                case CA_CB:
                case C_N:
                case C_HN:
                case CB_HB:
                /*case C_RCSA:*/ {
                    rdcBackCal = backComputeOneRdcSetFromGivenAT(p, AT, type, myConstantsForRdcs.DmaxScaled.get(type.toString()));
                    rdcCsaTableForThisMediumBackCal.put(type, rdcBackCal);
                    break;
                }
                default: {
                    System.out.println("The RDC type for which the back-computed values are sought is not implemented");
                    System.exit(1);
                }
            }
        }
        return rdcCsaTableForThisMediumBackCal;
    }

    private myTriple<Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>>, Matrix, Matrix> simulateRdcsFromPalesAT(myProtein p, Matrix AT, Vector<myDipolarCoupling.Type> rdcTypes) {
        Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMediumBackCal = backComputeRdcsFromGivenAT(p, AT, rdcTypes);


        System.out.println("Printing the back calculated RDCs, i.e., the PALES simulated RDCs");
        myDipolarCouplingTable.printInXplorFormat(rdcCsaTableForThisMediumBackCal); // test other format too

        System.out.println("Doing the eigenvalue decomposition (diagonalization) on the alignment tensor");
        System.out.println("Printing alignment tensor before diagonalization");
        AT.print(5, 12);
        // Diagonalize the alignment tensor
        Matrix mm = AT.copy(); //new Matrix(mArr);
        //mm.print(8, 3);
        EigenvalueDecomposition eigs = new EigenvalueDecomposition(mm, true);
        Matrix VV = eigs.getV();
        Matrix DD = eigs.getD();
        System.out.println("Printing VV matrix of diagonalization");
        VV.print(5, 12);
        System.out.println("Printing the diagonalized alignment tensor");
        DD.print(5, 12);
        double Sxx = DD.get(0, 0);
        double Syy = DD.get(1, 1);
        double Szz = DD.get(2, 2);
        System.out.println("Sxx: " + Sxx + "  Syy: " + Syy + "  Szz: " + Szz);
        //eigenValues[0] = Syy;
        //eigenValues[1] = Szz;
        System.out.println("The rhombicity is " + myBackboneModeler.computeRhombicity(Syy, Szz));


        myProtein rotatedFragment = new myProtein(p);
        rotatedFragment.rotate(new myMatrix(VV.transpose().getArrayCopy()));

        Vector<myPhiPsi> firstDihedralSet = p.computeDihedrals();
        Vector<myPhiPsi> secondDihedradSet = rotatedFragment.computeDihedrals();

        /*  REDUNDANT as the table of simulated RDC was filled above
        // Compute the RDC deviations and print the RDC rmsds
        rdcCsaTableForThisMediumBackCal.clear(); // flush the map
        for (myDipolarCoupling.Type type : rdcTypes) {
        System.out.println("Printing " + type.toString() + " RDC deviations");
        Vector<myDipolarCoupling> rdcBackCal = backComputeRdcs(p, type, VV.transpose(), Sxx, Syy, Szz, myConstantsForRdcs.DmaxScaled.get(type.toString()));

        //Vector<myDipolarCoupling> rdcVec = rdcCsaTableForThisMedium.get(type);
        //double rdcRmsd = computeRdcRmsd(thisFragment, rdcVec, VV.transpose(), Sxx, Syy, Szz, myConstantsForRdcs.DmaxScaled.get(type.toString()));
        //System.out.println(type.toString() + " RDC rmsd: " + rdcRmsd);
        rdcCsaTableForThisMediumBackCal.put(type, rdcBackCal);
        }
        myDipolarCouplingTable.printInXplorFormat(rdcCsaTableForThisMediumBackCal);
         */
        //Make sure what returned has the same handedness as the input PDB
        Matrix rotationToPOF = null;
        if (Math.abs(firstDihedralSet.elementAt(0).getPsi() - secondDihedradSet.elementAt(0).getPsi()) < myMiscConstants.eps) {
            System.out.println(" The same (i.e., the same handedness when checked from bestFit)");
            //return VV.transpose();AUG 02, 2010
            rotationToPOF = VV.transpose();
        } else {
            System.out.println(" Differ (i.e., different handedness when checked from bestFit)");
            //return Const.mLeftHand.times(VV.transpose());AUG 02, 2010
            rotationToPOF = Const.mLeftHand.times(VV.transpose());
// 	return VV.transpose();
        }

        System.out.println("Printing VV matrix of diagonalization");
        VV.print(5, 12);
        System.out.println("Printing rotationToPOF matrix");
        rotationToPOF.print(5, 12);

        return new myTriple<Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>>, Matrix, Matrix>(rdcCsaTableForThisMediumBackCal, rotationToPOF, DD);
    }

    public void printRdcTableForPales(myProtein p, Vector<myDipolarCoupling.Type> rdcTypes) {

        System.out.println("Printing the PALES RDC input table");
        System.out.println("VARS RESID_I RESNAME_I ATOMNAME_I RESID_J RESNAME_J ATOMNAME_J D DD W");
        System.out.println("FORMAT %5d %6s %6s %5d %6s %6s %9.3f %9.3f %.2f");

        for (myDipolarCoupling.Type thisRdcType : rdcTypes) {
            switch (thisRdcType) {
                case N_HN: {
                    for (myResidue r : p) {
                        int RESID_I = r.getResidueNumber();
                        String RESNAME_I = r.getResidueName();
                        String ATOMNAME_I = "N";
                        int RESID_J = RESID_I;
                        String RESNAME_J = RESNAME_I;
                        String ATOMNAME_J = null;

                        if (r.getResidueName().equalsIgnoreCase("PRO")) {
                            ATOMNAME_J = null;
                        } else {
                            ATOMNAME_J = "H";
                        }

                        double D = 99.999; // dummy value for RDC
                        double DD = 1.000; // dummy value for error
                        double W = 1.00; // dummy value for relative weight

                        if (!r.getResidueName().equalsIgnoreCase("PRO")) {
                            System.out.printf("%5d %6s %6s %5d %6s %6s %9.3f %9.3f %.2f\n", RESID_I, RESNAME_I, ATOMNAME_I, RESID_J, RESNAME_J, ATOMNAME_J, D, DD, W);
                        }
                    }
                }
                System.out.println();
                break;
                case CA_HA: {
                    for (myResidue r : p) {
                        int RESID_I = r.getResidueNumber();
                        String RESNAME_I = r.getResidueName();
                        String ATOMNAME_I = "CA";
                        int RESID_J = RESID_I;
                        String RESNAME_J = RESNAME_I;
                        String ATOMNAME_J = null;

                        if (r.getResidueName().equalsIgnoreCase("GLY")) {
                            ATOMNAME_J = "HA#";
                        } else {
                            ATOMNAME_J = "HA";
                        }

                        double D = 99.999; // dummy value for RDC
                        double DD = 1.000; // dummy value for error
                        double W = 1.00; // dummy value for relative weight

                        System.out.printf("%5d %6s %6s %5d %6s %6s %9.3f %9.3f %.2f\n", RESID_I, RESNAME_I, ATOMNAME_I, RESID_J, RESNAME_J, ATOMNAME_J, D, DD, W);
                    }
                }
                System.out.println();
                break;
                case CA_C: {
                    for (myResidue r : p) {
                        int RESID_I = r.getResidueNumber();
                        String RESNAME_I = r.getResidueName();
                        String ATOMNAME_I = "CA";
                        int RESID_J = RESID_I;
                        String RESNAME_J = RESNAME_I;
                        String ATOMNAME_J = "C";

                        double D = 99.999; // dummy value for RDC
                        double DD = 1.000; // dummy value for error
                        double W = 1.00; // dummy value for relative weight

                        System.out.printf("%5d %6s %6s %5d %6s %6s %9.3f %9.3f %.2f\n", RESID_I, RESNAME_I, ATOMNAME_I, RESID_J, RESNAME_J, ATOMNAME_J, D, DD, W);
                    }
                }
                System.out.println();
                break;
                case CA_CB: {
                    for (myResidue r : p) {
                        int RESID_I = r.getResidueNumber();
                        String RESNAME_I = r.getResidueName();
                        String ATOMNAME_I = "CA";
                        int RESID_J = RESID_I;
                        String RESNAME_J = RESNAME_I;
                        String ATOMNAME_J = null;

                        if (r.getResidueName().equalsIgnoreCase("GLY")) {
                            ATOMNAME_J = null;
                        } else {
                            ATOMNAME_J = "CB";
                        }

                        double D = 99.999; // dummy value for RDC
                        double DD = 1.000; // dummy value for error
                        double W = 1.00; // dummy value for relative weight

                        if (!r.getResidueName().equalsIgnoreCase("PRO")) {
                            System.out.printf("%5d %6s %6s %5d %6s %6s %9.3f %9.3f %.2f\n", RESID_I, RESNAME_I, ATOMNAME_I, RESID_J, RESNAME_J, ATOMNAME_J, D, DD, W);
                        }
                    }
                }
                System.out.println();
                break;
                case C_N: {
                    for (myResidue r : p) {
                        int RESID_I = r.getResidueNumber();
                        String RESNAME_I = r.getResidueName();
                        String ATOMNAME_I = "C";

                        boolean nextResExists = true;
                        if (p.residueAt(RESID_I + 1) == null) {
                            nextResExists = false;
                        }

                        if (nextResExists) {
                            int RESID_J = RESID_I + 1;
                            String RESNAME_J = p.residueAt(RESID_J).getResidueName();
                            String ATOMNAME_J = "N";

                            double D = 99.999; // dummy value for RDC
                            double DD = 1.000; // dummy value for error
                            double W = 1.00; // dummy value for relative weight

                            System.out.printf("%5d %6s %6s %5d %6s %6s %9.3f %9.3f %.2f\n", RESID_I, RESNAME_I, ATOMNAME_I, RESID_J, RESNAME_J, ATOMNAME_J, D, DD, W);
                        }
                    }
                }
                System.out.println();
                break;
                case C_HN: {
                    for (myResidue r : p) {
                        int RESID_I = r.getResidueNumber();
                        String RESNAME_I = r.getResidueName();
                        String ATOMNAME_I = "C";

                        boolean nextResExists = true;
                        if (p.residueAt(RESID_I + 1) == null) {
                            nextResExists = false;
                        }

                        if (nextResExists) {
                            int RESID_J = RESID_I + 1;
                            String RESNAME_J = p.residueAt(RESID_J).getResidueName();
                            String ATOMNAME_J = null;
                            if (!RESNAME_J.equalsIgnoreCase("PRO")) {
                                ATOMNAME_J = "H";
                            }

                            double D = 99.999; // dummy value for RDC
                            double DD = 1.000; // dummy value for error
                            double W = 1.00; // dummy value for relative weight

                            if (!RESNAME_J.equalsIgnoreCase("PRO")) {
                                System.out.printf("%5d %6s %6s %5d %6s %6s %9.3f %9.3f %.2f\n", RESID_I, RESNAME_I, ATOMNAME_I, RESID_J, RESNAME_J, ATOMNAME_J, D, DD, W);
                            }
                        }
                    }
                }
                System.out.println();
                break;
                case CB_HB:
                    System.out.println("The RDC type for which the PALES table is sought is not implemented");
                    System.exit(1);
                    break;
//                case C_RCSA:
//                    System.out.println("The CSA type for which the PALES table is sought is not implemented");
//                    System.exit(1);
//                    break;
                default:
                    System.out.println("The RDC type for which the PALES table is sought is not implemented");
                    System.exit(1);
            }
        }
    }
 

    public void doAndTestRdcSimulations() {
        System.out.println("================RDC Simulation Starts here using PALES alignment tensor=================");

        Matrix AT = __alignment_tensors.get(__universal_key);

        System.out.println("Printing the alignment tensor from PALES before scaling");
        AT.print(5, 12);        

        double scalingFactor = 21585.19; 
        System.out.println("The scaling factor used is " + scalingFactor);
        AT = AT.times(scalingFactor);

        System.out.println("Printing the  alignment tensor from PALES after scaling");
        AT.print(5, 12);        
        //System.exit(1);

        Vector<myProtein> vp = new Vector<myProtein>();
        myPdbParser pParser = new myPdbParser(__pdb_file);
        while (pParser.hasNextProtein()) {
            vp.add(pParser.nextProtein());
        }
        myProtein pp = vp.elementAt(0);

        myProtein ppClone1 = new myProtein(pp);
        myProtein ppClone2 = new myProtein(pp);

        System.out.println("Printing the sequence");
        for (myResidue r : pp) {
            //System.out.println(r.getResidueNumber() + " " + r.getResidueName());
            //r.mutate("ALA");
        }

        // printRdcTableForPales(pp, new Vector<myDipolarCoupling.Type>(Arrays.asList(typesForPales)));
        myTriple<Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>>, Matrix, Matrix> simulatedRdcTable_and_rotationToPOF_and_eigenvalues =
                simulateRdcsFromPalesAT(pp, AT, __types_for_pales);

        System.out.println("Printing the rotation to POF matrix");
        simulatedRdcTable_and_rotationToPOF_and_eigenvalues.second().print(5, 12);
        System.out.println("Printing the eigenvalues");
        simulatedRdcTable_and_rotationToPOF_and_eigenvalues.third().print(5, 12);
        System.out.println("Printing the protein backbone (only) in a POF");
        myProtein rotatedProtein = new myProtein(ppClone2);
        rotatedProtein.rotate(new myMatrix(simulatedRdcTable_and_rotationToPOF_and_eigenvalues.second().getArrayCopy()));
        myProtein.extractBackbone(rotatedProtein).print();



        System.out.println("++++++++++++++++++++++++++++++++++++++++++++++");
        System.out.println("+++++++SANITY    +++++     CHECK++++++++++++++");
        System.out.println("++++++++++++++++++++++++++++++++++++++++++++++");

        //System.exit(1);

        Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMediumBackCal = new TreeMap<myDipolarCoupling.Type, Vector<myDipolarCoupling>>();

        double[] saupe = new double[4];
        Matrix rotToPOF = myAlignmentTensorEstimator.bestFitUsingSVD(ppClone1, simulatedRdcTable_and_rotationToPOF_and_eigenvalues.first(),
                __types_for_pales, rdcCsaTableForThisMediumBackCal, saupe);

        System.out.println("The rhombicity is " + myBackboneModeler.computeRhombicity(saupe[0], saupe[1]));

        rotToPOF.print(5, 12);

        //myProtein prot = bbModeler.computeAlignmentTensorGivenStructureAndRdcs(thisRdcCsaTable, mediumName,
        //        typesOfRdcsToBeUsedForRefinement, pp, saupe, rdcCsaTableForThisMediumBackCal);

        //System.out.println("Printing backcomputed RDCs");
        //myDipolarCouplingTable.printInXplorFormat(rdcCsaTableForThisMediumBackCal);//(thisRdcCsaTable.getRdcMapForThisMedium(mediumName));

        myProtein thisProteinInPOF = new myProtein(ppClone2);
        thisProteinInPOF.rotate(new myMatrix(rotToPOF.getArrayCopy()));

        //return thisProteinInPOF;

        System.out.println("Printing the protein backbone (only) in a POF");
        myProtein.extractBackbone(thisProteinInPOF).print();
        System.out.println("Printing the backbone dihedrals");
        thisProteinInPOF.printBackboneDihedrals();
        System.out.println("Printing the sequence");
        for (myResidue r : thisProteinInPOF) {
            System.out.println(r.getResidueNumber() + " " + r.getResidueName());
            //r.mutate("ALA");
        }


    }

    /**
     * Check for a valid option.
     *
     * @param s string that specifies an option
     * @return return true if the option is valid
     */
    public static boolean validOption(String s) {
        return s.startsWith("-");
    }    

    public void parseCommandLineArgumentsAndInvokeRDCSimulator(String[] args) {
        String masterDir = null;
        String directoryArchitectureFile = null;       
        double Szz = Double.NaN, Sxx_yy = Double.NaN, Sxy = Double.NaN, Sxz = Double.NaN, Syz = Double.NaN;
        String pdbFileName = null;
        Vector<String> types = new Vector<String>();

        if (args.length == 0) {
            System.out.println("To learn more about this command, please type: " + new Exception().getStackTrace()[0].getClassName() + " -help");
        }
        
        // Parse the command line arguments 
        for (int i = 0; i < args.length && validOption(args[i]);) {
            String thisArgument = args[i++];

            if (thisArgument.equals("-help") || thisArgument.equals("-verbose")) {
                String helpString = "Usage: " + new Exception().getStackTrace()[0].getClassName() + " <options> \nwhere possible options include: \n"
                        + "-masterdir <file name>                 Specify the master directory name which contains all the input files and the output will also be redirected to this directory  " + '\n'
                        + "-directoryarchitecturefile <file name> Specify the directory architecture file name which contains the information about the input and output directories and files under the master directory. The default name searched for is dirArch.txt" + '\n'
                        + "-pdbfile <file name>                   Specify the name of the input PDB file (one model and must end with TER \\n END" + '\n'                                                
                        + "-rdctypes <list of RDC types>          Specify the RDC types to simulate (type: N_HN, CA_C, CA_HA, C_N, C_HN)" + '\n'
                        + "-Szz                                   Specify the value of Szz" + '\n'
                        + "-Sxx_yy                                Specify the the value of Sxx - Syy" + '\n'
                        + "-Sxy                                   Specify the value of Sxy" + '\n'
                        + "-Sxz                                   Specify the the value of Sxz" + '\n'
                        + "-Syz                                   Specify the value of Syz" + '\n'                       
                        + "-verbose                               Print a synopsis of standard options and return" + '\n'
                        + "-help                                  Print a synopsis of standard options and return" + '\n';
                System.out.println(helpString);
                return;
            } else if (thisArgument.equals("-masterdir")) {
                if (i < args.length) {
                    masterDir = args[i++];
                    if (validOption(masterDir)) {
                        System.out.println("Error: incorrect master directory name or missing argument(s)");
                        System.exit(1);
                    }
                } else {
                    System.out.println("Error: master directory name is not supplied");
                    System.exit(1);
                }
            } else if (thisArgument.equals("-directoryarchitecturefile")) {
                if (i < args.length) {
                    directoryArchitectureFile = args[i++];
                    if (validOption(directoryArchitectureFile)) {
                        System.out.println("Error: incorrect directory architecture file name or missing argument(s)");
                        System.exit(1);
                    }
                } else {
                    System.out.println("Error: directory architecture file name is not supplied");
                    System.exit(1);
                }
            } else if (thisArgument.equals("-pdbfile")) {
                if (i < args.length) {
                    pdbFileName = args[i++];
                    if (validOption(pdbFileName)) {
                        System.out.println("Error: incorrect pdb file name or missing argument(s)");
                        System.exit(1);
                    }
                } else {
                    System.out.println("Error: pdb file name is not supplied");
                    System.exit(1);
                }
            } else if (thisArgument.equals("-rdctypes")) {                
                while (true) {
                    //System.out.println("asdf");
                    if (i < args.length) {
                        String thisType = args[i++];
                        if (validOption(thisType)) {
                            if (types.isEmpty()) {
                                System.out.println("Error: incorrect RDC types or missing argument(s)");
                                System.exit(1);
                            } else {
                                i--;
                                break;
                            }
                        } else {
                            types.add(thisType);
                        }
                    }
                    else {
                        if (!types.isEmpty()) break;
                        System.out.println("Error: RDC types are not supplied");
                        System.exit(1);
                    }
                }
            } else if (thisArgument.equals("-Szz")) {
                if (i < args.length) {
                    Szz = Double.parseDouble(args[i++]);                    
                } else {
                    System.out.println("Error: Szz is not supplied");
                    System.exit(1);
                }
            } else if (thisArgument.equals("-Sxx_yy")) {
                if (i < args.length) {
                    Sxx_yy = Double.parseDouble(args[i++]);                    
                } else {
                    System.out.println("Error: Sxx_yy is not supplied");
                    System.exit(1);
                }
            } else if (thisArgument.equals("-Sxy")) {
                if (i < args.length) {
                    Sxy = Double.parseDouble(args[i++]);                    
                } else {
                    System.out.println("Error: Sxy is not supplied");
                    System.exit(1);
                }
            } else if (thisArgument.equals("-Sxz")) {
                if (i < args.length) {
                    Sxz = Double.parseDouble(args[i++]);                    
                } else {
                    System.out.println("Error: Sxz is not supplied");
                    System.exit(1);
                }
            } else if (thisArgument.equals("-Syz")) {
                if (i < args.length) {
                    Syz = Double.parseDouble(args[i++]);                    
                } else {
                    System.out.println("Error: Syz is not supplied");
                    System.exit(1);
                }
            } else {
                System.out.println("Error: incorrect argument specification: " + thisArgument);
                System.exit(1);
            }
        }
        
        if (Szz == Double.NaN || Sxx_yy == Double.NaN || Sxy == Double.NaN || Sxz == Double.NaN || Syz == Double.NaN) {
            System.out.println("Error: alignment tensor is not completely specified");
            System.exit(1);
        }

        if (masterDir == null) {
            System.out.println("Error: master directory name is not supplied");
            System.exit(1);
        }
        
        if (pdbFileName == null) {
            System.out.println("Error: pdb file name is not supplied");
            System.exit(1);
        } else {
            __pdb_file = pdbFileName;
        }

        if (directoryArchitectureFile == null) {
            System.out.println("Warning: directory architecture file name is not supplied. It is defaulted to dirArch.txt which must be present in the master directory");
            directoryArchitectureFile = "dirArch.txt";
        }
        
        if (types.isEmpty()) {
            System.out.println("Error: RDC types not supplied");
            System.exit(1);
        } else {
            for (String t : types) {
                t = t.trim();
                switch (t) {
                    case "N_HN":
                        __types_for_pales.add(myDipolarCoupling.Type.N_HN);
                        break;
                    case "CA_C":
                        __types_for_pales.add(myDipolarCoupling.Type.CA_C);
                        break;
                    case "CA_HA":
                        __types_for_pales.add(myDipolarCoupling.Type.CA_HA);
                        break;
                    case "C_N":
                        __types_for_pales.add(myDipolarCoupling.Type.C_N);
                        break;
                    case "C_HN":
                        __types_for_pales.add(myDipolarCoupling.Type.C_HN);
                        break;
                    default:
                        System.out.println("Error: incorrect RDC type supplied");
                        System.exit(1);
                }
            }
        }
        
        composeTheAlignmentTensor:
        {            
            double Syy = (Szz + Sxx_yy) / (-2);

            double[][] mArr = {
                {-Syy - Szz, Sxy, Sxz},
                {Sxy, Syy, Syz},
                {Sxz, Syz, Szz}
            };

            Matrix AT = new Matrix(mArr);
            __alignment_tensors.put(__universal_key, AT);
        }
        
        
        doRdcSimulation:
        {            
            myInputDataAndParameterManager mdpm = myInputDataAndParameterManager.getInstance();
            mdpm.setMasterDirectory(masterDir);
            mdpm.setDirectoryArchitectureFile(directoryArchitectureFile);
            
            doAndTestRdcSimulations();
        }
    }
    
    public static void main(String... args) {      
//        String arguments = "-masterdir EXPERIMENTS_TO_TEST_UTILITIES/1oyc/ -pdbfile EXPERIMENTS_TO_TEST_UTILITIES/1oyc/1oyc_afh.pdb -Szz 8.8465e-04 -Sxx_yy -1.2187e-03 -Sxy -6.5320e-04 -Sxz 6.0936e-05 -Syz 3.4005e-04 -rdctypes N_HN CA_C";
//        args = arguments.split("\\s+"); // Give a set of arguments for testing
//        
//        for (String s : args) {
//            System.out.println(s);
//        }
        
        long startTime = System.currentTimeMillis();
        
        RDCSimulator thisSimulator = new RDCSimulator();
        thisSimulator.parseCommandLineArgumentsAndInvokeRDCSimulator(args);

//        myDipolarCoupling.Type[] typesForPales = new myDipolarCoupling.Type[]{myDipolarCoupling.Type.N_HN,
//            myDipolarCoupling.Type.CA_HA,
//            myDipolarCoupling.Type.CA_C};
//        printRdcTableForPales(thisProtein, new Vector<myDipolarCoupling.Type>(Arrays.asList(typesForPales)));

        long endTime = System.currentTimeMillis();
        double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
        System.out.println("Time elapsed: " + totalTime + " minutes");
    }
}
