/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2012 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, August 04, 2012
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.1, August 04, 2012
 * @author        Lincong Wang (2001-2005)
 * @email         wlincong@cs.dartmouth.edu
 * @organization  Dartmouth College
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */

/**
 * Description of the class
 * A class modified from the Jama numeric package
 */
public class Maths {

   /** sqrt(a^2 + b^2) without under/overflow. **/

    public static double hypot(double a, double b) {
	double r;
	if (Math.abs(a) > Math.abs(b)) {
	    r = b/a;
	    r = Math.abs(a)*Math.sqrt(1+r*r);
	} else if (b != 0) {
	    r = a/b;
	    r = Math.abs(b)*Math.sqrt(1+r*r);
	} else {
	    r = 0.0;
	}
	return r;
    }
    
    /** solve the sin equation:
	a * sin(theta) + b * cos(theta) = c;
     */
    public static boolean sinAngles(double a, double b, double c, double[] solutions){
	double r = hypot(a, b);
	if (r == 0.0)
	    return false;
	else if (Math.abs( c / r) > 1.0)
	    return false;
	double theta0 = Math.asin(c / r);
	double sinCnt = b / r;
	double cosCnt = a / r;
	double cnt = 0.0;
	double theta = 0.0;
	if ( cosCnt >= 0 )
	    cnt = Math.asin(sinCnt);
	else cnt = Math.PI - Math.asin(sinCnt);
	solutions[0] = theta0 - cnt;
	solutions[1] = Math.PI - theta0 - cnt;
	return true;
    }
}
