/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2012 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, August 04, 2012
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.1, August 04, 2012
 * @author        Chittaranjan Tripathy (2007-2012)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */
import java.util.*;
import javax.vecmath.*;

/**
 * Description of the class
 */
public class myAlignmentTensorEstimator {

    /**
     * This method generates an equation consisting all the different internuclear
     * vectors whose RDCs are passed as argument.
     * 
     * @param thisFragment the protein fragment
     * @param rdcVec a vector of dipolar coupling objects
     * @param rdcRatio dipolar coupling gyromagnetic ratio
     * @return A 2D array with six columns corresponding to the five terms computed
     * using the direction cosines of the bond vectors, and last column holds the RDC value
     */
    public static double[][] generateLinearEquationsToComputeAlignmentTensor(final myProtein thisFragment, final Vector<myDipolarCoupling> rdcVec, double rdcRatio) {
        // TODO: Note about the (caveat) Rule-> first atom has to be a heavy atom, second atom can be anything.
        Vector<Double[]> xyzPlusRdcVec = new Vector<Double[]>();

        for (myDipolarCoupling thisDc : rdcVec) {
            String atom1Type = thisDc.getFirstAtomName();
            int firstAtomResidueNumber = thisDc.getFirstAtomResidueNumber();
            String atom2Type = thisDc.getSecondAtomName();
            int secondAtomResidueNumber = thisDc.getSecondAtomResidueNumber();
            //System.out.println("firstAtomName: " + firstAtomName + "secondAtomName: " + secondAtomName);

            double rdc = thisDc.getRdc();
            myResidue resA = thisFragment.residueAt(firstAtomResidueNumber);
            myResidue resB = thisFragment.residueAt(secondAtomResidueNumber);

            if (resA != null && resB != null) {
                String Aname = atom1Type;
                String Bname = atom2Type;
                myAtom atomA = resA.getAtom(atom1Type);
                myAtom atomB = resB.getAtom(atom2Type);

                System.out.println("atomA: " + Aname + " atomB: " + Bname);

                Vector<myAtom> vB = new Vector<myAtom>();

                if (atomA != null) {
                    if (atomB != null) { // means HA for CA type
                        vB.add(atomB);
                    } else if (Bname.endsWith("#")) {
                        // HA2 and HA3 for CA; HG12 and HG13 type for CG1; and similarly for CG2 and others
                        String newBname = Bname.substring(0, Bname.indexOf("#"));
                        myAtom atomB2 = resB.getAtom(newBname + '2');
                        myAtom atomB3 = resB.getAtom(newBname + '3');
                        if (atomB2 != null && atomB3 != null) {
                            vB.add(atomB2);
                            vB.add(atomB3);
                        }
                    }
                    
                    // now we have an atom A and the set of atoms {B} s.t. AB is a
                    // internuclear vector for which RDC has been recorded.

                    Vector<Double> xyzPlusRdcVec2 = new Vector<Double>();

                    for (myAtom B : vB) {
                        myPoint tail = B.getCoordinates();
                        myPoint head = atomA.getCoordinates();

                        myVector3D interNuclearVec = myVector3D.normalize(new myVector3D(tail, head)); // this is a unit vector, so the direction cosines are the components of the vector
                        double cosX = interNuclearVec.cosAlpha();
                        double cosY = interNuclearVec.cosBeta();
                        double cosZ = interNuclearVec.cosGamma();
                        //System.out.println("cosX: " + cosX + "    cosY: " + cosY + "    cosZ: " + cosZ);

                        xyzPlusRdcVec2.add(cosX);
                        xyzPlusRdcVec2.add(cosY);
                        xyzPlusRdcVec2.add(cosZ);
                    }

                    if (xyzPlusRdcVec2.size() != 0) {
                        xyzPlusRdcVec2.add(rdc);
                        Double[] vd = xyzPlusRdcVec2.toArray(new Double[1]); // The argument here is actually proxy
                        xyzPlusRdcVec.add(vd);
                    }
                }
            }
        }

        // These two constants can be useful to model CSA. For RDCs of type A-B# they are just 1.0.
        double lambda1 = 1.0;
        double lambda2 = 1.0;

        System.out.println("xysPlusSize: " + xyzPlusRdcVec.size());

        double[][] equationMatrix = new double[xyzPlusRdcVec.size()][6]; //6 = number of columns. First five columns hold the directional cosines and the last column holds the RDC value.
        for (int i = 0; i < xyzPlusRdcVec.size(); i++) {
            Double[] vd = xyzPlusRdcVec.elementAt(i);
            if (vd.length == 4) {
                double x = vd[0];
                double y = vd[1];
                double z = vd[2];
                double r = vd[3];
                equationMatrix[i][0] = y * y - x * x;
                equationMatrix[i][1] = z * z - x * x;
                equationMatrix[i][2] = 2.0 * x * y;
                equationMatrix[i][3] = 2.0 * x * z;
                equationMatrix[i][4] = 2.0 * y * z;
                equationMatrix[i][5] = r / rdcRatio; //All were converted to CH. TODO: see if this can be factored out.
            } else if (vd.length == 7) {
                double x1 = vd[0];
                double y1 = vd[1];
                double z1 = vd[2];
                double x2 = vd[3];
                double y2 = vd[4];
                double z2 = vd[5];
                double r = vd[6];
                equationMatrix[i][0] = lambda1 * (y1 * y1 - x1 * x1) + lambda2 * (y2 * y2 - x2 * x2);
                equationMatrix[i][1] = lambda1 * (z1 * z1 - x1 * x1) + lambda2 * (z2 * z2 - x2 * x2);
                equationMatrix[i][2] = 2.0 * (lambda1 * x1 * y1 + lambda2 * x2 * y2);
                equationMatrix[i][3] = 2.0 * (lambda1 * x1 * z1 + lambda2 * x2 * z2);
                equationMatrix[i][4] = 2.0 * (lambda1 * y1 * z1 + lambda2 * y2 * z2);
                equationMatrix[i][5] = r / rdcRatio; //All were converted to CH. TODO: see if this can be factored out.
            } else {
                System.out.println("vd.length(): " + vd.length);
                System.out.println("Error: Equation matrix formation for SVD failed xxx");
                System.exit(1);
            }
        }

        return equationMatrix;
    }

    /**
     * This method generates an equation consisting all the different internuclear
     * vectors whose RDCs are passed as argument.
     * 
     * @param thisFragment the protein fragment
     * @param rdcVec a vector of dipolar coupling objects
     * @param rdcRatio dipolar coupling gyromagnetic ratio
     * @return A 2D array with six columns corresponding to the five terms computed
     * using the direction cosines of the bond vectors, and last column holds the RDC value
     */
    // Note: This is the old version which is buggy (no mechanism!) when it comes to resolving '#'; therefore, deprecated on Mar 05, 2010.
    @Deprecated public static double[][] generateLinearEquationsToComputeAlignmentTensorObsolate(final myProtein thisFragment, final Vector<myDipolarCoupling> rdcVec, double rdcRatio) {
        Vector<Double[]> xyzPlusRdcVec = new Vector<Double[]>();
        for (myDipolarCoupling thisDc : rdcVec) {
            String firstAtomName = thisDc.getFirstAtomName();
            int firstAtomResidueNumber = thisDc.getFirstAtomResidueNumber();
            String secondAtomName = thisDc.getSecondAtomName();
            int secondAtomResidueNumber = thisDc.getSecondAtomResidueNumber();
            //System.out.println("firstAtomName: " + firstAtomName + "secondAtomName: " + secondAtomName);

            double rdc = thisDc.getRdc();

            myResidue r1 = thisFragment.residueAt(firstAtomResidueNumber);
            myResidue r2 = thisFragment.residueAt(secondAtomResidueNumber);

            if (r1 != null && r2 != null) {
                myAtom a1 = r1.getAtom(firstAtomName);
                myAtom a2 = r2.getAtom(secondAtomName);

                if (a1 != null && a2 != null) {
                    myAtom tailAtom = a2;
                    myAtom headAtom = a1;

                    myPoint tail = tailAtom.getCoordinates();
                    myPoint head = headAtom.getCoordinates();

                    myVector3D interNuclearVec = myVector3D.normalize(new myVector3D(tail, head)); // this is a unit vector, so the direction cosines are the components of the vector
                    double cosX = interNuclearVec.cosAlpha();
                    double cosY = interNuclearVec.cosBeta();
                    double cosZ = interNuclearVec.cosGamma();
                    //System.out.println("cosX: " + cosX + "    cosY: " + cosY + "    cosZ: " + cosZ);

                    Double[] vd = {cosX, cosY, cosZ, rdc};
                    xyzPlusRdcVec.add(vd);
                }
            }
        }

        double[][] equationMatrix = new double[xyzPlusRdcVec.size()][6]; //6 = number of columns. First five columns hold the directional cosines and the last column holds the RDC value.
        for (int i = 0; i < xyzPlusRdcVec.size(); i++) {
            Double[] vd = xyzPlusRdcVec.elementAt(i);
            double x = vd[0];
            double y = vd[1];
            double z = vd[2];
            double r = vd[3];
            equationMatrix[i][0] = y * y - x * x;
            equationMatrix[i][1] = z * z - x * x;
            equationMatrix[i][2] = 2.0 * x * y;
            equationMatrix[i][3] = 2.0 * x * z;
            equationMatrix[i][4] = 2.0 * y * z;
            equationMatrix[i][5] = r / rdcRatio; //All were converted to CH. TODO: see if this can be factored out.
        }

        return equationMatrix;
    }

    /**
     * Compute the alignment tensor from a given protein structure and sets of
     * dipolar couplings using singular value decomposition (SVD) and then
     * diagonalize the alignment tensor matrix to compute the diagonalized
     * elements and rotation matrix to one of the four possible principal order frames (POFs).
     * Additionally, compute the RDC rmsds for the two types of RDCs used.
     *
     * @param thisFragment the protein structure to be used to compute the alignment tensor
     * @param nhRdc a vector containing the N-HN RDCs
     * @param cahaRdc a vector containing the CA-HA RDCs
     * @param nhRdcBackCal this vector contains the back-computed N-HN RDCs to be returned to the caller
     * @param chRdcBackCal this vector contains the back-computed CA-HA RDCs to be returned to the caller
     * @param eigenValues eigenValues[] = {Syy, Szz, nhRdcRmsd, chRdcRmsd}
     * @return the rotation matrix rotating the original frame to one of four possible POFs
     */
    @Deprecated public static Matrix bestFitUsingSVD(final myProtein thisFragment, final Vector<myDipolarCoupling> nhRdc, final Vector<myDipolarCoupling> cahaRdc,
            Vector<myDipolarCoupling> nhRdcBackCal, Vector<myDipolarCoupling> chRdcBackCal, double[] eigenValues) {

        //System.out.println("I am inside bestFitUsingSVD");

        // clean up the vectors that hold the back computed RDCs
        nhRdcBackCal.clear();
        chRdcBackCal.clear();

        //The NH and CH RDC data: The two RDCs are concatenated together.
        double[][] nhArray = generateLinearEquationsToComputeAlignmentTensor(thisFragment, nhRdc, myConstantsForRdcs.nhRatio);
        double[][] cahaArray = generateLinearEquationsToComputeAlignmentTensor(thisFragment, cahaRdc, myConstantsForRdcs.cahaRatio);
        int nhNO = nhArray.length;   //number of rows of the NH Rdcs

        int cahaNO = cahaArray.length;
        int m = nhNO + cahaNO;	// total number of rows
        double[][] equ = new double[m][5];
        double[] b = new double[m];

        // stacking the first set of equations
        for (int i = 0; i < nhNO; i++) {
            for (int j = 0; j < 5; j++) {
                equ[i][j] = nhArray[i][j];
            }
            b[i] = nhArray[i][5]; // since the last column has RDC values
        }

        // stacking the second set of equations
        for (int i = nhNO; i < nhNO + cahaNO; i++) {
            for (int j = 0; j < 5; j++) {
                equ[i][j] = cahaArray[i - nhNO][j];
            }
            b[i] = cahaArray[i - nhNO][5];
        }

        int n = 5; // since there are 5 unknowns in the alignment tensor
        Matrix A = new Matrix(equ, m, n);

        // code to test how the matrix looks like.
        System.out.println("Printing the A matrix while doing the best fit");
        A.print(4, 4);

        SingularValueDecomposition SVD = A.svd();
        double[] singularValues = SVD.getSingularValues();

        System.out.print("The singular values are: ");
        for (int i = 0; i < singularValues.length; i++) {
            System.out.print(singularValues[i] + "    ");
        }
        System.out.println();
        System.out.println("Number of singular values " + singularValues.length);

        double[] singulars = new double[6];
        for (int k = 0; k < singularValues.length; k++) {
            //System.out.println(singularvalues[k]);
            if (singularValues[k] != 0.0) {
                singulars[k] = 1 / singularValues[k];
            } else {
                System.out.println("Error: Zero singluar values");
                System.exit(1);
            }
        }

        double[][] ArrS = {{singulars[0], 0.0, 0.0, 0.0, 0.0},
            {0.0, singulars[1], 0.0, 0.0, 0.0},
            {0.0, 0.0, singulars[2], 0.0, 0.0},
            {0.0, 0.0, 0.0, singulars[3], 0.0},
            {0.0, 0.0, 0.0, 0.0, singulars[4]}
        };

        Matrix U = SVD.getU();
        Matrix V = SVD.getV();
        //Matrix S = SVD.getS(); // This component of SVD is not used
        Matrix invS = new Matrix(ArrS, 5, 5);     
        
        try {
            check(A, SVD.getU().times(SVD.getS().times(SVD.getV().transpose())));
            System.out.println("Singular Value Decomposition (SVD): succeeds");
        } catch (java.lang.RuntimeException e) {
            System.out.println("Singular Value Decomposition (SVD): fails");
        }
        
        double[] saupe = V.times(invS.times(U.transpose().times(b)));
        double Syy = saupe[0], Szz = saupe[1], Sxy = saupe[2], Sxz = saupe[3], Syz = saupe[4];

        // So here is the alignment tensor
        double[][] mArr = {
            {-Syy - Szz, Sxy, Sxz},
            {Sxy, Syy, Syz},
            {Sxz, Syz, Szz}
        };

        // Diagonalize the alignment tensor
        Matrix mm = new Matrix(mArr);
        //mm.print(8, 3);
        EigenvalueDecomposition eigs = new EigenvalueDecomposition(mm, true);
        Matrix VV = eigs.getV();
        Matrix DD = eigs.getD();
        VV.print(5, 12);
        DD.print(5, 12);
        double Sxx = DD.get(0, 0);
        Syy = DD.get(1, 1);
        Szz = DD.get(2, 2);
        System.out.println("Sxx: " + Sxx + "  Syy: " + Syy + "  Szz: " + Szz);
        eigenValues[0] = Syy;
        eigenValues[1] = Szz;

        myProtein rotatedFragment = new myProtein(thisFragment);
        rotatedFragment.rotate(new myMatrix(VV.transpose().getArrayCopy()));

        Vector<myPhiPsi> firstDihedralSet = thisFragment.computeDihedrals();
        Vector<myPhiPsi> secondDihedradSet = rotatedFragment.computeDihedrals();
        double eps = 1.0E-10; // TODO: remove this hardcode

        // Compute NH RDC deviations and NH RDC rmsd
        System.out.println("Printing NH RDC deviations ");
        Vector<myDipolarCoupling> nhRdcBC = backComputeRdcs(thisFragment, nhRdc.elementAt(0), VV.transpose(), Sxx, Syy, Szz, myConstantsForRdcs.nhRatio);
        double nhRmsd = computeRdcRmsd(thisFragment, nhRdc, VV.transpose(), Sxx, Syy, Szz, myConstantsForRdcs.nhRatio);
        eigenValues[2] = nhRmsd;
        System.out.println("NH RDC RMSD: " + nhRmsd);
        nhRdcBackCal.addAll(nhRdcBC);

        // Compute CH RDC deviations and CH RDC rmsd
        System.out.println("Printing CH RDC deviations ");
        Vector<myDipolarCoupling> chRdcBC = backComputeRdcs(thisFragment, cahaRdc.elementAt(0), VV.transpose(), Sxx, Syy, Szz, myConstantsForRdcs.cahaRatio);
        double chRmsd = computeRdcRmsd(thisFragment, cahaRdc, VV.transpose(), Sxx, Syy, Szz, myConstantsForRdcs.cahaRatio);
        eigenValues[3] = chRmsd;
        System.out.println("CH RDC RMSD: " + chRmsd);
        chRdcBackCal.addAll(chRdcBC);

        //Make sure what returned has the same handedness as the input PDB
        if (Math.abs(firstDihedralSet.elementAt(0).getPsi() - secondDihedradSet.elementAt(0).getPsi()) < eps) {
            System.out.println(" The same (i.e., the same handedness when checked from bestFit)");
            return VV.transpose();
        }
        System.out.println(" Differ (i.e., different handedness when checked from bestFit)");
        return Const.mLeftHand.times(VV.transpose());
// 	return VV.transpose();
    }

    /**
     * Compute the alignment tensor from a given protein structure and sets of
     * dipolar couplings using singular value decomposition (SVD) and then
     * diagonalize the alignment tensor matrix to compute the diagonalized
     * elements and rotation matrix to one of the four possible principal order frames (POFs).
     * Additionally, compute the RDC rmsds for the two types of RDCs used.
     *
     * @param thisFragment the protein structure to be used to compute the alignment tensor
     * @param rdcCsaTableForThisMedium the RDC table for this medium
     * @param typesOfRdcsToBeUsedInTensorRefinement the types of RDCs to be used in tensor estimation
     * @param rdcCsaTableForThisMediumBackCal the back-computed RDC table
     * @param eigenValues the eigenvalues {Syy, Szz}
     * @return the rotation of the molecular frame relative to the principal order frame
     */
    public static Matrix bestFitUsingSVD(final myProtein thisFragment,
            Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMedium,
            Vector<myDipolarCoupling.Type> typesOfRdcsToBeUsedInTensorRefinement,
            Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMediumBackCal, double[] eigenValues) {

        rdcCsaTableForThisMediumBackCal.clear(); // clear off all old contents

        Vector<double[][]> eqnCoeffs = new Vector<double[][]>();

        // System.out.println("I am inside bestFitUsingSVD (new Ver)");
        for (myDipolarCoupling.Type type : typesOfRdcsToBeUsedInTensorRefinement) {
            Vector<myDipolarCoupling> rdcVec = rdcCsaTableForThisMedium.get(type);                        
            double[][] eqnArray = generateLinearEquationsToComputeAlignmentTensor(thisFragment, rdcVec, myConstantsForRdcs.DmaxScaled.get(type.toString()));            
            eqnCoeffs.add(eqnArray);
        }

        int totalNumberOfRows = 0;
        for (double[][] eqnArray : eqnCoeffs) {
            totalNumberOfRows += eqnArray.length;
        }

        double[][] equ = new double[totalNumberOfRows][5];
        double[] b = new double[totalNumberOfRows];

        int m = totalNumberOfRows;

        // Now stack the equations
        int currentRow = 0;
        for (double[][] eqnArray : eqnCoeffs) {
            for (int i = 0; i < eqnArray.length; i++) {
                for (int j = 0; j < 5; j++) {
                    equ[currentRow + i][j] = eqnArray[i][j];
                }
                b[currentRow + i] = eqnArray[i][5];
            }
            currentRow += eqnArray.length;
        }

        int n = 5; // since there are 5 unknowns in the alignment tensor
        Matrix A = new Matrix(equ, m, n);

        // code to test how the matrix looks like.
        System.out.println("Printing the A matrix while doing the best fit");
        A.print(4, 4);

        SingularValueDecomposition SVD = A.svd();
        double[] singularValues = SVD.getSingularValues();

        System.out.print("The singular values are: ");
        for (int i = 0; i < singularValues.length; i++) {
            System.out.print(singularValues[i] + "    ");
        }
        System.out.println();
        System.out.println("Number of singular values " + singularValues.length);

        double[] singulars = new double[6];
        for (int k = 0; k < singularValues.length; k++) {
            //System.out.println(singularvalues[k]);
            if (singularValues[k] != 0.0) {
                singulars[k] = 1 / singularValues[k];
            } else {
                System.out.println("Error: Zero singluar values");
                System.exit(1);
            }
        }

        double[][] ArrS = {{singulars[0], 0.0, 0.0, 0.0, 0.0},
            {0.0, singulars[1], 0.0, 0.0, 0.0},
            {0.0, 0.0, singulars[2], 0.0, 0.0},
            {0.0, 0.0, 0.0, singulars[3], 0.0},
            {0.0, 0.0, 0.0, 0.0, singulars[4]}
        };

        Matrix U = SVD.getU();
        Matrix V = SVD.getV();
        //Matrix S = SVD.getS(); // This component of SVD is not used
        Matrix invS = new Matrix(ArrS, 5, 5);

        try {
            check(A, SVD.getU().times(SVD.getS().times(SVD.getV().transpose())));
            System.out.println("Singular Value Decomposition (SVD): succeeds");
        } catch (java.lang.RuntimeException e) {
            System.out.println("Singular Value Decomposition (SVD): fails");
        }

        double[] saupe = V.times(invS.times(U.transpose().times(b)));
        double Syy = saupe[0], Szz = saupe[1], Sxy = saupe[2], Sxz = saupe[3], Syz = saupe[4];

        // So here is the alignment tensor
        double[][] mArr = {
            {-Syy - Szz, Sxy, Sxz},
            {Sxy, Syy, Syz},
            {Sxz, Syz, Szz}
        };

        // Diagonalize the alignment tensor
        Matrix mm = new Matrix(mArr);
        //mm.print(8, 3);
        EigenvalueDecomposition eigs = new EigenvalueDecomposition(mm, true);
        Matrix VV = eigs.getV();
        Matrix DD = eigs.getD();
        VV.print(5, 12);
        DD.print(5, 12);
        double Sxx = DD.get(0, 0);
        Syy = DD.get(1, 1);
        Szz = DD.get(2, 2);
        System.out.println("Sxx: " + Sxx + "  Syy: " + Syy + "  Szz: " + Szz);
        eigenValues[0] = Syy;
        eigenValues[1] = Szz;
        
        System.out.println("The rhombicity is " + myBackboneModeler.computeRhombicity(Syy, Szz));

        myProtein rotatedFragment = new myProtein(thisFragment);
        rotatedFragment.rotate(new myMatrix(VV.transpose().getArrayCopy()));

        Vector<myPhiPsi> firstDihedralSet = thisFragment.computeDihedrals();
        Vector<myPhiPsi> secondDihedradSet = rotatedFragment.computeDihedrals();

        // Compute the RDC deviations and print the RDC rmsds
        for (myDipolarCoupling.Type type : typesOfRdcsToBeUsedInTensorRefinement) {
            System.out.println("Printing " + type.toString() + " RDC deviations");
            Vector<myDipolarCoupling> rdcBackCal = backComputeRdcs(thisFragment, type, VV.transpose(), Sxx, Syy, Szz, myConstantsForRdcs.DmaxScaled.get(type.toString()));

            Vector<myDipolarCoupling> rdcVec = rdcCsaTableForThisMedium.get(type);
            double rdcRmsd = computeRdcRmsd(thisFragment, rdcVec, VV.transpose(), Sxx, Syy, Szz, myConstantsForRdcs.DmaxScaled.get(type.toString()));
            System.out.println(type.toString() + " RDC rmsd: " + rdcRmsd);
            rdcCsaTableForThisMediumBackCal.put(type, rdcBackCal);
        }

        //Make sure what returned has the same handedness as the input PDB
        if (Math.abs(firstDihedralSet.elementAt(0).getPsi() - secondDihedradSet.elementAt(0).getPsi()) < myMiscConstants.eps) {
            System.out.println(" The same (i.e., the same handedness when checked from bestFit)");
            return VV.transpose();
        }
        System.out.println(" Differ (i.e., different handedness when checked from bestFit)");
        return Const.mLeftHand.times(VV.transpose());
// 	return VV.transpose();
    }

    /**
     * Returns a vector of back-computed RDCs given the diagonalized alignment tensor
     * components and a protein Structure.
     * 
     * @param p the protein structure
     * @param dcType the type of dipolar coupling, i.e., the identity of internuclear vector
     * @param RotToPOF the rotation matrix that rotates the protein p into one of the four possible POFs
     * @param Sxx the diagonalized alignment tensor component
     * @param Syy the diagonalized alignment tensor component
     * @param Szz the diagonalized alignment tensor component
     * @param rdcRatio the dipolar coupling scaling constant
     * @return a vector containing the back-computed RDCs
     */
    @Deprecated public static Vector<myDipolarCoupling> backComputeRdcs(final myProtein p, final myDipolarCoupling dcType,
            Matrix RotToPOF, double Sxx, double Syy, double Szz, double rdcRatio) {
        // Note: about the (caveat) Rule-> first atom has to be a heavy atom, second atom can be anything.
        Vector<myDipolarCoupling> rdcBackCalVec = new Vector<myDipolarCoupling>();

        // get the RDC type from dcType object
        String atom1Type = dcType.getFirstAtomName();
        String atom2Type = dcType.getSecondAtomName();
        int firstAtomResNum = dcType.getFirstAtomResidueNumber();
        int secondAtomResNum = dcType.getSecondAtomResidueNumber();
        int diffRes = secondAtomResNum - firstAtomResNum;

        for (myResidue thisRes : p) {
            myResidue resA = thisRes;
            myResidue resB = p.residueAt(resA.getResidueNumber() + diffRes);
            if (resA == null || resB == null) {
                continue;
            } else { //if (atom1Type.length() == 2 && atom1Type.startsWith("C") && atom2Type.length() == 2) {
                Vector<myAtom> vA = new Vector<myAtom>();
                myAtom atomA1 = null, atomA2 = null;
                atomA1 = resA.getAtom(atom1Type);
                if (atomA1 != null) {
                    vA.add(atomA1);
                } else { // means CG1 CG2 type
                    atomA1 = resA.getAtom(atom1Type + '1');
                    atomA2 = resA.getAtom(atom1Type + '2');
                    if (atomA1 != null && atomA2 != null) {
                        vA.add(atomA1);
                        vA.add(atomA2);
                    }
                }

                for (myAtom A : vA) {
                    Vector<myAtom> vB = new Vector<myAtom>();
                    String Aname = A.getAtomName();
                    String Bname = atom2Type;
                    myAtom atomB = resB.getAtom(Bname);
                    if (atomB != null) { // means HA for CA
                        vB.add(atomB);
                    } else if (resB.getAtom(Bname + '2') != null && resB.getAtom(Bname + '3') != null) { // HA2 and HA2 for CA
                        vB.add(resB.getAtom(Bname + '2'));
                        vB.add(resB.getAtom(Bname + '3'));
                    } else if (Character.isDigit(Aname.charAt(Aname.length() - 1))) { // means CG1 CG2 type
                        myAtom atomB2 = resB.getAtom(Bname + Aname.charAt(Aname.length() - 1) + '2');   // e.g. HG12 type
                        myAtom atomB3 = resB.getAtom(Bname + Aname.charAt(Aname.length() - 1) + '3');   // e.g. HG13 type
                        if (atomB2 != null && atomB3 != null) {
                            vB.add(atomB2);
                            vB.add(atomB3);
                        }
                    }

                    // now we have an atom A and the set of atoms {B} s.t. AB is a
                    // internuclear vector for which RDC has to be back-computed.
                    double rdcBackCal = 0.0;
                    for (myAtom B : vB) {
                        myVector3D interNuclearVec = myVector3D.normalize(myVector3D.rotate(new myVector3D(A.getCoordinates(), B.getCoordinates()), RotToPOF)); // this is a unit vector, so the direction cosines are the components of the vector
                        double x = interNuclearVec.cosAlpha();
                        double y = interNuclearVec.cosBeta();
                        double z = interNuclearVec.cosGamma();
                        //System.out.println("cosX: " + cosX + "    cosY: " + cosY + "    cosZ: " + cosZ);
                        double rdcCalOnThisVector = rdcRatio * (x * x * Sxx + y * y * Syy + z * z * Szz);
                        rdcBackCal += rdcCalOnThisVector;
                    }

                    if (vB.size() == 1) {
                        rdcBackCalVec.add(new myDipolarCoupling(A.getAtomName(), resA.getResidueNumber(), Bname, resB.getResidueNumber(), rdcBackCal, 0.0));
                    } else if (vB.size() == 2) {
                        String newBname = vB.elementAt(0).getAtomName();
                        assert Character.isDigit(newBname.charAt(newBname.length() - 1)) == true : "Error: incorrect second atom name while back-computing RDC";
                        newBname = newBname.substring(0, newBname.length() - 1) + '#'; // remove last digit and replace with '#'
                        rdcBackCalVec.add(new myDipolarCoupling(A.getAtomName(), resA.getResidueNumber(), newBname, resB.getResidueNumber(), rdcBackCal, 0.0));
                    }
                }
            }
        }
        return rdcBackCalVec;
    }

    /**
     * Return a vector of back-computed RDCs given the diagonalized alignment
     * tensor components and a protein Structure.
     *
     * @param p the protein structure
     * @param type the dipolar coupling type
     * @param RotToPOF the rotation matrix that rotates the protein p into one of the four possible POFs
     * @param Sxx the diagonalized alignment tensor component
     * @param Syy the diagonalized alignment tensor component
     * @param Szz the diagonalized alignment tensor component
     * @param rdcRatio the dipolar coupling scaling constant
     * @return a vector containing the back-computed RDCs
     */
    public static Vector<myDipolarCoupling> backComputeRdcs(final myProtein p, myDipolarCoupling.Type type,
            Matrix RotToPOF, double Sxx, double Syy, double Szz, double rdcRatio) {
        // TODO: Note about the (caveat) Rule-> first atom has to be a heavy atom, second atom can be anything.
        Vector<myDipolarCoupling> rdcBackCalVec = new Vector<myDipolarCoupling>();

        // get the RDC type from dcType object
        myDipolarCoupling dcType = new myDipolarCoupling(type);
        String atom1Type = dcType.getFirstAtomName();
        String atom2Type = dcType.getSecondAtomName();
        int firstAtomResNum = dcType.getFirstAtomResidueNumber();
        int secondAtomResNum = dcType.getSecondAtomResidueNumber();
        int diffRes = secondAtomResNum - firstAtomResNum;

        for (myResidue thisRes : p) {
            myResidue resA = thisRes;
            myResidue resB = p.residueAt(resA.getResidueNumber() + diffRes);
            if (resA == null || resB == null) {
                continue;
            } else { //if (atom1Type.length() == 2 && atom1Type.startsWith("C") && atom2Type.length() == 2) {
                Vector<myAtom> vA = new Vector<myAtom>();
                myAtom atomA1 = null, atomA2 = null;
                atomA1 = resA.getAtom(atom1Type);
                if (atomA1 != null) {
                    vA.add(atomA1);
                } else { // means CG1 CG2 type
                    atomA1 = resA.getAtom(atom1Type + '1');
                    atomA2 = resA.getAtom(atom1Type + '2');
                    if (atomA1 != null && atomA2 != null) {
                        vA.add(atomA1);
                        vA.add(atomA2);
                    }
                }

                for (myAtom A : vA) {
                    Vector<myAtom> vB = new Vector<myAtom>();
                    String Aname = A.getAtomName();
                    String Bname = atom2Type;
                    myAtom atomB = resB.getAtom(Bname);
                    if (atomB != null) { // means HA for CA
                        vB.add(atomB);
                    } else if (resB.getAtom(Bname + '2') != null && resB.getAtom(Bname + '3') != null) { // HA2 and HA2 for CA
                        vB.add(resB.getAtom(Bname + '2'));
                        vB.add(resB.getAtom(Bname + '3'));
                    } else if (Character.isDigit(Aname.charAt(Aname.length() - 1))) { // means CG1 CG2 type
                        myAtom atomB2 = resB.getAtom(Bname + Aname.charAt(Aname.length() - 1) + '2');   // e.g. HG12 type
                        myAtom atomB3 = resB.getAtom(Bname + Aname.charAt(Aname.length() - 1) + '3');   // e.g. HG13 type
                        if (atomB2 != null && atomB3 != null) {
                            vB.add(atomB2);
                            vB.add(atomB3);
                        }
                    }

                    // now we have an atom A and the set of atoms {B} s.t. AB is a
                    // internuclear vector for which RDC has to be back-computed.
                    double rdcBackCal = 0.0;
                    for (myAtom B : vB) {
                        myVector3D interNuclearVec = myVector3D.normalize(myVector3D.rotate(new myVector3D(A.getCoordinates(), B.getCoordinates()), RotToPOF)); // this is a unit vector, so the direction cosines are the components of the vector
                        double x = interNuclearVec.cosAlpha();
                        double y = interNuclearVec.cosBeta();
                        double z = interNuclearVec.cosGamma();
                        //System.out.println("cosX: " + cosX + "    cosY: " + cosY + "    cosZ: " + cosZ);
                        double rdcCalOnThisVector = rdcRatio * (x * x * Sxx + y * y * Syy + z * z * Szz);
                        rdcBackCal += rdcCalOnThisVector;
                    }

                    if (vB.size() == 1) {
                        rdcBackCalVec.add(new myDipolarCoupling(A.getAtomName(), resA.getResidueNumber(), Bname, resB.getResidueNumber(), rdcBackCal, 0.0));
                    } else if (vB.size() == 2) {
                        String newBname = vB.elementAt(0).getAtomName();
                        assert Character.isDigit(newBname.charAt(newBname.length() - 1)) == true : "Error: incorrect second atom name while back-computing RDC";
                        newBname = newBname.substring(0, newBname.length() - 1) + '#'; // remove last digit and replace with '#'
                        rdcBackCalVec.add(new myDipolarCoupling(A.getAtomName(), resA.getResidueNumber(), newBname, resB.getResidueNumber(), rdcBackCal, 0.0));
                    }
                }
            }
        }
        return rdcBackCalVec;
    }

    /**
     * Compute the RDC rmsd given a protein structure, the rotation matrix that
     * rotates the protein to one of the POFs, the diagonalized alignment tensor
     * components, and the set of RDCs for a particular inter-nuclear vector type.
     *
     * @param p the protein for which the RDCs are to be back computed
     * @param rdcVec the vector containing the experimental RDCs
     * @param RotToPOF the matrix used to rotate the original PDB frame to one of the four POFs
     * @param Sxx the diagonalized tensor component
     * @param Syy the diagonalized tensor component
     * @param Szz the diagonalized tensor component
     * @param rdcRatio the relative strength of RDCs     
     * @return the rmsd between the back computed RDCs and the experimental RDCs
     */
    public static double computeRdcRmsd(final myProtein p, final Vector<myDipolarCoupling> rdcVec,
            Matrix RotToPOF, double Sxx, double Syy, double Szz, double rdcRatio) {

        myDipolarCoupling.Type dcType = rdcVec.elementAt(0).getType(); // = new myDipolarCoupling(atom1Type, atom1ResNum, atom2Type, atom2ResNum, 0.0, 0.0); // This only provides the type of RDC
        Vector<myDipolarCoupling> rdcBackCalVec = backComputeRdcs(p, dcType, RotToPOF, Sxx, Syy, Szz, rdcRatio);

        double squaredDev = 0.0;
        int effectiveNumberOfRdcsMatchedWithVectors = 0;

        // Note: The procedure below takes O(n^2) time, which can be made linear i.e., O(n)
        // if we sort rdcVec and rdcBackCalVec before iterating.
        for (myDipolarCoupling ddExp : rdcVec) {
            String firstAtomType = ddExp.getFirstAtomName();
            String secondAtomType = ddExp.getSecondAtomName();
            int firstAtomResNum = ddExp.getFirstAtomResidueNumber();
            int secondAtomResNum = ddExp.getSecondAtomResidueNumber();

            for (myDipolarCoupling ddCal : rdcBackCalVec) {
                if (firstAtomType.equalsIgnoreCase(ddCal.getFirstAtomName()) && firstAtomResNum == ddCal.getFirstAtomResidueNumber() &&
                        secondAtomType.equalsIgnoreCase(ddCal.getSecondAtomName()) && secondAtomResNum == ddCal.getSecondAtomResidueNumber()) {
                    // we have a back-computed RDC for an expreimental RDC
                    double rdcExp = ddExp.getRdc();
                    double rdcCal =  ddCal.getRdc();
                    System.out.println(ddExp.getResidueNumber() + "    " + ddCal.getRdc() + "    " + ddExp.getRdc() + "    " + (ddCal.getRdc() - ddExp.getRdc()));
                    squaredDev += (rdcCal - rdcExp) * (rdcCal - rdcExp);
                    effectiveNumberOfRdcsMatchedWithVectors++;
                }
            }
        }
        if (effectiveNumberOfRdcsMatchedWithVectors == 0) {
            System.out.println("Error: There exists no RDC that match an internuclear vector and vice versa");
            System.exit(1);
        }
        squaredDev = Math.sqrt(squaredDev / effectiveNumberOfRdcsMatchedWithVectors);
        //System.out.println("RDC rmsd: " + squaredDev);
        return squaredDev;
    }

    /**
     * Given the protein structure, the rotation matrix that rotates the protein
     * to one of the POFs, the diagonalized alignment tensor omponents, and the 
     * set of RDCs for a particular inter-nuclear vector type, return a pair whose
     * first element is the sum of squared deviations of the back-computed
     * RDCs from experimental RDCs, and the second element is the number of
     * RDC-and-inter-nulcear-vector matches. Note that, given such a pair it is
     * trivial to compute the RDC rmsd which is sqrt(pair.first() / pair.second()).
     * 
     * @param p the protein for which the RDCs are to be back computed
     * @param rdcVec the vector containing the experimental RDCs
     * @param RotToPOF the matrix used to rotate the original PDB frame to one of the four POFs
     * @param Sxx the diagonalized tensor component
     * @param Syy the diagonalized tensor component
     * @param Szz the diagonalized tensor component
     * @param rdcRatio the relative strength of RDCs
     * @return a pair whose first element is the sum of squared deviations of
     * back-computed RDCs from experimental RDCs, and the second element is the
     * number of RDC-and-inter-nulcear-vector matches
     */
    public static myPair<Double, Integer> computeRdcSumOfSquaredDeviation(final myProtein p, final Vector<myDipolarCoupling> rdcVec,
            Matrix RotToPOF, double Sxx, double Syy, double Szz, double rdcRatio) {
        myDipolarCoupling.Type dcType = rdcVec.elementAt(0).getType();
        Vector<myDipolarCoupling> rdcBackCalVec = backComputeRdcs(p, dcType, RotToPOF, Sxx, Syy, Szz, rdcRatio);

        double squaredDev = 0.0;
        int effectiveNumberOfRdcsMatchedWithVectors = 0;

        // Note: The procedure below takes O(n^2) time, which can be made linear i.e., O(n)
        // if we sort rdcVec and rdcBackCalVec before iterating.
        for (myDipolarCoupling ddExp : rdcVec) {
            String firstAtomType = ddExp.getFirstAtomName();
            String secondAtomType = ddExp.getSecondAtomName();
            int firstAtomResNum = ddExp.getFirstAtomResidueNumber();
            int secondAtomResNum = ddExp.getSecondAtomResidueNumber();

            for (myDipolarCoupling ddCal : rdcBackCalVec) {
                if (firstAtomType.equalsIgnoreCase(ddCal.getFirstAtomName()) && firstAtomResNum == ddCal.getFirstAtomResidueNumber() &&
                        secondAtomType.equalsIgnoreCase(ddCal.getSecondAtomName()) && secondAtomResNum == ddCal.getSecondAtomResidueNumber()) {
                    // we have a back-computed RDC for an expreimental RDC
                    double rdcExp = ddExp.getRdc();
                    double rdcCal =  ddCal.getRdc();
                    //System.out.println(ddExp.getResidueNumber() + "    " + ddCal.getRdc() + "    " + ddExp.getRdc() + "    " + (ddCal.getRdc() - ddExp.getRdc()));
                    squaredDev += (rdcCal - rdcExp) * (rdcCal - rdcExp);
                    effectiveNumberOfRdcsMatchedWithVectors++;
                }
            }
        }
        if (effectiveNumberOfRdcsMatchedWithVectors == 0) {
            System.out.println("Error: There exists no RDC that match an internuclear vector and vice versa");
            System.exit(1);
        }
        //squaredDev = Math.sqrt(squaredDev / effectiveNumberOfRdcsMatchedWithVectors);
        //System.out.println("RDC rmsd: " + squaredDev);
        return new myPair<Double, Integer>(squaredDev, effectiveNumberOfRdcsMatchedWithVectors);
    }

    /**
     * Given the protein structure, the diagonalized alignment tensor components
     * and the sets of RDCs do a grid search to find an optimal rotation to one
     * of the principal order frames (POFs) so that the RDC rmsd is minimized.
     *
     * @param thisFragment the protein structure to be used for RDC fit
     * @param nhRdcVec a vector containing the N-HN RDCs
     * @param chRdcVec a vector containing the CA-HA RDCs
     * @param Syy the diagonalized alignment tensor component
     * @param Szz the diagonalized alignment tensor component
     * @param resolution the angular resolution for grid search to estimate the best fit
     * @param rmsds 
     * @return the rotation matrix rotating the frame of the fragment to one of the four possible POFs
     */
    @Deprecated public static Matrix bestFitUsingGridSearchForGivenAT(final myProtein thisFragment, final Vector<myDipolarCoupling> nhRdcVec, final Vector<myDipolarCoupling> chRdcVec,
            final double Syy, final double Szz, final double resolution, double[] rmsds) {
        double [][] mat = new double[3][3];
        double alpha = 0.0, beta = 0.0, gamma = 0.0;
        double alphaBest = 0.0, betaBest = 0.0, gammaBest = 0.0;

        double bestSumOfSquaredDeviation = Double.MAX_VALUE;

        //Do the grid search ( 0 =< alpha <360, 0 =< beta <180, 0 =< gamma <360)
        for (int i = 0; i <= (180 / resolution); i++) {
            alpha = i * resolution * Math.PI / 180.0;
            for (int j = 0; j <= (180 / resolution); j++) {
                beta = j * resolution * Math.PI / 180.0;
                for (int k = 0; k <= (180 / resolution); k++) {
                    gamma = k * resolution * Math.PI / 180.0;

                    //Generate the Euler matrix
                    mat[0][0] = Math.cos(alpha) * Math.cos(beta) * Math.cos(gamma) - Math.sin(alpha) * Math.sin(gamma);
                    mat[0][1] = Math.sin(alpha) * Math.cos(beta) * Math.cos(gamma) + Math.cos(alpha) * Math.sin(gamma);
                    mat[0][2] = -Math.sin(beta) * Math.cos(gamma);
                    mat[1][0] = -Math.cos(alpha) * Math.cos(beta) * Math.sin(gamma) - Math.sin(alpha) * Math.cos(gamma);
                    mat[1][1] = -Math.sin(alpha) * Math.cos(beta) * Math.sin(gamma) + Math.cos(alpha) * Math.cos(gamma);
                    mat[1][2] = Math.sin(gamma) * Math.sin(beta);
                    mat[2][0] = Math.sin(beta) * Math.cos(alpha);
                    mat[2][1] = Math.sin(alpha) * Math.sin(beta);
                    mat[2][2] = Math.cos(beta);

                    Matrix R = new Matrix(mat);
                    
                    myPair<Double, Integer> nhDev =   computeRdcSumOfSquaredDeviation(thisFragment, nhRdcVec, R, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.nhRatio);
                    myPair<Double, Integer> chDev = computeRdcSumOfSquaredDeviation(thisFragment, chRdcVec, R, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.cahaRatio);

                    if (nhDev.first() + chDev.first() < bestSumOfSquaredDeviation) {
                        System.out.println("Previous Score: " + bestSumOfSquaredDeviation + "    Current Score: " + (nhDev.first() + chDev.first()) +
                                "    NH RDC RMSD: " + Math.sqrt(nhDev.first() / nhDev.second()) + "    CH RDC RMSD: " + Math.sqrt(chDev.first() / chDev.second()));
                        bestSumOfSquaredDeviation = nhDev.first() + chDev.first();
                        //System.out.println("*****" + nhDev.second() + "  ******" + chDev.second());
                        alphaBest = alpha;
                        betaBest = beta;
                        gammaBest = gamma;
                        System.out.println(" Best Euler so far: " + alphaBest + "  " + betaBest + "  " + gammaBest);
                    }
                }
            }
        }

        System.out.println(" Euler " + alphaBest + "  " + betaBest + "  " + gammaBest);
        Matrix RBest = Matrix.eulerMat(alphaBest, betaBest, gammaBest);

        // Compute NH RDC deviations and NH RDC rmsd
        System.out.println("Printing NH RDC deviations ");
        //Vector<myDipolarCoupling> nhRdcBC = backComputeRdcs(thisFragment, nhRdcVec.elementAt(0), RBest, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.nhRatio);
        double nhRmsd = computeRdcRmsd(thisFragment, nhRdcVec, RBest, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.nhRatio);
        System.out.println("NH RDC RMSD: " + nhRmsd);

        // Compute CH RDC deviations and CH RDC rmsd
        System.out.println("Printing CH RDC deviations ");
        //Vector<myDipolarCoupling> chRdcBC = backComputeRdcs(thisFragment, chRdcVec.elementAt(0), RBest, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.cahaRatio);
        double chRmsd = computeRdcRmsd(thisFragment, chRdcVec, RBest, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.cahaRatio);
        System.out.println("CH RDC RMSD: " + chRmsd);

        return RBest;
    }

    /**
     * Given the protein structure, the diagonalized alignment tensor components
     * and the sets of RDCs do a grid search to find an optimal rotation to one
     * of the principal order frames (POFs) so that the RDC rmsd is minimized.
     *
     * @param thisFragment the protein structure to be used for RDC fit
     * @param rdcCsaTableForThisMedium the RDC table
     * @param typesOfRdcsToBeUsedInTensorRefinement the types of RDCs to be used
     * @param rdcCsaTableForThisMediumBackCal the back-computed RDC table
     * @param Syy the diagonalized alignment tensor component
     * @param Szz the diagonalized alignment tensor component
     * @param resolution the angular resolution for grid search to estimate the best fit
     * @return the rotation matrix rotating the frame of the fragment to one of the four possible principal order frames
     */
    public static Matrix bestFitUsingGridSearchForGivenAT(final myProtein thisFragment,
            Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMedium,
            Vector<myDipolarCoupling.Type> typesOfRdcsToBeUsedInTensorRefinement,
            Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMediumBackCal,
            final double Syy, final double Szz, final double resolution) {

        rdcCsaTableForThisMediumBackCal.clear(); // clear off all old contents

        double [][] mat = new double[3][3];
        double alpha = 0.0, beta = 0.0, gamma = 0.0;
        double alphaBest = 0.0, betaBest = 0.0, gammaBest = 0.0;
        double bestSumOfSquaredDeviation = Double.MAX_VALUE;

        //Do the grid search ( 0 =< alpha <360, 0 =< beta <180, 0 =< gamma <360)
        for (int i = 0; i <= (180 / resolution); i++) {
            alpha = i * resolution * Math.PI / 180.0;
            for (int j = 0; j <= (180 / resolution); j++) {
                beta = j * resolution * Math.PI / 180.0;
                for (int k = 0; k <= (180 / resolution); k++) {
                    gamma = k * resolution * Math.PI / 180.0;

                    //Generate the Euler matrix
                    mat[0][0] = Math.cos(alpha) * Math.cos(beta) * Math.cos(gamma) - Math.sin(alpha) * Math.sin(gamma);
                    mat[0][1] = Math.sin(alpha) * Math.cos(beta) * Math.cos(gamma) + Math.cos(alpha) * Math.sin(gamma);
                    mat[0][2] = -Math.sin(beta) * Math.cos(gamma);
                    mat[1][0] = -Math.cos(alpha) * Math.cos(beta) * Math.sin(gamma) - Math.sin(alpha) * Math.cos(gamma);
                    mat[1][1] = -Math.sin(alpha) * Math.cos(beta) * Math.sin(gamma) + Math.cos(alpha) * Math.cos(gamma);
                    mat[1][2] = Math.sin(gamma) * Math.sin(beta);
                    mat[2][0] = Math.sin(beta) * Math.cos(alpha);
                    mat[2][1] = Math.sin(alpha) * Math.sin(beta);
                    mat[2][2] = Math.cos(beta);

                    Matrix R = new Matrix(mat);

                    double sumOfSquaredDeviation = 0.0;
                    
                    for (myDipolarCoupling.Type type : typesOfRdcsToBeUsedInTensorRefinement) {
                        Vector<myDipolarCoupling> rdcVec = rdcCsaTableForThisMedium.get(type);
                        myPair<Double, Integer> squaredRdcDev = computeRdcSumOfSquaredDeviation(thisFragment, rdcVec, R, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(type.toString()));
                        double thisRdcRatio = myConstantsForRdcs.DmaxScaled.get(type.toString());
                        //System.out.println("Dmax: " + thisRdcRatio + " for " + type.toString());
                        //sumOfSquaredDeviation += Math.sqrt(squaredRdcDev.first() / squaredRdcDev.second()) / (thisRdcRatio); // TODO: this is used on AUG 18
                        sumOfSquaredDeviation += (squaredRdcDev.first() / (thisRdcRatio * thisRdcRatio)); // TODO: Note that divider added AUG 16
                        //sumOfSquaredDeviation += squaredRdcDev.first() / thisRdcRatio; // TODO: Note that divider added AUG 16
                        //sumOfSquaredDeviation += Math.sqrt(squaredRdcDev.first() / squaredRdcDev.second()) / thisRdcRatio; // TODO: Note that divider added AUG 16
                        //System.out.print(type.toString() + " RDC RMSD: " + Math.sqrt(squaredRdcDev.first() / squaredRdcDev.second()) + "    ");
                    }
                    //System.out.println();

                    if (sumOfSquaredDeviation < bestSumOfSquaredDeviation) {
                        for (myDipolarCoupling.Type type : typesOfRdcsToBeUsedInTensorRefinement) {
                            Vector<myDipolarCoupling> rdcVec = rdcCsaTableForThisMedium.get(type);
                            myPair<Double, Integer> squaredRdcDev = computeRdcSumOfSquaredDeviation(thisFragment, rdcVec, R, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(type.toString()));
                            System.out.print(type.toString() + " RDC RMSD: " + Math.sqrt(squaredRdcDev.first() / squaredRdcDev.second()) + "    ");
                        }
                        System.out.println();
                        System.out.println("Previous Score: " + bestSumOfSquaredDeviation + "    Current Score: " + sumOfSquaredDeviation);

                        bestSumOfSquaredDeviation = sumOfSquaredDeviation;
                        alphaBest = alpha;
                        betaBest = beta;
                        gammaBest = gamma;
                        System.out.println(" Best Euler so far: " + alphaBest + "  " + betaBest + "  " + gammaBest);
                    }
                }
            }
        }

        System.out.println(" Euler " + alphaBest + "  " + betaBest + "  " + gammaBest);
        Matrix RBest = Matrix.eulerMat(alphaBest, betaBest, gammaBest);

        // Compute the RDC deviations and print the RDC rmsds
        for (myDipolarCoupling.Type type : typesOfRdcsToBeUsedInTensorRefinement) {
            System.out.println("Printing " + type.toString() + " RDC deviations");
            Vector<myDipolarCoupling> rdcBackCal = backComputeRdcs(thisFragment, type, RBest, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(type.toString()));

            Vector<myDipolarCoupling> rdcVec = rdcCsaTableForThisMedium.get(type);
            double rdcRmsd = computeRdcRmsd(thisFragment, rdcVec, RBest, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(type.toString()));
            System.out.println(type.toString() + " RDC rmsd: " + rdcRmsd);
            rdcCsaTableForThisMediumBackCal.put(type, rdcBackCal);
        }
        
        return RBest;
    }

    /**
     * Find the orientation of the strand (or SSE) so that it best-fits the RDCs
     * and NOEs. This method is used in determining the optimal orientation of
     * the first peptide plane for a short SSE.
     *
     * @param thisFragment the SSE fragment to be used for RDC fit
     * @param rdcCsaTableForThisMedium the RDC table
     * @param typesOfRdcsToBeUsedInTensorRefinement the types of RDCs to be used
     * @param rdcCsaTableForThisMediumBackCal the back-computed RDC table
     * @param Syy the diagonalized alignment tensor component
     * @param Szz the diagonalized alignment tensor component
     * @param resolution the angular resolution for grid search to estimate the best fit
     * @param hBondWeight
     * @param noeVec the NOE/hbond table
     * @param workingPartialSheet the partial beta sheet computed thus far
     * @param parallel true if the to-be-computed strand is parallel to its neighboring strand (described by prevStrandSpec); otherwise, false
     * @param prevStrandSpec the specification of the neighboring strand
     * @return a triple containing the best working partial sheet translated, the best fragment orientation, and the rotation matrix that specifies the orientation
     */
    public static myTriple<myProtein, myProtein, Matrix> bestFitUsingGridSearchForGivenATAndNoes__(myProtein thisFragment,
            Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMedium,
            Vector<myDipolarCoupling.Type> typesOfRdcsToBeUsedInTensorRefinement,
            Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMediumBackCal,
            final double Syy, final double Szz, /*final*/ double resolution,
            final double hBondWeight, Vector<myNoe> noeVec, myProtein workingPartialSheet, boolean parallel, mySseInfo prevStrandSpec) {

        rdcCsaTableForThisMediumBackCal.clear(); // clear off all old contents

        double [][] mat = new double[3][3];
        double alpha = 0.0, beta = 0.0, gamma = 0.0;
        double alphaBest = 0.0, betaBest = 0.0, gammaBest = 0.0;
        double bestScore = Double.MAX_VALUE;
        myProtein bestWorkingPartialSheetTranslated = null, bestPackedFragment = null;
        
        Vector<myProtein> vp = myNewPacker.getFourFolds(workingPartialSheet);

        // Note: we do this (the three statements below) since we don't have to resolve the four fold degeneracy
        // as the relative strand orientations are known beforehand by analyzing the NOEs
        myProtein dd = vp.elementAt(0);
        vp.clear();
        vp.add(dd);

        //int counter = 0;

        int nthBest = 0;
        int counter = 0;

boolean skipThisNow = true; // to test the svd way

if (skipThisNow) {
}
else {

        nthBestSolution:
        for (myProtein wps : vp) {
            // you can handle four-fold degeneracy by using the 6 lines below
//        for (int i = 0; i <= (180 / resolution); i++) {
//            alpha = i * resolution * Math.PI / 180.0;
//            for (int j = 0; j <= (180 / resolution); j++) {
//                beta = j * resolution * Math.PI / 180.0;
//                for (int k = 0; k <= (180 / resolution); k++) {
//                    gamma = k * resolution * Math.PI / 180.0;

            //Do the grid search ( 0 =< alpha <360, 0 =< beta <180, 0 =< gamma <360)
            for (int i = 0; i <= (360 / resolution); i++) { // 180 + 4-fold?
                alpha = i * resolution * Math.PI / 180.0;
                for (int j = 0; j <= (180 / resolution); j++) {
                    beta = j * resolution * Math.PI / 180.0;
                    for (int k = 0; k <= (360 / resolution); k++) {
                        gamma = k * resolution * Math.PI / 180.0;

                        //Generate the Euler matrix
                        mat[0][0] = Math.cos(alpha) * Math.cos(beta) * Math.cos(gamma) - Math.sin(alpha) * Math.sin(gamma);
                        mat[0][1] = Math.sin(alpha) * Math.cos(beta) * Math.cos(gamma) + Math.cos(alpha) * Math.sin(gamma);
                        mat[0][2] = -Math.sin(beta) * Math.cos(gamma);
                        mat[1][0] = -Math.cos(alpha) * Math.cos(beta) * Math.sin(gamma) - Math.sin(alpha) * Math.cos(gamma);
                        mat[1][1] = -Math.sin(alpha) * Math.cos(beta) * Math.sin(gamma) + Math.cos(alpha) * Math.cos(gamma);
                        mat[1][2] = Math.sin(gamma) * Math.sin(beta);
                        mat[2][0] = Math.sin(beta) * Math.cos(alpha);
                        mat[2][1] = Math.sin(alpha) * Math.sin(beta);
                        mat[2][2] = Math.cos(beta);

                        Matrix R = new Matrix(mat);


                        // Here we will check if we are actually searching over a spherical cap.
                        // The steps are below:
                        // (1) get axis of first strand and the second strand
                        myProtein firstStrand = wps;//new myProtein();
                        myProtein secondStrand = thisFragment;//new myProtein();

                        myVector3D firstStrandAxisVector = null;
                        myVector3D secondStrandAxisVector = null;

                        getFristStrandAxis: {
                            myPoint p1 = firstStrand.residueAt(prevStrandSpec.getBeginResidueNumber()).getAtom(myAtomLabel.__CA).getCoordinates();
                            myPoint p2 = firstStrand.residueAt(prevStrandSpec.getEndResidueNumber()).getAtom(myAtomLabel.__CA).getCoordinates();
                            firstStrandAxisVector = myVector3D.normalize(new myVector3D(p1, p2));
                        }

                        getSecondStrandAxis: {
                            myPoint p1 = secondStrand.residueAt(secondStrand.getBeginResidueNumber()).getAtom(myAtomLabel.__CA).getCoordinates();
                            myPoint p2 = secondStrand.residueAt(secondStrand.getEndResidueNumber()).getAtom(myAtomLabel.__CA).getCoordinates();
                            // Rotate the points
                            myPoint p1Rotated = new myPoint(R.times(p1.getXYZ()));
                            myPoint p2Rotated = new myPoint(R.times(p2.getXYZ()));
                            
                            secondStrandAxisVector = myVector3D.normalize(new myVector3D(p1Rotated, p2Rotated));
                        }

                        double angleBetweenAxes = myVector3D.angle(firstStrandAxisVector, secondStrandAxisVector);

                        double capAngle = myMiscConstants.sphericalCapAngle; // Math.PI / 5;

                        if (!parallel) {
                            if (!(/*angleBetweenAxes <= capAngle ||*/angleBetweenAxes >= Math.PI - capAngle)) {
                                //System.out.println("Bad angle between the axes of the strands: " + Math.toDegrees(angleBetweenAxes));
                                continue;
                            } else {
                                //System.out.println("-------I should --not-- continue");
                                //System.exit(1);
//                            if (counter % 10 == 0) {
//                                System.out.println("Counter: " + counter);
//                            }
                                counter++;
//                            boolean bb = true;
//                            if (bb) {
//                                continue;
//                            }
                            }
                        } else { // means parallel == true that is the two pair of strands being are parallel
                            if (!(angleBetweenAxes <= capAngle /*|| angleBetweenAxes >= Math.PI - capAngle*/)) {
                                //System.out.println("Bad angle between the axes of the strands: " + Math.toDegrees(angleBetweenAxes));
                                continue;
                            } else {
                                //System.out.println("-------I should --not-- continue");
                                //System.exit(1);
//                            if (counter % 10 == 0) {
//                                System.out.println("Counter: " + counter);
//                            }
                                counter++;
//                            boolean bb = true;
//                            if (bb) {
//                                continue;
//                            }
                            }
                        }

                        double sumOfSquaredDeviation = 0.0;
                        int totalNumberOfExperimentalRdcs = 0;

                        for (myDipolarCoupling.Type type : typesOfRdcsToBeUsedInTensorRefinement) {
                            Vector<myDipolarCoupling> rdcVec = rdcCsaTableForThisMedium.get(type);
                            myPair<Double, Integer> squaredRdcDev = computeRdcSumOfSquaredDeviation(thisFragment, rdcVec, R, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(type.toString()));
                            double thisRdcRatio = myConstantsForRdcs.DmaxScaled.get(type.toString());
                            sumOfSquaredDeviation += (squaredRdcDev.first() / (thisRdcRatio * thisRdcRatio));
                            totalNumberOfExperimentalRdcs += squaredRdcDev.second();
                            //sumOfSquaredDeviation += Math.sqrt(squaredRdcDev.first() / squaredRdcDev.second()) / thisRdcRatio; 
                            //System.out.print(type.toString() + " RDC RMSD: " + Math.sqrt(squaredRdcDev.first() / squaredRdcDev.second()) + "    ");
                        }

                        // Note: Important--whether to improve the scoring function using something similar to AnalyzeSolution??

                        // Compute the score from RDCs
                        // Note: Changed on AUG 18

                        double scoreTermFromRdcs = Math.sqrt(sumOfSquaredDeviation / totalNumberOfExperimentalRdcs);
                        //double scoreTermFromRdcs = sumOfSquaredDeviation;

                        if (scoreTermFromRdcs > myMiscConstants.scoreTermForRdcThreshold) {
                            continue; //Note: remove this hcode
                        } else {
                            //System.out.println("-------I should --not-- continue");
                            //System.exit(1);
                            //if (counter % 10 == 0) {
                                System.out.println("Counter: " + counter + "    scoreTermFromRdcs: " + scoreTermFromRdcs + "    angleBetweenStrandAxes: " + Math.toDegrees(angleBetweenAxes));
                            //}
                            //counter++;
                            //boolean bb = true;
                            //if (bb) {
                            //    continue;
                            //}
                        }

                        //else System.out.println("RDC Score term is: " + scoreTermFromRdcs);

                        // Rotate the SSE to the POF
                        myProtein thisFragmentRotated = new myProtein(thisFragment);
                        thisFragmentRotated.rotate(new myMatrix(R.getArrayCopy()));


                        //System.out.println("better--------"); System.exit(1);

                        
                        myNewPacker pk = new myNewPacker();
                        pk.setGlobalBestOrBestFirst(false);
                        //myTriple<myProtein, myProtein, Double> score = pk.centerFit__(/*workingPartialSheet*/wps, thisFragmentRotated, noeVec);
                        myTriple<myProtein, myProtein, Double> score = pk.__packUsingNoes__(/*workingPartialSheet*/wps, thisFragmentRotated, noeVec, null);


                        if (score.first() == null || score.second() == null) {
                            System.out.println("score---: " + score.third());
                            continue;
                        }

                        // Compute the score from NOEs plus the sheet packing

                        double scoreTermFromSheetPacking = score.third();

                        double jointScore = scoreTermFromRdcs + scoreTermFromSheetPacking; // Note: add HB weight here

                        //System.out.println("ScoreTermFromRdcs: " + scoreTermFromRdcs + "    scoreTermFromSheetPacking: " + scoreTermFromSheetPacking + "    jointScore: " + jointScore);

                        if (jointScore < bestScore) {
                            for (myDipolarCoupling.Type type : typesOfRdcsToBeUsedInTensorRefinement) {
                                Vector<myDipolarCoupling> rdcVec = rdcCsaTableForThisMedium.get(type);
                                myPair<Double, Integer> squaredRdcDev = computeRdcSumOfSquaredDeviation(thisFragment, rdcVec, R, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(type.toString()));
                                System.out.print(type.toString() + " RDC RMSD: " + Math.sqrt(squaredRdcDev.first() / squaredRdcDev.second()) + "    ");
                            }
                            System.out.println();
                            System.out.println("Previous Score: " + bestScore + "    Current Score: " + jointScore /*sumOfSquaredDeviation*/);
                            System.out.println("ScoreTermFromRdcs: " + scoreTermFromRdcs + "    scoreTermFromSheetPacking: " + scoreTermFromSheetPacking + "    jointScore: " + jointScore);

                            bestScore = jointScore/*sumOfSquaredDeviation*/;
                            bestWorkingPartialSheetTranslated = score.first();
                            bestPackedFragment = score.second();

                            System.out.println("Printing the sheet and strand: ");
                            bestWorkingPartialSheetTranslated.print();
                            bestPackedFragment.print();


                            alphaBest = alpha;
                            betaBest = beta;
                            gammaBest = gamma;
                            System.out.println("Best Euler so far: " + alphaBest + "  " + betaBest + "  " + gammaBest);

                            nthBest++;
                            System.out.println("BestRank is: " + nthBest);
                            if (nthBest == 10) break nthBestSolution; //Note: automate this in a better way
                        }
                    }
                }
            }

        }
        //System.out.println("exiting here...");
        //System.exit(1);
} // end skipThisNow

        if (bestPackedFragment == null) {
            System.out.println("Warning: The bestPackedFragment is NULL");
        } else {
            bestPackedFragment.print();
        }

/*******************************************************************************/

        System.out.println("------------------HERE**HERE**HERE----------------------------");
        // Note: added DEC 11 for ideal strands that have high RDC RMSDs
        if (bestPackedFragment == null) {

            // variables are reinitialized
            alpha = 0.0; beta = 0.0; gamma = 0.0;
            alphaBest = 0.0; betaBest = 0.0; gammaBest = 0.0;
            bestScore = Double.MAX_VALUE;
            bestWorkingPartialSheetTranslated = null; bestPackedFragment = null;


            //TEST if computing the first strand by SVD+EXACT will be a reasonably better option.;

            myBackboneModeler bbModeler =  myBackboneModeler.getInstance();

            mySseInfo thisBetaStrandInfo = new mySseInfo(thisFragment.getBeginResidueNumber(), thisFragment.getEndResidueNumber(), "E");

            thisFragment = bbModeler.computeSseGivenAlignmentTensorAndRdcs(
                    rdcCsaTableForThisMedium,
                    typesOfRdcsToBeUsedInTensorRefinement,
                    typesOfRdcsToBeUsedInTensorRefinement,
                    thisBetaStrandInfo,
                    Syy, Szz,
                    2, // refineCycle
                    5000, // dfsCycle
                    4.0, // weightInTheScoringFunction
                    2.0, // resolution
                    false);

            //thisFragment = strand2;


            nthBest = 0;

            resolution = 5.0; // DEC 15

            nthBestSolution:
            for (myProtein wps : vp) {
                // you can handle four-fold degeneracy by using the 6 lines below
//        for (int i = 0; i <= (180 / resolution); i++) {
//            alpha = i * resolution * Math.PI / 180.0;
//            for (int j = 0; j <= (180 / resolution); j++) {
//                beta = j * resolution * Math.PI / 180.0;
//                for (int k = 0; k <= (180 / resolution); k++) {
//                    gamma = k * resolution * Math.PI / 180.0;

                //Do the grid search ( 0 =< alpha <360, 0 =< beta <180, 0 =< gamma <360)
                for (int i = 0; i <= (360 / resolution); i++) { // 180 + 4-fold?
                    alpha = i * resolution * Math.PI / 180.0;
                    for (int j = 0; j <= (180 / resolution); j++) {
                        beta = j * resolution * Math.PI / 180.0;
                        for (int k = 0; k <= (360 / resolution); k++) {
                            gamma = k * resolution * Math.PI / 180.0;

                            //Generate the Euler matrix
                            mat[0][0] = Math.cos(alpha) * Math.cos(beta) * Math.cos(gamma) - Math.sin(alpha) * Math.sin(gamma);
                            mat[0][1] = Math.sin(alpha) * Math.cos(beta) * Math.cos(gamma) + Math.cos(alpha) * Math.sin(gamma);
                            mat[0][2] = -Math.sin(beta) * Math.cos(gamma);
                            mat[1][0] = -Math.cos(alpha) * Math.cos(beta) * Math.sin(gamma) - Math.sin(alpha) * Math.cos(gamma);
                            mat[1][1] = -Math.sin(alpha) * Math.cos(beta) * Math.sin(gamma) + Math.cos(alpha) * Math.cos(gamma);
                            mat[1][2] = Math.sin(gamma) * Math.sin(beta);
                            mat[2][0] = Math.sin(beta) * Math.cos(alpha);
                            mat[2][1] = Math.sin(alpha) * Math.sin(beta);
                            mat[2][2] = Math.cos(beta);

                            Matrix R = new Matrix(mat);


                            // Here we will check if we are actually searching over a spherical cap.
                            // The steps are below:
                            // (1) get axis of first strand and the second strand
                            myProtein firstStrand = wps;//new myProtein();
                            myProtein secondStrand = thisFragment;//new myProtein();

                            myVector3D firstStrandAxisVector = null;
                            myVector3D secondStrandAxisVector = null;

                            getFristStrandAxis:
                            {
                                myPoint p1 = firstStrand.residueAt(prevStrandSpec.getBeginResidueNumber()).getAtom(myAtomLabel.__CA).getCoordinates();
                                myPoint p2 = firstStrand.residueAt(prevStrandSpec.getEndResidueNumber()).getAtom(myAtomLabel.__CA).getCoordinates();
                                firstStrandAxisVector = myVector3D.normalize(new myVector3D(p1, p2));
                            }

                            getSecondStrandAxis:
                            {
                                myPoint p1 = secondStrand.residueAt(secondStrand.getBeginResidueNumber()).getAtom(myAtomLabel.__CA).getCoordinates();
                                myPoint p2 = secondStrand.residueAt(secondStrand.getEndResidueNumber()).getAtom(myAtomLabel.__CA).getCoordinates();
                                // Rotate the points
                                myPoint p1Rotated = new myPoint(R.times(p1.getXYZ()));
                                myPoint p2Rotated = new myPoint(R.times(p2.getXYZ()));

                                secondStrandAxisVector = myVector3D.normalize(new myVector3D(p1Rotated, p2Rotated));
                            }

                            double angleBetweenAxes = myVector3D.angle(firstStrandAxisVector, secondStrandAxisVector);

                            double capAngle = Math.PI / 3.0;// 3.0; //myMiscConstants.sphericalCapAngle; // Math.PI / 5;

                            if (!parallel) {
                                if (!(/*angleBetweenAxes <= capAngle ||*/angleBetweenAxes >= Math.PI - capAngle)) {
                                    //System.out.println("Bad angle between the axes of the strands: " + Math.toDegrees(angleBetweenAxes));
                                    continue;
                                } else {
                                    //System.out.println("-------I should --not-- continue");
                                    //System.exit(1);
//                            if (counter % 10 == 0) {
//                                System.out.println("Counter: " + counter);
//                            }
                                    counter++;
//                            boolean bb = true;
//                            if (bb) {
//                                continue;
//                            }
                                }
                            } else { // means parallel == true that is the two pair of strands being are parallel
                                if (!(angleBetweenAxes <= capAngle /*|| angleBetweenAxes >= Math.PI - capAngle*/)) {
                                    //System.out.println("Bad angle between the axes of the strands: " + Math.toDegrees(angleBetweenAxes));
                                    continue;
                                } else {
                                    //System.out.println("-------I should --not-- continue");
                                    //System.exit(1);
//                            if (counter % 10 == 0) {
//                                System.out.println("Counter: " + counter);
//                            }
                                    counter++;
//                            boolean bb = true;
//                            if (bb) {
//                                continue;
//                            }
                                }
                            }

                            double sumOfSquaredDeviation = 0.0;
                            int totalNumberOfExperimentalRdcs = 0;

                            for (myDipolarCoupling.Type type : typesOfRdcsToBeUsedInTensorRefinement) {
                                Vector<myDipolarCoupling> rdcVec = rdcCsaTableForThisMedium.get(type);
                                myPair<Double, Integer> squaredRdcDev = computeRdcSumOfSquaredDeviation(thisFragment, rdcVec, R, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(type.toString()));
                                double thisRdcRatio = myConstantsForRdcs.DmaxScaled.get(type.toString());
                                sumOfSquaredDeviation += (squaredRdcDev.first() / (thisRdcRatio * thisRdcRatio));
                                totalNumberOfExperimentalRdcs += squaredRdcDev.second();
                                //sumOfSquaredDeviation += Math.sqrt(squaredRdcDev.first() / squaredRdcDev.second()) / thisRdcRatio;
                                //System.out.print(type.toString() + " RDC RMSD: " + Math.sqrt(squaredRdcDev.first() / squaredRdcDev.second()) + "    ");
                            }

                            // Note: Important--whether to improve the scoring function using something similar to AnalyzeSolution??

                            // Compute the score from RDCs
                            // Note: Changed on AUG 18

                            double scoreTermFromRdcs = Math.sqrt(sumOfSquaredDeviation / totalNumberOfExperimentalRdcs);
                            //double scoreTermFromRdcs = sumOfSquaredDeviation;

                            if (scoreTermFromRdcs >/* 5 **/ myMiscConstants.scoreTermForRdcThreshold) {
                                continue; //Note: remove this hcode
                            } else {
                                //System.out.println("-------I should --not-- continue");
                                //System.exit(1);
                                //if (counter % 10 == 0) {
                                System.out.println("Counter: " + counter + "    scoreTermFromRdcs: " + scoreTermFromRdcs + "    angleBetweenStrandAxes: " + Math.toDegrees(angleBetweenAxes));
                                //}
                                //counter++;
                                //boolean bb = true;
                                //if (bb) {
                                //    continue;
                                //}
                            }

                            //else System.out.println("RDC Score term is: " + scoreTermFromRdcs);

                            // Rotate the SSE to the POF
                            myProtein thisFragmentRotated = new myProtein(thisFragment);
                            thisFragmentRotated.rotate(new myMatrix(R.getArrayCopy()));


                            //System.out.println("better--------"); System.exit(1);


                            myNewPacker pk = new myNewPacker();
                            pk.setGlobalBestOrBestFirst(false);
                            //myTriple<myProtein, myProtein, Double> score = pk.centerFit__(/*workingPartialSheet*/wps, thisFragmentRotated, noeVec);
                            myTriple<myProtein, myProtein, Double> score = pk.__packUsingNoes__(/*workingPartialSheet*/wps, thisFragmentRotated, noeVec, null);


                            if (score.first() == null || score.second() == null) {
                                System.out.println("score---: " + score.third());
                                continue;
                            }

                            // Compute the score from NOEs plus the sheet packing

                            double scoreTermFromSheetPacking = score.third();

                            double jointScore = scoreTermFromRdcs + /*2 **/ scoreTermFromSheetPacking; // Note: add HB weight here // DEC 11, multiplier 2 added to compensate for high RDC score

                            //System.out.println("ScoreTermFromRdcs: " + scoreTermFromRdcs + "    scoreTermFromSheetPacking: " + scoreTermFromSheetPacking + "    jointScore: " + jointScore);

                            if (jointScore < bestScore) {
                                for (myDipolarCoupling.Type type : typesOfRdcsToBeUsedInTensorRefinement) {
                                    Vector<myDipolarCoupling> rdcVec = rdcCsaTableForThisMedium.get(type);
                                    myPair<Double, Integer> squaredRdcDev = computeRdcSumOfSquaredDeviation(thisFragment, rdcVec, R, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(type.toString()));
                                    System.out.print(type.toString() + " RDC RMSD: " + Math.sqrt(squaredRdcDev.first() / squaredRdcDev.second()) + "    ");
                                }
                                System.out.println();
                                System.out.println("Previous Score: " + bestScore + "    Current Score: " + jointScore /*sumOfSquaredDeviation*/);
                                System.out.println("ScoreTermFromRdcs: " + scoreTermFromRdcs + "    scoreTermFromSheetPacking: " + scoreTermFromSheetPacking + "    jointScore: " + jointScore);

                                bestScore = jointScore/*sumOfSquaredDeviation*/;
                                bestWorkingPartialSheetTranslated = score.first();
                                bestPackedFragment = score.second();

                                System.out.println("Printing the sheet and strand: ");
                                bestWorkingPartialSheetTranslated.print();
                                bestPackedFragment.print();


                                alphaBest = alpha;
                                betaBest = beta;
                                gammaBest = gamma;
                                System.out.println("Best Euler so far: " + alphaBest + "  " + betaBest + "  " + gammaBest);

                                nthBest++;
                                System.out.println("BestRank is: " + nthBest);
                                if (nthBest == 30) {
                                    break nthBestSolution; //Note: automate this in a better way
                                }
                            }
                        }
                    }
                }

            }



        } // end DEC 11

/*******************************************************************************/

/*      // Note: commented on AUG 18
        // Note: AUG 03, Optimization of the first peptide plane to ensure that the RDC
        myProtein bestPackedFragmentClone = new myProtein(bestPackedFragment);

        Vector<myDipolarCoupling> nhRdcVec = rdcCsaTableForThisMedium.get(myDipolarCoupling.Type.N_HN);

        myDipolarCoupling dc = null;

        int beginResNum = bestPackedFragmentClone.getBeginResidueNumber();
        for (myDipolarCoupling dd : nhRdcVec) {
            if (dd.getResidueNumber() == beginResNum) {
                dc = dd;
            }
        }

        System.out.println("Best Fragment before first peptide plane N_HN rotated: ");

        bestPackedFragmentClone.print();
//        optimizeFirstPeptidePlane(bestPackedFragmentClone, -(Syy + Szz), Syy, Szz, dc, myConstantsForRdcs.DmaxScaled.get(myDipolarCoupling.Type.N_HN.toString()),
//                -myMiscConstants.maximumDeviationFromMean, myMiscConstants.maximumDeviationFromMean, myMiscConstants.stepSizeUsedInEstimatingFirstPeptidePlane);

        optimizeFirstPeptidePlane(bestPackedFragmentClone, rdcCsaTableForThisMedium, myDipolarCoupling.Type.N_HN, -(Syy + Szz), Syy, Szz,
                -myMiscConstants.maximumDeviationFromMean, myMiscConstants.maximumDeviationFromMean, myMiscConstants.stepSizeUsedInEstimatingFirstPeptidePlane);

        System.out.println("Best Fragment after first peptide plane N_HN rotated: ");
        bestPackedFragmentClone.print();
        System.out.println("-----------------------------------------------------------");
        inputChar.readChar();

        System.out.println("Euler(alpha, beta, gamma): " + alphaBest + "  " + betaBest + "  " + gammaBest);
        System.out.println("After bestFitUsingGridSearchForGivenATAndNoes__ the best fragments are: ");
        bestWorkingPartialSheetTranslated.print();
        bestPackedFragment.print();

        inputChar.readChar();
*/
//System.exit(1);

        Matrix RBest = Matrix.eulerMat(alphaBest, betaBest, gammaBest);

        // Compute the RDC deviations and print the RDC rmsds
        for (myDipolarCoupling.Type type : typesOfRdcsToBeUsedInTensorRefinement) {
            System.out.println("Printing " + type.toString() + " RDC deviations");
            Vector<myDipolarCoupling> rdcBackCal = backComputeRdcs(thisFragment, type, RBest, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(type.toString()));

            Vector<myDipolarCoupling> rdcVec = rdcCsaTableForThisMedium.get(type);
            double rdcRmsd = computeRdcRmsd(thisFragment, rdcVec, RBest, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(type.toString()));
            System.out.println(type.toString() + " RDC rmsd: " + rdcRmsd);
            rdcCsaTableForThisMediumBackCal.put(type, rdcBackCal);
        }

        //return RBest;
        return new myTriple<myProtein, myProtein, Matrix>(bestWorkingPartialSheetTranslated, bestPackedFragment, RBest);
    }
    
    /**
     * Optimize the first peptide plane orientation so that it satisfies the RDCs
     * and the Ramachandran angle constraints.
     *
     * @param thisFragment this fragment is assumed to be in the POF
     * @param rdcCsaTableForThisMedium the RDC table
     * @param typesOfRdcToBeUsedForFirstPeptidePlaneRefinement the RDCs to be used
     * @param Sxx the diagonalized alignment tensor component
     * @param Syy the diagonalized alignment tensor component
     * @param Szz the diagonalized alignment tensor component
     * @param angularDeviationInNegativeDirection the angular deviation in the negative direction
     * @param angularDeviationInPositiveDirection the angular deviation in the positive direction
     * @param stepSize the step size of the 1D search
     * @return the protein fragment with the first peptide plane (the phi angle) fixed to minimize the RDC deviation
     */
    public static myProtein optimizeFirstPeptidePlane(myProtein thisFragment, Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMedium,
            //final myDipolarCouplingTable thisRdcCsaTable, String mediumName,
            myDipolarCoupling.Type typesOfRdcToBeUsedForFirstPeptidePlaneRefinement, double Sxx, double Syy, double Szz,
            double angularDeviationInNegativeDirection, double angularDeviationInPositiveDirection, double stepSize) {
        
        //Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMedium = thisRdcCsaTable.getRdcMapForThisMedium(mediumName);
        Vector<myDipolarCoupling> nhRdcVec = rdcCsaTableForThisMedium.get(typesOfRdcToBeUsedForFirstPeptidePlaneRefinement);

        if (nhRdcVec == null) {
            System.out.println("Error: The type of RDC being used for first peptide plane refinement is not proper or the module is yet to be implemented");
            System.exit(1);
        }

        myDipolarCoupling dc = null;
        int beginResNum = thisFragment.getBeginResidueNumber();
        for (myDipolarCoupling dd : nhRdcVec) {
            if (dd.getResidueNumber() == beginResNum) {
                dc = dd;
                break;
            }
        }        
        double rdcRatio = myConstantsForRdcs.DmaxScaled.get(dc.getType().toString());
        
        return optimizeFirstPeptidePlane(thisFragment, Sxx, Syy, Szz, dc, rdcRatio, angularDeviationInNegativeDirection, angularDeviationInPositiveDirection, stepSize);
    }

    /**
     * Optimize the first peptide plane orientation so that it satisfies the RDC
     * and the Ramachandran angle constraints.
     * 
     * @param thisFragment this fragment is assumed to be in the POF
     * @param Sxx the diagonalized alignment tensor component
     * @param Syy the diagonalized alignment tensor component
     * @param Szz the diagonalized alignment tensor component
     * @param dc the RDC which the plane has to satisfy
     * @param rdcRatio the dipolar coupling constant
     * @param angularDeviationInNegativeDirection the angular deviation in the negative direction
     * @param angularDeviationInPositiveDirection the angular deviation in the positive direction
     * @param stepSize the step size of the 1D search
     * @return the protein fragment with the first peptide plane (the phi angle) fixed to minimize the RDC deviation
     */
    private static myProtein optimizeFirstPeptidePlane(myProtein thisFragment, double Sxx, double Syy, double Szz, myDipolarCoupling dc, double rdcRatio,
            double angularDeviationInNegativeDirection, double angularDeviationInPositiveDirection, double stepSize) {
        
        myProtein p = new myProtein(thisFragment);

        switch (dc.getType()) {
            case N_HN:
                caseForN_HNRdc:
                {
                    double[] bestVec = null;
                    double bestRdcDev = Double.MAX_VALUE;
                    double bestAlpha = Double.MAX_VALUE;

                    myResidue r = p.residueAt(p.getBeginResidueNumber());
                    myPoint N = r.getAtom(myAtomLabel.__N).getCoordinates();
                    myPoint HN = r.getAtom(myAtomLabel.__H).getCoordinates();
                    myPoint Ca = r.getAtom(myAtomLabel.__CA).getCoordinates();

                    myVector3D thetaHat = myVector3D.normalize(new myVector3D(N, Ca));
                    double[] vecPrev = myVector3D.normalize(new myVector3D(N, HN)).getXYZ();

                    bestVec = vecPrev;

                    for (double alpha = angularDeviationInNegativeDirection; alpha < angularDeviationInPositiveDirection; alpha += stepSize) {
                        Matrix rotMat = getRotationMatrixFromAxisAngle(thetaHat, alpha);
                        double[] vecNew = rotMat.times(vecPrev);
                        double x = vecNew[0];
                        double y = vecNew[1];
                        double z = vecNew[2];

                        double rdcCalOnThisVector = rdcRatio * (x * x * Sxx + y * y * Syy + z * z * Szz);

                        //System.out.println ("xxx: " + Math.sqrt(x * x + y * y + z * z) + "    rat: " + rdcRatio);

                        if (Math.abs(dc.getRdc() - rdcCalOnThisVector) < bestRdcDev) {
                            bestRdcDev = Math.abs(dc.getRdc() - rdcCalOnThisVector);
                            bestVec = vecNew;
                            bestAlpha = alpha;
                            //System.out.println("Best RDC deviation so far: " + bestRdcDev);
                        }
//                      if (iter %100 == 0) {
//                          System.out.println("iter: " + iter + "    bestAlpha: " + Math.toDegrees(bestAlpha) +
//                              "rdcCalOnThisVector: " + rdcCalOnThisVector + "    rdcExp: " + dc.getRdc() + "    rdcDev: " + rdcDev);
//                      }
                    }

                    System.out.println("Best angular deviation: " + Math.toDegrees(bestAlpha) + "    Best RDC deviation: " + bestRdcDev);

                    double lengthOfVec = (new myVector3D(N, HN)).norm();
                    myPoint newHN = new myPoint(bestVec[0] * lengthOfVec + N.getX(), bestVec[1] * lengthOfVec + N.getY(), bestVec[2] * lengthOfVec + N.getZ());
                    System.out.println("lengthOfVec: " + lengthOfVec + "    vec: " + newHN.toString());

                    HN.setX(newHN.getX());
                    HN.setY(newHN.getY());
                    HN.setZ(newHN.getZ());
                }
                break;
            default:
                System.out.println("Error: Optimization of first peptideplane using " + dc.getType().toString() + " dipolar coupling is yet to be implemented");
                System.exit(1);
        }
        return p;
    }

    /**
     * Return the rotation matrix from axis angle form.
     *
     * @param axis the specification of the axis (a vector along the axis)
     * @param angle the angle of rotation along the axis in radians
     * @return the rotation matrix
     */
    public static Matrix getRotationMatrixFromAxisAngle(myVector3D axis, double angle) {
        axis = myVector3D.normalize(axis);
        AxisAngle4d aa = new AxisAngle4d(axis.getX(), axis.getY(), axis.getZ(), angle);
        Quat4d q = new Quat4d();
        q.set(aa);
        Matrix3d m = new Matrix3d();
        m.set(q);
        Matrix mNew = new Matrix(new double[][]{{m.m00, m.m01, m.m02}, {m.m10, m.m11, m.m12}, {m.m20, m.m21, m.m22}});
//        if (__verbose) {
//            System.out.println("Printing the rotation matrix");
//            mNew.print(4, 3);
//        }
        return mNew;
    }

    /**
     * Check for SVD success.
     * @param X first matrix
     * @param Y second matrix
     */
    private static void check(Matrix X, Matrix Y) {
        double eps = Math.pow(2.0, -52.0);
        if (X.norm1() == 0. & Y.norm1() < 10 * eps) {
            return;
        }
        if (Y.norm1() == 0. & X.norm1() < 10 * eps) {
            return;
        }
        if (X.minus(Y).norm1() > 1000 * eps * Math.max(X.norm1(), Y.norm1())) {
            throw new RuntimeException("The norm of (X-Y) is too large: " + Double.toString(X.minus(Y).norm1()));
        }
    }


//    myVector3D getSseAxis(myProtein p, int lowBoundaryResidueNumber, int highBoundaryResidueNumber) {
//        myPoint p1 = p.residueAt(lowBoundaryResidueNumber).getAtom(myAtomLabel.__CA).getCoordinates();
//        myPoint p2 = p.residueAt(highBoundaryResidueNumber).getAtom(myAtomLabel.__CA).getCoordinates();
//
//        return myVector3D.normalize(new myVector3D(p1, p2));
//    }

    /**
     * Testing some of the methods of this class
     * @param str the argument string
     */
    public static void main(String... str) {
        myPoint p1 = new myPoint(1.0, 2.0, 3.0);
        myPoint p2 = new myPoint(3.0, 2.0, 1.0);

        myVector3D axis = new myVector3D(p1, p2);

        axis = myVector3D.normalize(axis);

        double angle = 0.5;

        Matrix m = getRotationMatrixFromAxisAngle(axis, angle);

        Vector<myProtein> vp = new Vector<myProtein>();
        myPdbParser pParser = new myPdbParser("mm11.pdb");
        while (pParser.hasNextProtein()) {
            vp.add(pParser.nextProtein());
        }
        myProtein p = vp.elementAt(0);

        p.print();

        double Syy = 15.457906239256296;  double Szz = 24.715711838306476;

        optimizeFirstPeptidePlane(p, -(Syy + Szz), Syy, Szz, new myDipolarCoupling("N", 2, "H", 2, -8.17, 0.0), 0.4991227672784315, Math.toRadians(-40), Math.toRadians(40), 0.0001);

        p.print();

    }
}
