/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2012 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, August 04, 2012
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.1, August 04, 2012
 * @author        Chittaranjan Tripathy (2007-2012)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */
import java.io.*;
import java.util.Scanner;
import java.util.Vector;
import java.util.Formatter;
import java.util.Locale;
/**
 * Description of the class
 */
public class myDihedralRestraint extends myPhiPsi {
    //protected double __phi;
    //protected double __psi;
    //protected int __residue_number;  // number of first residue, though Phi and Psi are between two residues
    //protected String __residue_name; //We count it as belonging to the first of the two PPs
    
    //protected double __phi_lb;
    //protected double __phi_ub;
    //protected double __psi_lb;
    //protected double __psi_ub;
    
    protected double __sigma_phi;
    protected double __sigma_psi;    
    protected double __RCI_order_parameter_S2;
    protected String __class;
    
    // REMARK  CLASS is the classification of the prediction result:
    // REMARK    None: no torsion prediction was made.
    // REMARK    Good: majority consensus in database matches;
    // REMARK          prediction is likely to be good.
    // REMARK    Warn: no consensus in database matches, do not use prediction.
    // REMARK    Dyn:  RCI S2 value indicates that residue has dynamic conformation.
    
    private double __multiplier_sigma_phi = 1.0; // multipliers for standard deviations: m * sigma, note m = 3 => 99% coverage
    private double __multiplier_sigma_psi = 1.0; // multipliers for standard deviations: m * sigma, note m = 3 => 99% coverage

    public myDihedralRestraint() {
    }

    public myDihedralRestraint(int resNum, String resName, double phi, double psi, double sigmaPhi, double sigmaPsi, String classification) {
        super(resNum, resName, phi, psi);
        //__residue_number = resNum;
        //__residue_name = resName;
        //__phi = phi;
        //__psi = psi;
        __sigma_phi = sigmaPhi;
        __sigma_psi = sigmaPsi;
        __class = classification;        
    }

    public myDihedralRestraint(int resNum, String resName, double phi, double psi, double sigmaPhi, double sigmaPsi, double RCIOrderParameterS2, String classification) {
        super(resNum, resName, phi, psi);
        //__residue_number = resNum;
        //__residue_name = resName;
        //__phi = phi;
        //__psi = psi;
        __sigma_phi = sigmaPhi;
        __sigma_psi = sigmaPsi;
        __class = classification;  
        __RCI_order_parameter_S2 = RCIOrderParameterS2;
    }    
                                  
    public double getPhiLowerBound() {
        return this.getPhi() - /*__multiplier_sigma_phi * __sigma_phi*/ Math.max(Math.PI/6, __multiplier_sigma_phi * __sigma_phi); // PI/6 = 30 deg
    }
    
    public double getPhiUpperBound() {
        return this.getPhi() + /*__multiplier_sigma_phi * __sigma_phi*/ Math.max(Math.PI/6, __multiplier_sigma_phi * __sigma_phi);
    }
    
    public double getPsiLowerBound() {
        return this.getPsi() - /*__multiplier_sigma_psi * __sigma_psi*/ Math.max(Math.PI/6, __multiplier_sigma_psi * __sigma_psi);
    }
    
    public double getPsiUpperBound() {
        return this.getPsi() + /*__multiplier_sigma_psi * __sigma_psi*/ Math.max(Math.PI/6, __multiplier_sigma_psi * __sigma_psi);
    }    
    
    public double getPhiStandardDeviation() {
        return __sigma_phi;
    }

    public double getPsiStandardDeviation() {
        return __sigma_psi;
    }
    
    public double getS2() {
        return __RCI_order_parameter_S2;
    }
    
    public String getClassification() {
        return __class;
    }
        
    public boolean isNone() {
        return getClassification().equalsIgnoreCase("None");
    }
    
    public boolean isGood() {
        return getClassification().equalsIgnoreCase("Good");
    }
    
    public boolean isWarn() {
        return getClassification().equalsIgnoreCase("Warn");
    }
    
    public boolean isDyn() {
        return getClassification().equalsIgnoreCase("Dyn");
    }
    
    @Override public String toString() {
        return super.toString() +  "  " + String.valueOf(Math.toDegrees(getPhiStandardDeviation())) + "  " + String.valueOf(Math.toDegrees(getPsiStandardDeviation())) 
                + String.valueOf(getS2()) + "  " + getClassification();
    }   
    
    /**
     * Read the dihedral angle restraint file.
     * 
     * @param talosFileName the file containing the dihedral angle restraint (TALOS) information
     * @return a vector of dihedral angle restraints
     */
    public static Vector<myDihedralRestraint> parseTalosRestraintFile(String talosFileName) {
        if (!((new File(talosFileName)).exists())) {
            return null;
        }
        
        Vector<myDihedralRestraint> dihedralRestrVec = new Vector<myDihedralRestraint>();

        try {
            Scanner scanner = new Scanner(new File(talosFileName));
            scanner.useDelimiter(System.getProperty("line.separator"));
            while (scanner.hasNextLine()) {
                String line = myMiscUtilities.stripLineComment(scanner.nextLine()).trim();
                line = line.replaceAll("\\(|\\)", " ").trim();
                //System.out.println(line);
                if (line.length() != 0) {
                    String[] words = line.split("\\s+");
                    // The format expected is :
                    // VARS   RESID RESNAME PHI PSI DPHI DPSI DIST S2 COUNT CS_COUNT CLASS 
                    // FORMAT %4d %s %8.3f %8.3f %8.3f %8.3f %8.3f %5.3f %2d %2d %s

                    if (words.length != 11) {
                        System.out.println("Error: Incorrect TALOS+ format. The correct format is \nVARS   RESID RESNAME PHI PSI DPHI DPSI DIST S2 COUNT CS_COUNT CLASS\nFORMAT %4d %s %8.3f %8.3f %8.3f %8.3f %8.3f %5.3f %2d %2d %s");
                        System.exit(1);
                    }
                    if (words.length >= 11 && words[0].trim().matches("[0-9]+")) {
                        int residueNumber = Integer.parseInt(words[0]);
                        String residueName = words[1].trim().toUpperCase();
                        if (residueName.length() == 1) {
                            residueName = myAminoAcidResidueMap.getThreeLetterCode(residueName.charAt(0));
                        }

                        double phi = Double.parseDouble(words[2]);
                        double psi = Double.parseDouble(words[3]);
                        double phiDev = Double.parseDouble(words[4]);
                        double psiDev = Double.parseDouble(words[5]);                        
                        double RCIOrderParameterS2 = Double.parseDouble(words[7]);

                        String classification = words[10];

                        myDihedralRestraint thisRestraint = new myDihedralRestraint(residueNumber, residueName,
                                Math.toRadians(phi), Math.toRadians(psi), Math.toRadians(phiDev), Math.toRadians(psiDev),
                                RCIOrderParameterS2, classification);
                        dihedralRestrVec.add(thisRestraint);                       
                    }
                }
            }
            scanner.close();
        } catch (FileNotFoundException e) {
            System.out.println("Error: Input file " + talosFileName + " not found");
            e.printStackTrace();
        } 

        if (dihedralRestrVec.isEmpty()) {
            return null;
        } else {
            return dihedralRestrVec;
        }
    }
    
    public static void printTalosRestraintVector(Vector<myDihedralRestraint> vdr) {
        if (vdr == null) {
            return;
        }        
        System.out.println("Format %4d %s %8.3f %8.3f %8.3f %8.3f %5.3f %s\nRes# Res      Phi      Psi   PhiDev   PsiDev    S2 Class");
        for (myDihedralRestraint dr : vdr) {
            System.out.println(dr.getTalosRestraintRow());
        }
    }
    
    public String getTalosRestraintRow() {
        // VARS   RESID RESNAME PHI PSI DPHI DPSI DIST S2 COUNT CS_COUNT CLASS 
        // FORMAT %4d %s %8.3f %8.3f %8.3f %8.3f %8.3f %5.3f %2d %2d %s
        // The fields DIST COUNT CS_COUNT are not used here.
        StringBuilder sb = new StringBuilder(80);
        Formatter formatter = new Formatter(sb, Locale.US);

        formatter.format("%4d", this.getResidueNumber());        
        formatter.format(" %s", this.getResidueName());
        formatter.format(" %8.3f", Math.toDegrees(this.getPhi()));
        formatter.format(" %8.3f", Math.toDegrees(this.getPsi()));
        formatter.format(" %8.3f", Math.toDegrees(this.getPhiStandardDeviation()));
        formatter.format(" %8.3f", Math.toDegrees(this.getPsiStandardDeviation()));
        formatter.format(" %5.3f", this.getS2());
        formatter.format(" %s", this.getClassification());     

        return sb.toString();
    }
    
    
    public static Vector<myDihedralRestraint> parseTalosRestraintFileAndGetRestraintsForGivenRange(String talosFileName, int beginResidueNumber, int endResidueNumber) {
        Vector<myDihedralRestraint> v = parseTalosRestraintFile(talosFileName);
        if (v == null) {
            return null;
        }
        
        //System.out.println("size of TALOS restraints: " + v.size());
        Vector<myDihedralRestraint> vRet = new Vector<myDihedralRestraint>();
        for (int i = beginResidueNumber; i <= endResidueNumber; i++) {
            for (myDihedralRestraint dr : v) {
                if (dr.getResidueNumber() == i) {
                    vRet.add(dr);
                }
            }
        }
        
        if (vRet.isEmpty()) {
            return null;
        } else {
            return vRet;
        }        
    }
    
    public static myDihedralRestraint getDihedralRestraint(Vector<myDihedralRestraint> v, int residueNumber) {
        if (v == null) {
            return null;
        }
        for (myDihedralRestraint dr : v) {
            if (dr.getResidueNumber() == residueNumber) {
                return dr;
            }
        }
        return null;
    }
    
}
