/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2009 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, 01 December, 2009
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.0, Nov 18, 2009
 * @author        Chittaranjan Tripathy (2007-2009)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */
import java.io.*;
import java.util.*;

/**
 * Description of the class
 */
public class myInputDataAndParameterManager implements java.io.Serializable {

    private String __directory_architecture_file = null;

    private String __rdc_file_medium1 = null;
    private String __rdc_file_medium2 = null;
    private String __noe_file = null;
    private String __seq_file = null;

    //private String __talos_dihedral_constraints_file = null;
    private String __sse_type_and_boundary_spec_file = null;
    private String __program_parameter_file = null;

    /**
     * Construct an input parameter manager object from the input directory
     * architecture information which contains the information about the
     * input data files, the parameter files, and the program configuration files.
     *
     * @param directoryArchitecture the file that contains the input directory structure
     */
    public myInputDataAndParameterManager(String directoryArchitectureFile) {
        __directory_architecture_file = directoryArchitectureFile;
        
        String inputDataDirectory = System.getProperty("user.dir") + System.getProperty("file.separator") +
                getInputDirectory(__directory_architecture_file) + System.getProperty("file.separator") +
                getInputDataDirectory(__directory_architecture_file) + System.getProperty("file.separator");

        String inputParameterDirectory = System.getProperty("user.dir") + System.getProperty("file.separator") +
                getInputDirectory(__directory_architecture_file) + System.getProperty("file.separator") +
                getInputParameterDirectory(__directory_architecture_file) + System.getProperty("file.separator");

        __rdc_file_medium1 = inputDataDirectory + getRdcFileForMedium1(__directory_architecture_file);
        if (!getRdcFileForMedium2(__directory_architecture_file).equalsIgnoreCase("null")) {
            __rdc_file_medium2 = inputDataDirectory + getRdcFileForMedium2(__directory_architecture_file);
        }

        __noe_file = inputDataDirectory + "noe.txt";

        __seq_file = inputDataDirectory + "seq.txt";

        __sse_type_and_boundary_spec_file = inputParameterDirectory + getSSeInfoFile(__directory_architecture_file);
        __program_parameter_file = inputParameterDirectory + getProgramParameterFile(__directory_architecture_file);
    }

    /**
     * Return the name of the input directory.
     *
     * @return the name of the input directory
     */
    public static String getInputDirectory(String directoryArchitectureFile) {
        return inputDirectoryHelper(directoryArchitectureFile, "inputDirectory");
    }

    /**
     * Return the name of the input data directory.
     *
     * @return the name of the input data directory
     */
    public static String getInputDataDirectory(String directoryArchitectureFile) {
        return inputDirectoryHelper(directoryArchitectureFile, "inputDataDirectory");
    }

    /**
     * Return the name of the input parameter directory.
     *
     * @return the name of the input parameter directory
     */
    public static String getInputParameterDirectory(String directoryArchitectureFile) {
        return inputDirectoryHelper(directoryArchitectureFile, "inputParameterDirectory");
    }

    /**
     * Return the name of the program parameter file.
     *
     * @return the name of the program parameter file
     */
    public static String getProgramParameterFile(String directoryArchitectureFile) {
        return inputDirectoryHelper(directoryArchitectureFile, "programParameterFile");
    }

    /**
     * Return the name of the secondary structure specification file.
     * 
     * @return the name of the secondary structure specification file
     */
    public static String getSSeInfoFile(String directoryArchitectureFile) {
        //String directoryArchitectureFile = "dirArch.txt";
        return inputDirectoryHelper(directoryArchitectureFile, "SSeInfoFile");
    }

    /**
     * Return the dipolar coupling data file name for medium 1.
     *
     * @return the dipolar coupling data file name for medium 1
     */
    public static String getRdcFileForMedium1(String directoryArchitectureFile) {
        //String directoryArchitectureFile = "dirArch.txt";
        return inputDirectoryHelper(directoryArchitectureFile, "RDC and CSA Files in Medium1");
    }

    /**
     * Return the dipolar coupling data file name for medium 2.
     *
     * @return the dipolar coupling data file name for medium 2
     */
    public static String getRdcFileForMedium2(String directoryArchitectureFile) {
        //String directoryArchitectureFile = "dirArch.txt";
        return inputDirectoryHelper(directoryArchitectureFile, "RDC and CSA Files in Medium2");
    }

    /**
     * Return the NOE data file.
     *
     * @return the NOE data file
     */
    public String getNoeFile() {
        //String directoryArchitectureFile = "dirArch.txt";
        return __noe_file;
    }

    /**
     * Return the sequence data file.
     *
     * @return the sequence data file
     */
    public String getSeqFile() {
        //String directoryArchitectureFile = "dirArch.txt";
        return __seq_file;
    }

    /**
     * Read the file whose name is supplied as the first argument, and return the
     * string that corresponds to the tag supplied as the second argument.
     *
     * @param directoryArchitectureFile the file depicting the input directory structure
     * @param description the tag for a particular type of file/directory
     * @return the string that corresponds to the tag
     */
    private static String inputDirectoryHelper(String directoryArchitectureFile, String description) {
        String directoryOrFileName = null;

        try {
            Scanner scanner = new Scanner(new File(directoryArchitectureFile));
            scanner.useDelimiter(System.getProperty("line.separator"));
            while (scanner.hasNextLine()) {
                String line = scanner.nextLine().trim();
                if (line.length() != 0) {
                    String[] words = line.split(":");

                    if (words[0].equalsIgnoreCase(description) && words.length == 2) {
                        directoryOrFileName = words[1].trim();
                    }
                }
            }
            scanner.close();
        } catch (FileNotFoundException e) {
            System.out.println("Error: Input file " + directoryArchitectureFile + " not found");
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }

        if (directoryOrFileName == null) {
            System.out.println("Error: In the file " + directoryArchitectureFile + " the " + description + " is not found");
            System.exit(1);
        }

        return directoryOrFileName;
    }

    /**
     * Return the dipolar coupling table constructed after reading the RDC files.
     *
     * @return the dipolar coupling table
     */
    public myDipolarCouplingTable getDipolarCouplingTable() {
        if (__rdc_file_medium2 == null) {
            return new myDipolarCouplingTable(__rdc_file_medium1);
        }
        return new myDipolarCouplingTable(__rdc_file_medium1, __rdc_file_medium2);
    }

    public Map<String, Map<String, String>> parseParameterFile() {
        return parseParameterFile(__program_parameter_file);
    }

    /**
     * Parse the parameter file and extract the parameters for alignment tensor,
     * helix and sheet computation.
     *
     * @param parameterFileName the parameter file name
     * @return a table of parameters extracted from the parameter file
     */
    private Map<String, Map<String, String>> parseParameterFile(String parameterFileName) {
        Map<String, Map<String, String>> params = new TreeMap<String, Map<String, String>>();
        try {
            Scanner scanner = new Scanner(new File(parameterFileName));
            scanner.useDelimiter(System.getProperty("line.separator"));

            Vector<String> block = new Vector<String>();

            while (scanner.hasNextLine()) {
                String line = myMiscUtilities.stripLineComment(scanner.nextLine()).trim();
                if (line.startsWith("@parametersForAlignmentTensorComputation")) {
                    block.clear();
                    block.add(line);
                    while (scanner.hasNextLine()) {
                        line = myMiscUtilities.stripLineComment(scanner.nextLine()).trim();
                        block.add(line);
                        if (line.equalsIgnoreCase("}")) {
                            break;
                        }
                    }
                    Map<String, String> parametersForAlignmentTensorComputation = new TreeMap<String, String>();
                    for (int i = 1; i < block.size() - 1; i++) {
                        String currLine = block.elementAt(i);
                        int index = currLine.indexOf("=");
                        String key = currLine.substring(0, index).trim();
                        String value = currLine.substring(index + 1, currLine.length()).trim();
                        parametersForAlignmentTensorComputation.put(key, value);
                    }
                    params.put("parametersForAlignmentTensorComputation", parametersForAlignmentTensorComputation);
                    block.clear();
                } else if (line.startsWith("@parametersForHelixComputation")) {
                    block.clear();
                    block.add(line);
                    while (scanner.hasNextLine()) {
                        line = myMiscUtilities.stripLineComment(scanner.nextLine()).trim();
                        block.add(line);
                        if (line.equalsIgnoreCase("}")) {
                            break;
                        }
                    }
                    Map<String, String> parametersForHelixComputation = new TreeMap<String, String>();
                    for (int i = 1; i < block.size() - 1; i++) {
                        String currLine = block.elementAt(i);
                        int index = currLine.indexOf("=");
                        String key = currLine.substring(0, index).trim();
                        String value = currLine.substring(index + 1, currLine.length()).trim();
                        parametersForHelixComputation.put(key, value);
                    }
                    params.put("parametersForHelixComputation", parametersForHelixComputation);
                    block.clear();
                } else if (line.startsWith("@parametersForStrandComputation")) {
                    block.clear();
                    block.add(line);
                    while (scanner.hasNextLine()) {
                        line = myMiscUtilities.stripLineComment(scanner.nextLine()).trim();
                        block.add(line);
                        if (line.equalsIgnoreCase("}")) {
                            break;
                        }
                    }
                    Map<String, String> parametersForStrandComputation = new TreeMap<String, String>();
                    for (int i = 1; i < block.size() - 1; i++) {
                        String currLine = block.elementAt(i);
                        int index = currLine.indexOf("=");
                        String key = currLine.substring(0, index).trim();
                        String value = currLine.substring(index + 1, currLine.length()).trim();
                        parametersForStrandComputation.put(key, value);
                    }
                    params.put("parametersForStrandComputation", parametersForStrandComputation);
                    block.clear();
                }
            }
            scanner.close();
        } catch (FileNotFoundException e) {
            System.out.println("Error: Input file " + parameterFileName + " not found");
            e.printStackTrace();
        } catch (IOException e) {
            System.out.println("IOException occured. The stack trace is: ");
            e.printStackTrace();
        }

        return (params.size() == 0) ? null : params;
    }

    /**
     * This method parses the secondary structure element type and boundary
     * specifications and returns a vector of secondary structure specifications.
     *
     * @return a vector of secondary structure specifications
     */
    public Vector<mySseInfo> parseSseIdFile() {
        mySseInfo sseIds = new mySseInfo();
        Vector<mySseInfo> idVec = sseIds.parseSseIds(__sse_type_and_boundary_spec_file);
        return idVec;
    }

    /**
     * This method parses the secondary structure element type and boundary
     * specifications and returns the beta sheet specifications.
     *
     * @return beta sheet specifications
     */
    public Vector<Vector<myTriple<mySseInfo, mySseInfo, Boolean>>> parseSseIdFileAndReturnSheetSpecifications() {
        mySseInfo sseIds = new mySseInfo();
        return sseIds.parseSseIdFileAndReturnSheetSpecifications(__sse_type_and_boundary_spec_file);
    }

    /**
     * This method parses the secondary structure element type and boundary
     * specifications and returns the SSE specification to be used for the
     * alignment tensor computation.
     *
     * @return the SSE specification to be used for the alignment tensor computation
     */
    public mySseInfo parseSseIdFileAndReturnSseForAlignmentTensorComputation() {
        mySseInfo sseIds = new mySseInfo();
        return sseIds.parseSseIdFileAndReturnSseForAlignmentTensorComputation(__sse_type_and_boundary_spec_file);
    }
}

