/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2009 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, 01 December, 2009
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.0, Nov 18, 2009
 * @author        Chittaranjan Tripathy (2007-2009)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */

/**
 * Description of the class
 */
public class myPeptidePlane {
    /**
     * For this program we implicitly mean the following: p1 = N, p2 = H, and p3 = Ca
     *
     */
    private myPoint p1; // N
    private myPoint p2; // H
    private myPoint p3; // CA

    /**
     * Construct a peptide plane from three points.
     *
     * @param point1 the first point
     * @param point2 the second point
     * @param point3 the third point
     */
    public myPeptidePlane(final myPoint point1, final myPoint point2, final myPoint point3) {
        p1 = new myPoint(point1);
        p2 = new myPoint(point2);
        p3 = new myPoint(point3);
    }

    /**
     * A copy constructor that constructs a plane from another plane.
     *
     * @param pp the plane which is to be cloned
     */
    public myPeptidePlane(final myPeptidePlane pp) {
        p1 = new myPoint(pp.getN());
        p2 = new myPoint(pp.getH());
        p3 = new myPoint(pp.getCa());
    }

    /**
     * Return the coordinates of N.
     *
     * @return return the coordinates of N
     */
    public myPoint getN() {
        return new myPoint(p1);
    }

    /**
     * Return the coordinates of H.
     *
     * @return return the coordinates of H
     */
    public myPoint getH() {
        return new myPoint(p2);
    }

    /**
     * Return the coordinates of CA.
     *
     * @return return the coordinates of CA
     */
    public myPoint getCa() {
        return new myPoint(p3);
    }

    /**
     * Set the coordinates of N.
     *
     * @param p the coordinates of N
     */
    public void setN(myPoint p) {
        p1 = new myPoint(p);
    }

    /**
     * Set the coordinates of H.
     *
     * @param p the coordinates of H
     */
    public void setH(myPoint p) {
        p2 = new myPoint(p);        
    }

    /**
     * Set the coordinates of CA.
     *
     * @param p the coordinates of CA
     */
    public void setCa(myPoint p) {
        p3 = new myPoint(p);
    }

    /**
     * Print the peptide plane.
     */
    void print() {
        System.out.println("N: " + p1.toString() + "  H: " + p2.toString() + "  Ca: " + p3.toString());
    }

    /**
     * Returns the angle between two vectors in the range [0, pi].
     *
     * @param v1 the first vector
     * @param v2 the second vector
     * @return the angle between two vectors in the range [0, pi]
     */
    public static double angleBetweenTwoVectors(myVector3D v1, myVector3D v2) {
        double c = myVector3D.cosineOfAngle(v1, v2);
        if (c > 0 && Math.abs(c - 1.00) < Const.eps) {
            return 0.0;
        }
        if (c < 0 && Math.abs(c + 1.00) < Const.eps) {
            return Math.PI;
        }
        return Math.acos(c);  // return acos [0, PI] so should be OK
    }

    /**
     * This method computes the rotation matrix for a frame defined on the
     * peptide plane with respect to the (global) reference frame (can be the
     * principal order frame). The frame on the peptide plane is defined as follows:
     * +Z-axis is along the bond HN->N, i.e., from HN to N,
     * +Y-axis is on the peptide plane and the angle between +Y and N->CA is less than 90 degrees,
     * +X-axis is defined using right hand rule.
     *
     * @param pPlane the peptide plane the rotation of which is t be determined
     * @return (one of the) the rotation matrix that represents the rotation of
     * the peptide plane wrt. the (global) reference frame
     */
    public static Matrix RotationWrtGlobalFrame(final myPeptidePlane pPlane) {
        myVector3D N2HVec = new myVector3D(pPlane.getN(), pPlane.getH());
        myVector3D N2CaVec = new myVector3D(pPlane.getN(), pPlane.getCa());
        return myPeptidePlane.RotationWrtGlobalFrame(N2HVec, N2CaVec);
    }

    /**
     * This method computes the rotation matrix for a frame defined on the
     * peptide plane with respect to the (global) reference frame (can be the
     * principal order frame). The frame on the peptide plane is defined as follows:
     * +Z-axis is along the bond HN->N, i.e., from HN to N,
     * +Y-axis is on the peptide plane and the angle between +Y and N->CA is less than 90 degrees,
     * +X-axis is defined using right hand rule.
     *
     * @param pPlane the peptide plane the rotation of which is t be determined
     * @param rightHand if true the right-handedness (chirality) is preserved;
     * otherwise, left-handedness is preserved
     * @return (one of the) the rotation matrix that represents the rotation of
     * the peptide plane wrt. the (global) reference frame
     */
    public static Matrix RotationWrtGlobalFrame(final myPeptidePlane pPlane, boolean rightHand) {
        myVector3D N2HVec = new myVector3D(pPlane.getN(), pPlane.getH());
        myVector3D N2CaVec = new myVector3D(pPlane.getN(), pPlane.getCa());        
        return myPeptidePlane.RotationWrtGlobalFrame(N2HVec, N2CaVec, rightHand);
    }

    /**
     * This method computes the rotation matrix for a frame defined on the
     * peptide plane with respect to the (global) reference frame (can be the
     * principal order frame). The frame on the peptide plane is defined as follows:
     * +Z-axis is along the bond HN->N, i.e., from HN to N,
     * +Y-axis is on the peptide plane and the angle between +Y and N->CA is less than 90 degrees,
     * +X-axis is defined using right hand rule.
     *
     * @param nToh the N->HN vector which with the N->CA vector constitutes the peptide plane
     * @param nToca the N->CA vector which with the N->HN vector constitutes the peptide plane
     * @return (one of the) the rotation matrix that represents the rotation of
     * the peptide plane wrt. the (global) reference frame
     */
    public static Matrix RotationWrtGlobalFrame(final myVector3D nToh, final myVector3D nToca) {
        double[] n2h = myVector3D.normalize(nToh).getXYZ(); // get the direction cosines
        double[] n2ca = myVector3D.normalize(nToca).getXYZ(); // get the direction cosines

        double theta = Math.PI - angleBetweenTwoVectors(nToh, nToca);

        double xNH = n2h[0], yNH = n2h[1], zNH = n2h[2];
        double xNCA = n2ca[0], yNCA = n2ca[1], zNCA = n2ca[2];

        double[][] mat = new double[3][3];

        double sinTheta = 9.99;
        double alpha = 9.99, beta = 9.99, gamma = 9.99;

        if (Math.abs(zNH - 1.0) < Const.eps) { //when the NH vector is in +Z direction
            if (Math.abs(zNCA - 1.0) > Const.eps && Math.abs(zNCA + 1.0) > Const.eps) { // i.e., (!(zNCA == 1.0) && !(zNCA == -1.0))
                sinTheta = Math.sqrt(1.0 - zNCA * zNCA); //under such condition cosTheta = zNCA

                mat[0][0] = yNCA / sinTheta;
                mat[0][1] = xNCA / sinTheta;
                mat[0][2] = 0.0;
                mat[1][0] = -xNCA / sinTheta;
                mat[1][1] = yNCA / sinTheta;
                mat[1][2] = 0.0;
                mat[2][0] = 0.0;
                mat[2][1] = 0.0;
                mat[2][2] = 1.0;
                return (new Matrix(mat)).transpose(); //return directly the transposed matrix

            } else { //impossible for normal PDB file
                System.out.println("zNCA is 1 or -1");
                System.exit(0);
            }
        } else if (Math.abs(zNH + 1.0) < Const.eps) { //when the NH vector is in -Z direction

            if (Math.abs(zNCA - 1.0) > Const.eps && Math.abs(zNCA + 1.0) > Const.eps) {
                sinTheta = Math.sqrt(1.0 - zNCA * zNCA); //under such condition cosTheta = zNCA

                mat[0][0] = yNCA / sinTheta;
                mat[0][1] = xNCA / sinTheta;
                mat[0][2] = 0.0;
                mat[1][0] = xNCA / sinTheta;
                mat[1][1] = yNCA / sinTheta;
                mat[1][2] = 0.0;
                mat[2][0] = 0.0;
                mat[2][1] = 0.0;
                mat[2][2] = 1.0;
                return (new Matrix(mat)).transpose();
            } else { //impossible for normal PDB file
                System.out.println("zNCA is 1 or -1");
                System.exit(0);
            }
        } else {
            beta = Math.acos(zNH);
            double sinAlpha = yNH / Math.sin(beta);
            double cosAlpha = xNH / Math.sin(beta);
            if (cosAlpha >= 0) {
                alpha = Math.asin(sinAlpha);
            } else {
                alpha = Math.PI - Math.asin(sinAlpha);
            }
            double sinGamma = (zNCA + Math.cos(beta) * Math.cos(theta)) / (Math.sin(theta) * Math.sin(beta));
            double cosGamma = (yNCA * Math.cos(alpha) - xNCA * Math.sin(alpha)) / Math.sin(theta);
            if (cosGamma >= 0) {
                gamma = Math.asin(sinGamma);
            } else {
                gamma = Math.PI - Math.asin(sinGamma);
            }
        }

        // Construct an Euler rotation matrix from three Euler angles
        Matrix mm = Matrix.eulerMat(alpha, beta, gamma);
        //invert the +z axis: point along the NH->N direction for the sake of easy math        
        Matrix MM = Matrix.rotationMat(Math.PI, "+y");
        return MM.times(mm);
    }

    /**
     * This method computes the rotation matrix for a frame defined on the
     * peptide plane with respect to the (global) reference frame (can be the
     * principal order frame). The frame on the peptide plane is defined as follows:
     * +Z-axis is along the bond HN->N, i.e., from HN to N,
     * +Y-axis is on the peptide plane and the angle between +Y and N->CA is less than 90 degrees,
     * +X-axis is defined using right hand rule.
     *
     * @param nToh the N->HN vector which with the N->CA vector constitutes the peptide plane
     * @param nToca the N->CA vector which with the N->HN vector constitutes the peptide plane
     * @param rightHand if true the right-handedness (chirality) is preserved;
     * otherwise, left-handedness is preserved
     * @return (one of the) the rotation matrix that represents the rotation of
     * the peptide plane wrt. the (global) reference frame
     */
    public static Matrix RotationWrtGlobalFrame(final myVector3D nToh, final myVector3D nToca, boolean rightHand) {
        return rightHand ? RotationWrtGlobalFrame(nToh, nToca) : (Const.mLeftHand).times(RotationWrtGlobalFrame(nToh, nToca));
    }

    /**
     * This method computes the rotation matrix for a frame defined on the
     * peptide plane with respect to the (global) reference frame (can be the
     * principal order frame). The frame on the peptide plane is defined as follows:
     * +Z-axis is along the bond HN->N, i.e., from HN to N,
     * +Y-axis is on the peptide plane and the angle between +Y and N->CA is less than 90 degrees,
     * +X-axis is defined using right hand rule.
     *
     * @param nToh the N->HN vector which with the N->CA vector constitutes the peptide plane
     * @param gamma is Euler angle specifies the direction of N->CA vector given N-HN vector
     * @return (one of the) the rotation matrix that represents the rotation of
     * the peptide plane wrt. the (global) reference frame
     */
    public Matrix RotationWrtGlobalFrame(myVector3D nToh, double gamma){
	double[] n2h = myVector3D.normalize(nToh).getXYZ(); // get the direction cosines
	double xNH = n2h[0], yNH = n2h[1], zNH=n2h[2];
	double[][] mat = new double[3][3];
	double sinTheta = 9.99;
	Matrix A = new Matrix(3,3);
	double alpha1=9.99, alpha2=9.99, beta1=9.99, beta2=9.99, gamma1=9.99, gamma2=9.99;

	beta1 = Math.acos(zNH);
	double sinAlpha = yNH / Math.sin(beta1);
	double cosAlpha = xNH / Math.sin(beta1);
	if (( cosAlpha > 0) || (cosAlpha == 0) )
	    alpha1 = Math.asin(sinAlpha);
	else if ( cosAlpha < 0)
	    alpha1 = Math.PI-Math.asin(sinAlpha);
	//the second set
	beta2 = -beta1;
	sinAlpha = yNH / Math.sin(beta2);
	cosAlpha = xNH / Math.sin(beta2);
	if (( cosAlpha > 0) || (cosAlpha == 0) )
	    alpha2 = Math.asin(sinAlpha);
	else if ( cosAlpha < 0)
	    alpha2 = Math.PI-Math.asin(sinAlpha);
	//make a euler rotation matrix from three Euler angles
	Matrix mm = Matrix.eulerMat(alpha1, beta1, gamma);
	//invert the +z axis: point along the NH->N direction for the sake of easy math
	Matrix MM = Matrix.rotationMat(Math.PI, "+y");
	return (MM.times(mm));
    }    
}