/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2009 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, 01 December, 2009
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.0, Nov 18, 2009
 * @author        Chittaranjan Tripathy (2007-2009)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */
import java.io.*;
import java.util.*;

/**
 * Description of the class
 */
public class myProtein implements Iterable<myResidue>, java.io.Serializable {

    protected Vector<myResidue> __residues = new Vector<myResidue>(1);
    protected String __chain_ids = ""; // '' is the default chain id taken.
    protected String __header = ""; //new String(); //("HEADER    " + new Date().toString()); // It basically keeps the header info read from a PDB file
    protected int beginResidueNumber = 1; //TODO: it is not yet used. Might find useful in addResidue() and in the copy constructor.

    /**
     * Default constructor which creates an instance of an empty protein.
     */
    public myProtein() {
    }

    /**
     * This method is yet to be implemented.
     * @param inputFileName
     */
    protected myProtein(String inputFileName) {
        System.out.println("TODO/CAVEAT: This method is yet to be implemented. We don't see a requirement for implementing this method now");
        System.exit(1);
    }

    /**
     * A copy constructor which takes an instance of a protein and constructs
     * another protein with the same parameters.
     *
     * @param p the residue to be cloned
     */
    public myProtein(final myProtein p) {
        __chain_ids = p.getChainIds();
        __header = p.getHeader();
        for (myResidue r : p) {
            addResidue(new myResidue(r));
        }
        synchronizeProtein();
    }

    /**
     * Return the chain ids.
     *
     * @return a string containing all chain ids
     */
    public String getChainIds() {
        return __chain_ids;
    }

    /**
     * Set chain id.
     *
     * @param ch the chain id
     */
    public void setChainId(char ch) {
        if (__chain_ids.indexOf(ch) == -1)
            __chain_ids += ch;
    }

    /**
     * Return the header information.
     *
     * @return the header information
     */
    public String getHeader() {
        return __header;
    }

    /**
     * Modify/set the header information.
     *
     * @param headerForPdb the header information
     */
    public void setHeader(String headerForPdb) {
        __header = headerForPdb;
    }

    /**
     * Return an iterator that can iterate through the residues in the protein.
     *
     * @return an iterator that can iterate through the residues in the protein
     */
    public Iterator<myResidue> iterator() {
        return new ResidueIterator().iterator();
    }

    /**
     * Return an iterator that can iterate through the residues in the protein.
     *
     * @return an iterator that can iterate through the residues in the protein
     */
    public ResidueIterator residueIterator() {
        return new ResidueIterator();
    }

    /**
     * This class implements an iterator that can iterate through the residues
     * of a protein.
     */
    public class ResidueIterator implements Iterator<myResidue>, Iterable<myResidue> {

        private Iterator<myResidue> iter = __residues.iterator();

        public boolean hasNext() {
            return iter.hasNext();
        }

        public myResidue next() throws NoSuchElementException {
            if (!iter.hasNext()) {
                throw new NoSuchElementException();
            }
            return iter.next();
        }

        public void remove() {
            iter.remove();
        }

        public Iterator<myResidue> iterator() {
            return __residues.iterator();
        }
    }

    /**
     * Return an iterator that can iterate through the atoms of a protein.
     *
     * @return an iterator that can iterate through the atoms of a protein
     */
    public Iterator<myAtom> atomIterator() {
        return new AtomIterator();
    }

    /**
     * This class implements an iterator that can iterate through the atoms
     * of a protein.
     */
    public class AtomIterator implements Iterator<myAtom>, Iterable<myAtom> {

        private ResidueIterator resIter = residueIterator();
        private myResidue thisRes = resIter.next(); // Can be fatal. Throw exception when the Protein does not contain any residue!
        private Iterator<myAtom> atomIter = thisRes.iterator();

        public boolean hasNext() {
            if (!atomIter.hasNext() && !resIter.hasNext()) {
                return false;
            }
            if (atomIter.hasNext()) {
                return true;
            }
            if (!atomIter.hasNext() && resIter.hasNext()) {
                return true;
            }
            return false;
        }

        public myAtom next() throws NoSuchElementException {
            if (atomIter.hasNext()) {
                return atomIter.next();
            } else if (!atomIter.hasNext() && resIter.hasNext()) {
                thisRes = resIter.next();
                atomIter = thisRes.iterator();
                return atomIter.next();
            } else {//if (!resIter.hasNext()) // this is by default
                throw new NoSuchElementException();
            }
        }

        public void remove() {
            atomIter.remove();
        }

        public Iterator<myAtom> iterator() {
            return thisRes.iterator();
        }
    }

    /**
     * Return the number of residues in the protein.
     *
     * @return the number of residues
     */
    public int numberOfResidues() {
        return __residues.size();
    }

    /**
     * Return true if there is no residue.
     *
     * @return true if the number of residues is zero
     */
    public boolean isEmpty() {
        return numberOfResidues() == 0;
    }

    /**
     * Return the number of atoms in the protein.
     *
     * @return the number of atoms in the protein
     */
    public int numberOfAtoms() {
        int numAtoms = 0;
        if (numberOfResidues() != 0) {
            for (myResidue r : this) {
                numAtoms += r.numberOfAtoms();
            }
        }
        return numAtoms;            
    }

    /**
     * Return the sequence of the protein.
     *
     * @return a vector of strings which contains the sequence of the protein
     */
    public Vector<String> getSequence() {
        Vector<String> seq = new Vector<String>(numberOfResidues());
        for (myResidue r : this) {
            seq.add(r.getResidueName());
        }
        return seq;        
    }

    /**
     * Return a reference to the residue in the protein with the specified residue
     * number.
     *
     * @param resNum the residue number which specifies the residue being sought
     * @return a reference to the residue with the specified residue number
     */
    public myResidue residueAt(int resNum) {
        for (myResidue r: this) {
            if (resNum == r.getResidueNumber()) {
                return r;
            }
        }  
        return null;
    }

    /**
     * Return true if the protein has the residue with specified residue number
     * and chain id.
     *
     * @param resNum the residue number
     * @param chainId the chain id
     * @return true if the protein has the residue with specified residue number
     * and chain id
     */
    public boolean hasResidue(int resNum, char chainId) {
        for (ResidueIterator resIter = this.residueIterator(); resIter.hasNext(); ) {
             myResidue r = resIter.next();
             if (r.getResidueNumber() == resNum && getChainIds().indexOf(chainId) != -1) {
                 return true;
             }
        }
        return false;
    }

    /**
     * Add a residue to the protein.
     *
     * @param r the residue to be added
     */
    public void addResidue(myResidue r) { // TODO: whether you need to uncomment this part
        //Also, check whether a residue with this number is present.
        //if (beginResidueNumber + __residues.size() != __residues.size()) {// means the first residue number in the protein is not 1.
        //    r.setResidueNumber(beginResidueNumber + __residues.size()); 
        //}
        __residues.add(r);    
    }

    /**
     * Remove a residue with the specified residue number. If the residue is not
     * present then do nothing.
     * 
     * @param resNum the residue with this residue number is to be removed
     */
    public void removeResidue(int resNum) {        
        for (ResidueIterator resIter = this.residueIterator(); resIter.hasNext(); ) {
            myResidue r = resIter.next();
            if (r.getResidueNumber() == resNum) {
                resIter.remove();
            }
        }
    }

    /**
     * Add an atom to a residue in the protein. If the residue is not present
     * then do nothing.
     *
     * @param resNum specifies the residue number
     * @param a the atom to be added
     */
    public void addAtom(int resNum, myAtom a) {
        for (ResidueIterator resIter = this.residueIterator(); resIter.hasNext(); ) {
            myResidue r = resIter.next();
            if (r.getResidueNumber() == resNum) {
                r.addAtom(a);
                return;
            }
        }
    }

    /**
     * Remove an atom to a residue in the protein. If the residue is not present
     * then do nothing.
     *
     * @param resNum specifies the residue number
     * @param IUPAC_atom_name the atom to be added
     */
    public void removeAtom(int resNum, String IUPAC_atom_name) {
        for (ResidueIterator resIter = this.residueIterator(); resIter.hasNext(); ) {
            myResidue r = resIter.next();
            if (r.getResidueNumber() == resNum) {
                r.removeAtom(IUPAC_atom_name);                
                return;
            }
        }
    }

    /**
     * Connect the residues in the protein.
     */
    public void connectResidues() {
        if (this.numberOfResidues() < 2) {
            return;
        }
        //We now know that we have at least two residues.
        ResidueIterator resIter = this.residueIterator();
        myResidue rPrev = resIter.next();
        for (; resIter.hasNext();) {
            myResidue r = resIter.next();
            if (r.getResidueNumber() - rPrev.getResidueNumber() == 1) {
                // The two residues are adjacent, so connect them.
                myAtom C_i_1 = rPrev.getAtom(myAtomLabel.__C);
                myAtom N_i = r.getAtom(myAtomLabel.__N);
                if (C_i_1 != null && N_i != null) { //to take care of LINK
                    C_i_1.addBond(N_i);
                    N_i.addBond(C_i_1);
                    //System.out.println("Interlinking residues: " + r.getResidueNumber() + " " + rPrev.getResidueNumber());
                }
            }
            rPrev = r;
        }        
    }

    /**
     * Synchronize the protein by connecting the residues. This method connects
     * intra- and inter-residue bonds, but does not update the serial number,
     * chain id etc. TODO: In future this method might be further enhanced.
     */
    public void synchronizeProtein() {
        /*
         * (1) connect residues and complete the intra- and inter-residue bonds.
         * (2) Check and Update the serial number, chain id, residue number, residue name
         * (3) TODO: Think of other things that need to be synchronized.
         * (4) Think of how to represent and synchronize atom with some SSEs missing.
         */
        // (1) connect residues and complete the intra- and inter-residue bonds.
        connectResidues();               
    }    

    /**
     * Print the protein in PDB format.
     */
    public void print() {
        for (myResidue r : this) {
            Vector<String> res = r.getResidueRowsInOrder();
            for (String row : res) {
                System.out.println(row);
            }
        }
    }    

    /**
     * Print the protein into the file. This method either can append to an existing
     * file or can make a new file and print the protein into it.
     *
     * @param fileName the file where the protein to be output
     * @param append if true the the protein is appended to the existing file;
     * otherwise, overwrites the file.
     */
    public void print(String fileName, boolean append) {
        try {
            Writer outputFile = null;
            File file = new File(fileName);
            if (file.exists() && append == false) {
                file.delete();
            }
            outputFile = new BufferedWriter(new FileWriter(file, append)); 

            // write residues to the Pdb file
            for (myResidue r : this) {
                Vector<String> res = r.getResidueRowsInOrder();
                for (String row : res) {
                    outputFile.write(row + '\n');
                }
            }
            outputFile.write("TER                                                                             \n");
            outputFile.write("END                                                                             \n");
            outputFile.close();
        } catch (IOException e) {
            System.out.println("An IOException is thrown while writing to the outputfile inside myProtein->print()");
            e.printStackTrace();
            System.exit(1);
        }
    }

    /**
     * Print the protein to the writer object.
     *
     * @param outputFile the writer object which provides a handle for the output
     */
    public void print(Writer outputFile) { // This appends the protein to out
        try {            
            for (myResidue r : this) {
                Vector<String> res = r.getResidueRowsInOrder();
                for (String row : res) {
                    outputFile.write(row + '\n');
                }
            }
            outputFile.write("TER                                                                             \n");
            outputFile.write("END                                                                             \n");
            outputFile.close();
        } catch (IOException e) {
            System.out.println("An IOException is thrown while writing to the outputfile inside myProtein->print()");
            e.printStackTrace();
            System.exit(1);
        }
    }

    /**
     * Translate the protein by the amounts speficied.
     *
     * @param dx amount of translation along x-axis
     * @param dy amount of translation along y-axis
     * @param dz amount of translation along z-axis
     */
    public void translate(double dx, double dy, double dz) {
        translate(new myVector3D(dx, dy, dz));
    }

    /**
     * Translate the protein by the speficied vector.
     *
     * @param v the vector that specifies the translation
     */
    public void translate(myVector3D v) {
        for (myResidue r : this) {
            for (myAtom a : r) {
                a.getCoordinates().translate(v);
            }
        }
    }
    
    /**
     * Compute new coordinated after a rotation specified by the rotation matrix R.
     *
     * @param R the rotation matrix
     */
    public void rotate(final myMatrix R) {
        for (myResidue r : this) {
            for (myAtom a : r) {
                a.setCoordinates(new myPoint(R.times(a.getCoordinates().getXYZ())));
            }
        }
    }

    /**
     * Compute the dihedral angle given four points. The first three points define
     * the first plane and the second three points define the second plane.
     *
     * @param p1 the first point
     * @param p2 the second point
     * @param p3 the third point
     * @param p4 the fourth point
     * @return the dihedral angle in radian
     */
    public static double dihedralAngle(myPoint p1, myPoint p2, myPoint p3, myPoint p4) {
        myVector3D b1 = new myVector3D(p1, p2);
        myVector3D b2 = new myVector3D(p2, p3);        
        myVector3D b3 = new myVector3D(p3, p4);
        /*
        myVector3D Normal1 = myVector3D.crossProduct(v21, v23);
        myVector3D Normal2 = myVector3D.crossProduct(v32, v34);        
        double cosOfDihedral = myVector3D.cosineOfAngle(Normal1, Normal2);        
        int sign = (cosOfDihedral >= 0)? -1: 1;        
        double dihedral = Math.acos(cosOfDihedral) * sign;
        */        
        
        double arg1 = b2.norm() * myVector3D.dotProduct(b1, myVector3D.crossProduct(b2, b3));
        double arg2 = myVector3D.dotProduct(myVector3D.crossProduct(b1, b2), myVector3D.crossProduct(b2, b3));
        
        double dihedral = Math.atan2(arg1, arg2);
        return dihedral;       
    }

    /**
     * Print the PHI, PSI and OMEGA dihedral angles of the protein backbone.
     */    
    public void printBackboneDihedrals() {
        Iterator<myResidue> resIter = this .residueIterator();
        
        myResidue prevRes = null, currRes = null, nextRes = null;
        
        if (resIter.hasNext())
            prevRes = resIter.next();
        if (resIter.hasNext())
            currRes = resIter.next();
        
        double phi = 0.0, psi = 0.0;
        
        if (currRes.getResidueNumber() - prevRes.getResidueNumber() == 1) { // means these two residues are adjacent
            myAtom N1 = prevRes.getAtom(myAtomLabel.__N);
            myAtom Ca1 = prevRes.getAtom(myAtomLabel.__CA);
            myAtom C1 = prevRes.getAtom(myAtomLabel.__C);
            myAtom N2 = currRes.getAtom(myAtomLabel.__N);
            if ( N1 != null && Ca1 != null && C1 != null && N2 != null) {
                psi = dihedralAngle(N1.getCoordinates(), Ca1.getCoordinates(), C1.getCoordinates(), N2.getCoordinates());
                System.out.printf("Residue %4d: Psi  : %8.3f\n", prevRes.getResidueNumber(), (psi * 180.0 / Math.PI));
            }
        }        
 
        for (; resIter.hasNext(); ) {
            nextRes = resIter.next();
            if (currRes.getResidueNumber() - prevRes.getResidueNumber() == 1) { // means these two residues are adjacent               
                myAtom C0 = prevRes.getAtom(myAtomLabel.__C);
                myAtom N1 = currRes.getAtom(myAtomLabel.__N);
                myAtom Ca1 = currRes.getAtom(myAtomLabel.__CA);
                myAtom C1 = currRes.getAtom(myAtomLabel.__C);     
                if (C0 != null && N1 != null && Ca1 != null && C1 != null) {
                    phi = dihedralAngle(C0.getCoordinates(), N1.getCoordinates(), Ca1.getCoordinates(), C1.getCoordinates());
                    System.out.printf("Residue %4d: Phi  : %8.3f\n", currRes.getResidueNumber(), (phi * 180.0 / Math.PI));
                }
            }            
            if (nextRes.getResidueNumber() - currRes.getResidueNumber() == 1) { // means these two residues are adjacent
                myAtom N1 = currRes.getAtom(myAtomLabel.__N);
                myAtom Ca1 = currRes.getAtom(myAtomLabel.__CA);                
                myAtom C1 = currRes.getAtom(myAtomLabel.__C);   
                myAtom N2 = nextRes.getAtom(myAtomLabel.__N);                
                if (N1 != null && Ca1 != null && C1 != null && N2 != null) {
                    psi = dihedralAngle(N1.getCoordinates(), Ca1.getCoordinates(), C1.getCoordinates(), N2.getCoordinates());
                    System.out.printf("Residue %4d: Psi  : %8.3f\n", currRes.getResidueNumber(), (psi * 180.0 / Math.PI));
                }                
            }                      
            if (nextRes.getResidueNumber() - currRes.getResidueNumber() == 1) { // means these two residues are adjacent                
                myAtom Ca1 = currRes.getAtom(myAtomLabel.__CA);                
                myAtom C1 = currRes.getAtom(myAtomLabel.__C);   
                myAtom N2 = nextRes.getAtom(myAtomLabel.__N);  
                myAtom Ca2 = nextRes.getAtom(myAtomLabel.__CA);
                if (Ca1 != null && C1 != null && N2 != null && Ca2 != null) {
                    psi = dihedralAngle(Ca1.getCoordinates(), C1.getCoordinates(), N2.getCoordinates(), Ca2.getCoordinates());
                    System.out.printf("Residue %4d: Omega: %8.3f\n", currRes.getResidueNumber() + 1, (psi * 180.0 / Math.PI));
                    //System.out.printf("%4d    %8.3f\n", currRes.getResidueNumber() + 1, (psi * 180.0 / Math.PI));
                }                
            }             
            prevRes = currRes;
            currRes = nextRes;           
        }       

        if (currRes.getResidueNumber() - prevRes.getResidueNumber() == 1) { // means these two residues are adjacent
            myAtom C0 = prevRes.getAtom(myAtomLabel.__C);
            myAtom N1 = currRes.getAtom(myAtomLabel.__N);
            myAtom Ca1 = currRes.getAtom(myAtomLabel.__CA);
            myAtom C1 = currRes.getAtom(myAtomLabel.__C);
            if (C0 != null && N1 != null && Ca1 != null && C1 != null) {
                phi = dihedralAngle(C0.getCoordinates(), N1.getCoordinates(), Ca1.getCoordinates(), C1.getCoordinates());
                System.out.printf("Residue %4d: Phi  : %8.3f\n", currRes.getResidueNumber(), (phi * 180.0 / Math.PI));
            }
        }
    }

    /**
     * Compute the (PHI, PSI) pairs of the protein backbone.
     *
     * @return a vector of (PHI, PSI) pairs
     */
    public Vector<myPhiPsi> computeDihedrals() {
        Iterator<myResidue> resIter = this.residueIterator();

        myResidue prevRes = null, currRes = null, nextRes = null;

        if (resIter.hasNext())
            prevRes = resIter.next();
        if (resIter.hasNext())
            currRes = resIter.next();

        double phi = 0.0, psi = 0.0;

        Vector<myPhiPsi> phiPsiVec = new Vector<myPhiPsi>();

        if (currRes.getResidueNumber() - prevRes.getResidueNumber() == 1) { // means these two residues are adjacent
            myAtom N1 = prevRes.getAtom(myAtomLabel.__N);
            myAtom Ca1 = prevRes.getAtom(myAtomLabel.__CA);
            myAtom C1 = prevRes.getAtom(myAtomLabel.__C);
            myAtom N2 = currRes.getAtom(myAtomLabel.__N);
            if ( N1 != null && Ca1 != null && C1 != null && N2 != null) {
                psi = dihedralAngle(N1.getCoordinates(), Ca1.getCoordinates(), C1.getCoordinates(), N2.getCoordinates());
                //System.out.printf("Residue %4d: Psi  : %8.3f\n", prevRes.getResidueNumber(), (psi * 180.0 / Math.PI));
                phiPsiVec.add(new myPhiPsi(prevRes.getResidueNumber(), prevRes.getResidueName(), phi, psi));
            }
        }

        for (; resIter.hasNext(); ) {
            nextRes = resIter.next();
            if (currRes.getResidueNumber() - prevRes.getResidueNumber() == 1) { // means these two residues are adjacent
                myAtom C0 = prevRes.getAtom(myAtomLabel.__C);
                myAtom N1 = currRes.getAtom(myAtomLabel.__N);
                myAtom Ca1 = currRes.getAtom(myAtomLabel.__CA);
                myAtom C1 = currRes.getAtom(myAtomLabel.__C);
                if (C0 != null && N1 != null && Ca1 != null && C1 != null) {
                    phi = dihedralAngle(C0.getCoordinates(), N1.getCoordinates(), Ca1.getCoordinates(), C1.getCoordinates());
                    //System.out.printf("Residue %4d: Phi  : %8.3f\n", currRes.getResidueNumber(), (phi * 180.0 / Math.PI));
                }
            }
            if (nextRes.getResidueNumber() - currRes.getResidueNumber() == 1) { // means these two residues are adjacent
                myAtom N1 = currRes.getAtom(myAtomLabel.__N);
                myAtom Ca1 = currRes.getAtom(myAtomLabel.__CA);
                myAtom C1 = currRes.getAtom(myAtomLabel.__C);
                myAtom N2 = nextRes.getAtom(myAtomLabel.__N);
                if (N1 != null && Ca1 != null && C1 != null && N2 != null) {
                    psi = dihedralAngle(N1.getCoordinates(), Ca1.getCoordinates(), C1.getCoordinates(), N2.getCoordinates());
                    //System.out.printf("Residue %4d: Psi  : %8.3f\n", currRes.getResidueNumber(), (psi * 180.0 / Math.PI));
                    phiPsiVec.add(new myPhiPsi(currRes.getResidueNumber(), currRes.getResidueName(), phi, psi));
                }
            }
            if (nextRes.getResidueNumber() - currRes.getResidueNumber() == 1) { // means these two residues are adjacent
                myAtom Ca1 = currRes.getAtom(myAtomLabel.__CA);
                myAtom C1 = currRes.getAtom(myAtomLabel.__C);
                myAtom N2 = nextRes.getAtom(myAtomLabel.__N);
                myAtom Ca2 = nextRes.getAtom(myAtomLabel.__CA);
                if (Ca1 != null && C1 != null && N2 != null && Ca2 != null) {
                    psi = dihedralAngle(Ca1.getCoordinates(), C1.getCoordinates(), N2.getCoordinates(), Ca2.getCoordinates());
                    //System.out.printf("Residue %4d: Omega: %8.3f\n", currRes.getResidueNumber() + 1, (psi * 180.0 / Math.PI));
                    //System.out.printf("%4d    %8.3f\n", currRes.getResidueNumber() + 1, (psi * 180.0 / Math.PI));//old
                }
            }
            prevRes = currRes;
            currRes = nextRes;
        }

        if (currRes.getResidueNumber() - prevRes.getResidueNumber() == 1) { // means these two residues are adjacent
            myAtom C0 = prevRes.getAtom(myAtomLabel.__C);
            myAtom N1 = currRes.getAtom(myAtomLabel.__N);
            myAtom Ca1 = currRes.getAtom(myAtomLabel.__CA);
            myAtom C1 = currRes.getAtom(myAtomLabel.__C);
            if (C0 != null && N1 != null && Ca1 != null && C1 != null) {
                phi = dihedralAngle(C0.getCoordinates(), N1.getCoordinates(), Ca1.getCoordinates(), C1.getCoordinates());
                //System.out.printf("Residue %4d: Phi  : %8.3f\n", currRes.getResidueNumber(), (phi * 180.0 / Math.PI));
                phiPsiVec.add(new myPhiPsi(currRes.getResidueNumber(), currRes.getResidueName(), phi, Const.phiAveHelix)); 
            }
        }

        return phiPsiVec;
    }
    
    /**
     * Compute the coordinates of the backbone atoms (N, H, CA, HA, CB, C and O)
     * of a protein fragment given the first peptide plane and a sequence of PHI
     * and PSI angles using ideal peptide geometry assumptions.
     *
     * @param phiPsiVec phiPsiVec a vector of (PHI, PSI) pairs
     * @param initialPeptidePlane initialPeptidePlane the initial peptide plane
     * @return the constructed protein fragment
     */
    public static myProtein buildProteinBackboneFromDihedrals(final Vector<myPhiPsi> phiPsiVec, final myPeptidePlane initialPeptidePlane) {
        return myForKin. buildBackbone(phiPsiVec, initialPeptidePlane);

//        Vector<Pdb> pdbVec = myModelBackboneUsingRdc.CRTModelBuild(phiPsiVec, pPlane, false);
//        int numberOfResidues = phiPsiVec.size();
//        return myTestProteinClone.getProteinFromLFormat(pdbVec);
/*
        myPeptidePlane pPlane = new myPeptidePlane(initialPeptidePlane); //get a local copy of the initial pp and use that.
        myProtein thisFragment = new myProtein();
        int serialNum = 1;
        Vector<myAtom> atomsOfThisResidue = new Vector<myAtom>(7);

        for (myPhiPsi thisPhiPsiPair : phiPsiVec) {
            atomsOfThisResidue.clear(); // clear this vector so that it can be reused

            int thisResidueNumber = thisPhiPsiPair.getResidueNumber();
            String thisResidueName = thisPhiPsiPair.getResidueName();
            double thisPhi = thisPhiPsiPair.getPhi();
            double thisPsi = thisPhiPsiPair.getPsi();

            myPoint Ni = pPlane.getN();
            myPoint Hi = pPlane.getH();
            myPoint CAi = pPlane.getCa();

            atomsOfThisResidue.add(new myAtom(myAtomLabel.__N, Ni));
            atomsOfThisResidue.add(new myAtom(myAtomLabel.__H, Hi));
            atomsOfThisResidue.add(new myAtom(myAtomLabel.__CA, CAi));

            Matrix rg = myPeptidePlane.RotationWrtGlobalFrame(pPlane);
            Matrix rgInv = rg.transpose();
            Matrix r2yInv = Matrix.rotationMat(thisPhi, "-y"); //transpose == inverse == rotate in opposite direction
            Matrix r4zInv = Matrix.rotationMat(thisPsi + Math.PI, "-z");
            //rg.print(8, 3);
            Matrix matT = rgInv.times(Const.r1x9yInv.times(r2yInv));

            // Compute the HA coordinate of residue (i)
            myPoint HAi = new myPoint(0.0, 0.0, Const.dCA2HA);
            myVector3D Ca2HaVec = new myVector3D(matT.times(Const.rHA2HA1Inv.times(HAi.getXYZ())));
            HAi = myPoint.translate(CAi, Ca2HaVec);
            atomsOfThisResidue.add(new myAtom(myAtomLabel.__HA, HAi));

            // Compute the CB coordinate of residue (i)
            myPoint CBi = new myPoint(0.0, 0.0, Const.dCA2CB);
            myVector3D Ca2CbVec = new myVector3D(matT.times(Const.rCb2Cb1Inv.times(CBi.getXYZ())));
            CBi = myPoint.translate(CAi, Ca2CbVec);
            atomsOfThisResidue.add(new myAtom(myAtomLabel.__CB, CBi));

            // Compute the C' coordinate residue (i)
            matT = matT.times(Const.r3xInv);
            myPoint COi = new myPoint(0.0, 0.0, Const.dCA2CO);
            myVector3D Ca2CoVec = new myVector3D(matT.times(COi.getXYZ()));
            COi = myPoint.translate(CAi, Ca2CoVec);
            atomsOfThisResidue.add(new myAtom(myAtomLabel.__C, COi));

            // Compute the O coordinate residue (i)
            myPoint Oi = new myPoint(0.0, 0.0, Const.dCO2O);
            myVector3D Co2OVec = new myVector3D(matT.times(r4zInv.times(Const.rOInv.times(Oi.getXYZ()))));
            Oi = myPoint.translate(COi, Co2OVec);
            atomsOfThisResidue.add(new myAtom(myAtomLabel.__O, Oi));

            // Compute the N coordinates for residue (i + 1)
            myPoint Ni1 = new myPoint(0.0, Const.dCO2N, 0.0);
            matT = matT.times(r4zInv.times(Const.r5xInv));
            myVector3D Co2NVec = new myVector3D(matT.times(Ni1.getXYZ()));
            Ni1 = myPoint.translate(COi, Co2NVec);
            pPlane.setN(Ni1);

            // Compute the CA coordinate residue (i + 1)
            myPoint CAi1 = new myPoint(0.0, Const.dN2CA, 0.0);
            matT = matT.times(Const.r6yInv.times(Const.r7xInv));
            myVector3D N2CaVec1 = new myVector3D(matT.times(Const.r8zInv.times(Const.r1xInv.times(CAi1.getXYZ()))));
            CAi1 = myPoint.translate(Ni1, N2CaVec1);
            pPlane.setCa(CAi1);

            // Compute the HN coordinate residue (i + 1)
            myPoint Hi1 = new myPoint(0.0, 0.0, -Const.dN2H);
            myVector3D N2HVec1 = new myVector3D(matT.times(Hi1.getXYZ()));
            Hi1 = myPoint.translate(Ni1, N2HVec1);
            pPlane.setH(Hi1);

            // Given Ni, Hi, CAi, we computed HAi, CBi COi, Oi, Ni+1, Hi+1, CAi+1
            myResidue r = new myResidue(thisResidueName, thisResidueNumber);
            r.clear();
            for (myAtom a: atomsOfThisResidue) {
                myAtom thisAtom = new myAtom("ATOM", serialNum++, a.getAtomName(), thisResidueName, 'A',
                        thisResidueNumber, new myPoint(a.getCoordinates()), 1.00, 0.00, "    ", "  ", "  ");
                r.addAtom(thisAtom);
            }
            thisFragment.addResidue(r);
        }

        // Add the last half residue computed since PSI for this is given.
        myResidue r = new myResidue("ALA", phiPsiVec.lastElement().getResidueNumber() + 1);
        r.clear();
        r.addAtom(new myAtom("ATOM", serialNum++, myAtomLabel.__N, "ALA", 'A',
                phiPsiVec.lastElement().getResidueNumber() + 1, new myPoint(pPlane.getN()), 1.00, 0.00, "    ", "  ", "  "));
        r.addAtom(new myAtom("ATOM", serialNum++, myAtomLabel.__H, "ALA", 'A',
                phiPsiVec.lastElement().getResidueNumber() + 1, new myPoint(pPlane.getH()), 1.00, 0.00, "    ", "  ", "  "));
        r.addAtom(new myAtom("ATOM", serialNum++, myAtomLabel.__CA, "ALA", 'A',
                phiPsiVec.lastElement().getResidueNumber() + 1, new myPoint(pPlane.getCa()), 1.00, 0.00, "    ", "  ", "  "));
        thisFragment.addResidue(r);

        thisFragment.synchronizeProtein();
        return thisFragment;
        */
    }

    /**
     * p3 = clone(p1) + clone(p2).
     * 
     * @param p1 the first protein
     * @param p2 the second protein
     * @return a new protein that represents the merged protein
     */
    public static myProtein cloneAndMergeTwoProteinFragments(myProtein p1, myProtein p2) {
        if (p1 == null && p2 == null) {
            return null;
        } else if (p1 == null) {
            return new myProtein(p2);
        } else if (p2 == null) {
            return new myProtein(p1);
        }

        myProtein clone = new myProtein(p1);
        clone.__chain_ids += p2.__chain_ids;
        clone.__header += p2.__header;

        for (myResidue r : p2) {
            clone.addResidue(new myResidue(r));
        }
        clone.synchronizeProtein();

        return clone;
    }

    /**
     * Clone and sort the residues of a protein.
     *
     * @param p the protein which is to be cloned and the residues will be sorted.
     * @return the cloned protein with residues sorted
     */
    public static myProtein cloneAndSortResidues(myProtein p) {
        if (p == null) {
            return null;
        }

        myProtein clone = new myProtein();
        clone.__chain_ids = p.__chain_ids;
        clone.__header = p.__header;
        int lowResNum = 99999;
        int hiResNum = 0;

        for (myResidue r : p) {
            if (r.getResidueNumber() < lowResNum) {
                lowResNum = r.getResidueNumber();
            }
            if (r.getResidueNumber() > hiResNum) {
                hiResNum = r.getResidueNumber();
            }
        }

        for (int i = lowResNum; i <= hiResNum; i++) {
            if (p.residueAt(i) != null) {
                clone.addResidue(new myResidue(p.residueAt(i)));
            }
        }
        clone.synchronizeProtein();

        return clone;
    }

    public static myProtein extractBackbone(myProtein prot) {
        myProtein p = new myProtein(prot);
        for (myResidue r : p) {
            // Remove illegal atoms for this residue
            Vector<myAtom> listOfIllegalAtoms = new Vector<myAtom>();
            for (myAtom a : r) {
                if (a.getAtomName().equalsIgnoreCase("N") || a.getAtomName().equalsIgnoreCase("CA") ||
                        a.getAtomName().equalsIgnoreCase("C") || a.getAtomName().equalsIgnoreCase("O") ||
                        a.getAtomName().equalsIgnoreCase("OXT") || a.getAtomName().equalsIgnoreCase("H") ||
                        a.getAtomName().equalsIgnoreCase("HN") || a.getAtomName().equalsIgnoreCase("H1") ||
                        a.getAtomName().equalsIgnoreCase("H2") || a.getAtomName().equalsIgnoreCase("H3") ||
                        a.getAtomName().equalsIgnoreCase("HA2") || a.getAtomName().equalsIgnoreCase("HA3") ||
                        a.getAtomName().equalsIgnoreCase("HA") || a.getAtomName().equalsIgnoreCase("CB")) {
                    ; // do nothing
                } else {
                    listOfIllegalAtoms.add(a);
                }
            }                   
            for (myAtom a : listOfIllegalAtoms) {
                r.removeAtom(a.getAtomName());
            }
        }
        p.synchronizeProtein();
        return p;
    }

    /**
     * The main method tests this class.
     *
     * @param args
     */
    public static void main(String[] args) {
        //myProtein p = new myProtein("1D3ZModel1.pdb");
        //myProtein p = new myProtein("bf.pdb");
        //myProtein p = new myProtein("testAct.pdb");
        //myProtein p = new myProtein("1H3D.pdb");
        //myProtein p = myPdbParser.parsePdb("1H3D.pdb");   
        //myPdbParser pParser =  new myPdbParser();
//        //myProtein p = pParser.parsePdb("2I5O.pdb");
        //myProtein p = pParser.parsePdb("1H3D.pdb");
        //myProtein p = new myProtein("2PQ1Test.pdb");
        //myProtein p = new myProtein("2PQ1TestFH.pdb");
        //myProtein p = new myProtein("2I5OModel1.pdb");
        myProtein p = null;
        Vector<myProtein> vp = new Vector<myProtein>();
        myPdbParser pParser = new myPdbParser("chittuTesting.pdb");//("h1331.pdb");//("1EF1.pdb");//("1D3Z.pdb");
        while (pParser.hasNextProtein()) {
            vp.add(pParser.nextProtein());
        }
        
        System.out.println(vp.size());
        p = vp.elementAt(0);
        
        System.out.println("Number of Residues: " + p.numberOfResidues() + " Number of Atoms: " + p.numberOfAtoms());
        System.out.println("There are " + p.getSequence().size() + " residues in the following order: " + p.getSequence().toString());
        //System.out.println("Printing the header: --------------------\n" + p.getHeader() + "------------------------------------------");
        System.out.println("Has residue 45? " + p.hasResidue(45, 'A'));
        System.out.println("Chain Id: " + p.getChainIds());
        
        System.out.println("Congratulations! The PDB file has been parsed successfully and all the models have been extracted");
        //System.exit(1);        
        
        
        System.out.println("Printing the protein residue by residue");
        System.out.println("---------------------------------------");
        for (myResidue r : p) {
            System.out.println("Residue Name: " + r.getResidueName() + " Residue Number: " + r.getResidueNumber() + " #Atoms: " + r.numberOfAtoms() + " #Bonds :" + r.numberOfBonds());
            for (myAtom a : r) {
                a.print();
            //break;
            }
        }
        
        System.out.println("Printing the protein atom by atom");
        System.out.println("---------------------------------");
        int atomCounter = 0;
        Iterator<myAtom> iter = p.atomIterator();
        while (iter.hasNext()) {
            iter.next().print(); 
            atomCounter++;
        }
        System.out.println("Total number of atoms printed: " + atomCounter);
        
        //p.residueAt(2).print();
        //p.synchronizeProtein();
        System.out.println("Printing all bond information");
        System.out.println("-----------------------------");
        for (myResidue r : p) {
            for (myAtom a : r) {
                System.out.print(a.getAtomName() + " of " + a.getResidueName() + " " + a.getResidueNumber() + " is bonded to: ");
                a.printBonds();
            }
        }
        
        System.out.println("Printing all carbonyl carbons");
        System.out.println("-----------------------------");
        for (myResidue r : p) {
            for (myAtom a : r) {
                if (a.getVdwRadius() == 165.0 && a.getAtomName().length() >= 1) {
                    System.out.print(a.getAtomName() + " of " + a.getResidueName() + " " + a.getResidueNumber() + " #bonds: " + a.numberOfBonds() + " vdw: " + a.getVdwRadius() + " is bonded to: ");
                    a.printBonds();
                }
            }
        }
        
        System.out.println("Printing all aromatic and polar hydrogens");
        System.out.println("-----------------------------------------");
        for (myResidue r : p) {
            for (myAtom a : r) {
                if (a.getVdwRadius() == 100 && a.getAtomName().startsWith("H")) {
                    System.out.print(a.getAtomName() + " " + a.getResidueName() + " " + a.getResidueNumber() + " vdw: " + a.getVdwRadius() + " is bonded to: ");
                    a.printBonds();
                }
            }
        }
        
        System.out.println("Printing steric check results");
        System.out.println("-----------------------------");
        myStericChecker sc = new myStericChecker(1.0, 0.8, 0.4); // clashThresh, hBondThresh, softStericThresh
                
//        int countClashes = 0;
//        for (myResidue r1 : p) {
//            for (myAtom a1 : r1)
//                for (myResidue r2 : p)
//                    //if (r2.getResidueNumber() != r1.getResidueNumber())
//                    for (myAtom a2 : r2) {                        
//                        if (sc.hasStericClash(a1, a2, 0.40)) {                             
//                            System.out.println("The atoms are: " + '\t' + a1.getResidueName() + '\t' + a1.getResidueNumber() + '\t' + a1.getAtomName() +
//                                    '\t' + a2.getResidueName() + '\t' + a2.getResidueNumber() + '\t' + a2.getAtomName() + '\t' +
//                                    (a1.distance(a2) - ((a1.getVdwRadius() + a2.getVdwRadius()) / 100)));
//                            countClashes++;
//                        }
//                    }
//        }
//        System.out.println("Total number of clashes: " + countClashes);    
//        Iterator<myAtom> iter = p.atomIterator();    
////        for (; iter.hasNext();) {
////            myAtom a = iter.next();
////            a.print();
////        }
        
        sc.checkStericClash(p);
        
//        int ctr = 0;
//        for (myResidue r1 : p)
//            for (myAtom a1 : r1)
//                for (myResidue r2 : p)                    
//                    for (myAtom a2 : r2) ctr++;
//        System.out.println("#atoms: " + p.numberOfAtoms() + "     crt_Sqrt: " + Math.sqrt(ctr));           
        
        System.out.println("#atoms: " + p.numberOfAtoms() + "   #residues: " + p.numberOfResidues());
        
        
        System.out.println("Testing orderly print file...");
        p.print("crapDelete.txt", false);
        
        System.out.println("Testing PhiPsi...");
        p.printBackboneDihedrals();
        
        System.out.println("Testing rotations...");
        //myMatrix Rx = { {1, 0, 0}, {
                        
        //p.rotate(rotX(Math.PI * 2));
        
        p.rotate(new myMatrix(Const.xInv));
        
        p.print();
                
        Vector<myProtein> vp2 = new Vector<myProtein>();
        myPdbParser pParser2 = new myPdbParser("hChittu1.pdb");//("MH1.pdb");//("ChitSSE1.pdb");//("h1331.pdb");//("1EF1.pdb");//("1D3Z.pdb");
        while (pParser2.hasNextProtein()) {
            vp2.add(pParser2.nextProtein());
        }                      
        myProtein p1 = vp2.elementAt(0);

        Vector<myProtein> vp3 = new Vector<myProtein>();
        myPdbParser pParser3 = new myPdbParser("burritto.pdb"); //("MH2.pdb");//("ChitSSE2.pdb");//("h1331.pdb");//("1EF1.pdb");//("1D3Z.pdb");
        while (pParser3.hasNextProtein()) {
            vp3.add(pParser3.nextProtein());
        }                      
        myProtein p2 = vp3.elementAt(0);        
        
        p2.rotate(new myMatrix(Const.yzInv));
        //p2.rotate(new myMatrix(Const.xzInv));
        
        p1.print();
        p2.print();
        
        //System.out.println("Translating Protein 2");
        //p1.translate(-20, 20, 20);
        //p2.translate(50, 50, 50);
        //p1.print();
        //p2.translate(-11.2670, 7.6820, 10.9040);
        //p2.print();        
        //System.out.println("Exiting...");
        //System.exit(0);
        
        //System.out.println("Wait, I am packing...");
        //double[] dummy = new double[3];
        //myPacker.doSsePackingByNoeSet(p1, p2, noeVec, dummy);

        //p1.print();
        //p2.print();
    }
    
      
    /**
     * *************************Design Place***************************
     */

    public static myMatrix rotX(double angle) {
        // Assuming the angle is in radians.
        double c = Math.cos(angle);
        double s = Math.sin(angle);
        return new myMatrix(new double[][]{{1.0, 0.0, 0.0}, {0.0, c, -s}, {0.0, s, c}});
    }

    public static myMatrix rotY(double angle) {
        // Assuming the angle is in radians. (?)
        Double c = Math.cos(angle);
        double s = Math.sin(angle);
        return new myMatrix(new double[][]{{c, 0.0, s}, {0.0, 1.0, 0.0}, {-s, 0.0, c}});
    }

    public static myMatrix rotZ(double angle) {
        // Assuming the angle is in radians. (?)
        Double c = Math.cos(angle);
        double s = Math.sin(angle);
        return new myMatrix(new double[][]{{c, -s, 0.0}, {s, c, 0.0}, {0.0, 0.0, 1}});
    }



    /**
     * Return the first residue number.
     *
     * @return the first residue number
     */
    int getBeginResidueNumber() {
        int beginResNum = __residues.elementAt(0).getResidueNumber();
        for (myResidue r : __residues) {
            beginResNum = Math.min(beginResNum, r.getResidueNumber());
        }
        return beginResNum;
    }

    /**
     * Return the last residue number.
     *
     * @return the last residue number
     */
    int getEndResidueNumber() {
        int endResNum = __residues.elementAt(0).getResidueNumber();
        for (myResidue r : __residues) {
            endResNum = Math.max(endResNum, r.getResidueNumber());
        }
        return endResNum;
    }

    
}

