/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2009 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, 01 December, 2009
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.0, Nov 18, 2009
 * @author        Chittaranjan Tripathy (2007-2009)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */
import java.io.*;
import java.util.*;

/**
 * Description of the class
 */
public class myBackboneModeler {
    /**
     * A random number generator that is used to model experimental errors in
     * dipolar couplings.
     */
    private static final Random rr = new Random(myMiscConstants.CRTRandomSeed);

    /**
     * This is the only instance (singleton) of this class. For now, we designed
     * this class to be a singleton. Later the decision can be changed, if needed.
     */
    private static final myBackboneModeler __instance = new myBackboneModeler();

    /**
     * This method returns the only instance of the object of this type.
     *
     * @return the only instance of an object of this type
     */
    public static synchronized myBackboneModeler getInstance() {
        return __instance;
    }

    /**
     * This is overriden to make sure that a singleton cannot be cloned.
     * 
     * @return
     * @throws CloneNotSupportedException
     */
    @Override public Object clone() throws CloneNotSupportedException {
        throw new CloneNotSupportedException();
    }

    /**
     * Default constructor which creates an instance of an empty modeler. This
     * constructor is private to enforce the singleton property of the class.
     */
    private myBackboneModeler() {
        //throw new AssertionError(); // to indicate that the object construction failed
    }
    
    /**
     * computes the rhombicity = 2/3 * |(Syy - Szz) / Sxx|.
     *
     * @param Syy the y-component of the diagonalized alignment tensor
     * @param Szz the z-component of the diagonalized alignment tensor
     * @return rhombicity
     */
    public static double computeRhombicity(double Syy, Double Szz) {
        double Sxx = -(Syy + Szz);
        double SxxAbs = Math.abs(Sxx);
        double SyyAbs = Math.abs(Syy);
        double SzzAbs = Math.abs(Szz);

        double min = Math.min(SxxAbs, Math.min(SyyAbs, SzzAbs));
        double max = Math.max(SxxAbs, Math.max(SyyAbs, SzzAbs));

        double rho = 2. / 3. * ((max - min) - min) / (max); // One can think of: max - min = Syy and min = Szz
        return rho;
    }

    /**
     * Return the dipolar coupling object with specified residue number if present
     * in the vector of dipolar coupling; otherwise, return null.
     *
     * @param dcVec the vector of dipolar couplings
     * @param residueNumber the residue number of the dipolar coupling object being searched
     * @return the dipolar coupling object with specified residue number if present
     * in the vector of dipolar coupling; otherwise, return null
     */
    private static myDipolarCoupling findInRdcVector(final Vector<myDipolarCoupling> dcVec, int residueNumber) {
        for (myDipolarCoupling dc : dcVec) {
            if (dc.getResidueNumber() == residueNumber) {
                return dc;
            }
        }
        return null;
    }

    /**
     * This method fills in the missing RDCs by the back computed RDCs on the
     * same vectors. It returns an array of integers which encodes a 1/0 sequence
     * describing whether experimental RDC is present or back computed RDC is used.
     *
     * @param thisSseId the SSE boundary definition
     * @param expRdcs the experiemntal RDCs
     * @param backCalRdcs the back computed RDCs
     * @param filledRdcVec this vector is is filled in so that the caller can use
     * it further. It contains all the RDCs for the SSE with nothing missing
     * @return a 1/0 array describing whether experimental RDC is used (1) or back
     * computed RDC is used for a residue
     */
    private static int[] fillInMissingRdcs(final mySseInfo thisSseId, final Vector<myDipolarCoupling> expRdcs,
            final Vector<myDipolarCoupling> backCalRdcs, Vector<myDipolarCoupling> filledRdcVec) {
        assert filledRdcVec != null : "Error: null RDC vector passed as argument";
        filledRdcVec.clear(); // clear all old data if there
        int[] expOrBackCal = new int[thisSseId.getEndResidueNumber() - thisSseId.getBeginResidueNumber() + 1];

        // We assume that expRdcs and backCalRdcs are sorted according to their ResidueNumber keys and they are the same type of RDCs
        for (int i = thisSseId.getBeginResidueNumber(); i <= thisSseId.getEndResidueNumber(); i++) {
            myDipolarCoupling dcExp = findInRdcVector(expRdcs, i);
            if (dcExp != null) {
                myDipolarCoupling dcExpClone = new myDipolarCoupling(dcExp);
                filledRdcVec.add(dcExpClone);
                expOrBackCal[i - thisSseId.getBeginResidueNumber()] = 1; // means experimental RDC is used
            } else { // use the back calculated RDC to fill in
                myDipolarCoupling dcCal = findInRdcVector(backCalRdcs, i);
                if (dcCal != null) {
                    myDipolarCoupling dcCalClone = new myDipolarCoupling(dcCal);
                    filledRdcVec.add(dcCalClone);
                    expOrBackCal[i - thisSseId.getBeginResidueNumber()] = 0; // means back computed RDC is used
                } else { // no back-computed RDC found for the residue
                    System.out.println(backCalRdcs.size());
                    for (myDipolarCoupling dc : backCalRdcs)
                        System.out.println(dc.toString());
                    System.out.println("Error: While filling the missing RDCs no back-computed RDC for the residue " + i);
                    System.exit(1);
                }
            }
        }
        return expOrBackCal;
    }

    /**
     * Count and return the number of experimental RDCs present in the 1/0 array
     * encoding the presence of experimental/back-computed RDCs.
     *
     * @param whetherExpRdcPresent 1/0 array encoding the presence of experimental/back-computed RDCs
     * @return the number of experimental RDCs
     */
    private int countExperimentalRdcs(final int[] whetherExpRdcPresent) {
        int ctr = 0;
        for (int i = 0; i < whetherExpRdcPresent.length; i++) {
            if (whetherExpRdcPresent[i] == 1) {
                ctr++;
            }
        }
        return ctr;
    }

    /**
     * This method generates a vector of sampled RDCs given a vector of RDCs to
     * be sampled along with their identifications (experimental/back-computed),
     * the rmsd threshold, and a random number generator (Gaussian) object. It stores the
     * sampled RDCs in the vector passed as an argument, and returns the sum of
     * squared deviations of sampled RDCs from experimental RDCs.
     *
     * @param rdcExpWithMissingRdcsFilled the RDC vector with both experimental and back-computed RDCs
     * @param expOrBackCal the vector which identifies whether an RDC to be sampled is experimental or back computed
     * @param rmsdThreshold the rmsd threshold for the sampled set
     * @param rdcSampled the vector that holds the sampled RDCs
     * @param rr a random number generator (Gaussian) object
     * @return the sum of squared deviations of sampled RDCs from experimental RDCs
     */
    private double generateSampledRdcVector(final Vector<myDipolarCoupling> rdcExpWithMissingRdcsFilled,
            final int[] expOrBackCal, final double rmsdThreshold, Vector<myDipolarCoupling> rdcSampled, Random rr) {
//        System.out.println("rdcExpWithMissingRdcsFilled.size():  " + rdcExpWithMissingRdcsFilled.size() + "    " +
//                "rdcSampled.size(): " + rdcSampled.size() + "    " + "expOrBackCal.length: " + expOrBackCal.length);
        assert rdcExpWithMissingRdcsFilled.size() == rdcSampled.size() && rdcExpWithMissingRdcsFilled.size() == expOrBackCal.length : "Error: The lengths of the sampled RDC vector and the filled experimental RDC vector differ";

        double sumOfSquaredDeviationThreshold = countExperimentalRdcs(expOrBackCal) * rmsdThreshold * rmsdThreshold;
        double squaredSum = 0;
        do {
            squaredSum = 0.0;
            for (int i = 0; i < rdcExpWithMissingRdcsFilled.size(); i++) {
                rdcSampled.elementAt(i).setRdc(rdcExpWithMissingRdcsFilled.elementAt(i).getRdc() + myMiscConstants.CRTGaussianEpsilon * rmsdThreshold * rr.nextGaussian());
                squaredSum += expOrBackCal[i] * (rdcSampled.elementAt(i).getRdc() - rdcExpWithMissingRdcsFilled.elementAt(i).getRdc()) * (rdcSampled.elementAt(i).getRdc() - rdcExpWithMissingRdcsFilled.elementAt(i).getRdc());
            }
        } while (squaredSum > sumOfSquaredDeviationThreshold);
        return squaredSum;
    }

    /**
     * Test if RDCs exist for boundary residues of the SSE so that the SSE can be
     * computed reliably.
     *
     * @param thisRdcCsaTable the table containing all RDCs
     * @param mediumName specification of the medium
     * @param typesOfRdcsToBeUsed specify the types of RDCs used
     * @param thisSseId the specification of SSE boundary
     * @return true if RDCs exist for both begin and end residues of the SSE; otherwise, return false
     */
    private boolean testIfRdcsExistForBoundaryResiduesOfSse(final myDipolarCouplingTable thisRdcCsaTable, String mediumName,
            Vector<myDipolarCoupling.Type> typesOfRdcsToBeUsed, final mySseInfo thisSseId) {
        if (myMiscConstants.testIfRdcsExistForBoundaryResiduesOfSse == false) {
            return true;
        }

        int beginResidueNumber = thisSseId.getBeginResidueNumber();
        int endResidueNumber = thisSseId.getEndResidueNumber();
        boolean beg = false, end = false;

        for (myDipolarCoupling.Type type : typesOfRdcsToBeUsed) {
            if (thisRdcCsaTable.getRdc(mediumName, type, beginResidueNumber) != null) {
                beg = true;
                break;
            }
        }

        for (myDipolarCoupling.Type type : typesOfRdcsToBeUsed) {
            if (thisRdcCsaTable.getRdc(mediumName, type, endResidueNumber) != null) {
                end = true;
                break;
            }
        }
        
        if (beg == true && end == true) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Compute the alignment tensor given the RDCs and the protein structure.
     *
     * @param thisRdcCsaTable the table containing all RDCs
     * @param mediumName specification of the medium
     * @param typesOfRdcsToBeUsed specify the types of RDCs used
     * @param p the protein structure
     * @param saupe this array holds the computed Syy and Szz values
     * @return the conformation of a copy of p in one of the principal order frames
     */
    public myProtein computeAlignmentTensorGivenStructureAndRdcs(final myDipolarCouplingTable thisRdcCsaTable, String mediumName,
            Vector<myDipolarCoupling.Type> typesOfRdcsToBeUsed, final myProtein p, double[] saupe) {

        Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMedium = thisRdcCsaTable.getRdcMapForThisMedium(mediumName);
        Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMediumBackCal = new TreeMap<myDipolarCoupling.Type, Vector<myDipolarCoupling>>();

        // Compute the alignment tensor and the rotation to one of the POFs
        Matrix rotToPOF = myAlignmentTensorEstimator.bestFitUsingSVD(p, rdcCsaTableForThisMedium,
                typesOfRdcsToBeUsed, rdcCsaTableForThisMediumBackCal, saupe);

        myProtein thisFragmentInPOF = new myProtein(p);
        thisFragmentInPOF.rotate(new myMatrix(rotToPOF.getArrayCopy()));

        return thisFragmentInPOF;
    }

    /**
     * Compute the alignment tensor given the RDCs and the protein structure.
     *
     * @param thisRdcCsaTable the table containing all RDCs
     * @param mediumName specification of the medium
     * @param typesOfRdcsToBeUsed specify the types of RDCs used
     * @param p the protein structure
     * @param saupe this array holds the computed Syy and Szz values
     * @param rdcCsaTableForThisMediumBackCal a table containing all back-computed RDCs
     * @return the conformation of a copy of p in one of the principal order frames
     */
    public myProtein computeAlignmentTensorGivenStructureAndRdcs(final myDipolarCouplingTable thisRdcCsaTable, String mediumName,
            Vector<myDipolarCoupling.Type> typesOfRdcsToBeUsedInTensorRefinement, final myProtein p, double[] saupe,
            Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMediumBackCal) {

        Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMedium = thisRdcCsaTable.getRdcMapForThisMedium(mediumName);
        //Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMediumBackCal = new TreeMap<myDipolarCoupling.Type, Vector<myDipolarCoupling>>();

        // Compute the alignment tensor and the rotation to one of the POFs
        Matrix rotToPOF = myAlignmentTensorEstimator.bestFitUsingSVD(p, rdcCsaTableForThisMedium,
                typesOfRdcsToBeUsedInTensorRefinement, rdcCsaTableForThisMediumBackCal, saupe);

        myProtein thisFragmentInPOF = new myProtein(p);
        thisFragmentInPOF.rotate(new myMatrix(rotToPOF.getArrayCopy()));

        return thisFragmentInPOF;
    }

    /**
     * Bootstrap the alignment tensor computation and iteratively refine it to
     * finally obtain a good estimate of the alignment tensor.
     *
     * @param thisRdcCsaTable the table containing all RDCs
     * @param mediumName specification of the medium 
     * @param typesOfRdcsToBeUsedInTensorRefinement specify the types of RDCs to be used for alignment tensor computation
     * @param typesOfRdcsToBeUsedForAnalyticSolution speficy the types of RDCs to be used for constructing the secondary structure element
     * @param thisSseId the secondary structure element to be used to bootstrap the alignment tensor
     * @param refineCycle the number of iterations to be done during the alignment tensor computation
     * @param dfsCycle the number of search trees to be enumerated, i.e., the number of sampled RDC sets to be used to compute the maximum likelihood structure
     * @param weightInScoringFunction relative weight for rms dihedral deviation from standard values
     * @param saupe this array holds the computed Syy and Szz values
     * @return the conformation of the secondary structure element in one of the principal order frames
     */
    public myProtein computeAlignmentTensor(final myDipolarCouplingTable thisRdcCsaTable, String mediumName,
            Vector<myDipolarCoupling.Type> typesOfRdcsToBeUsedInTensorRefinement, 
            Vector<myDipolarCoupling.Type> typesOfRdcsToBeUsedForAnalyticSolution, final mySseInfo thisSseId,
            final int refineCycle, final int dfsCycle, final double weightInScoringFunction,
            final double resolution, final boolean printResults, double[] saupe) {
        // Do sanity check and build an ideal SSE model
        if (!testIfRdcsExistForBoundaryResiduesOfSse(thisRdcCsaTable, mediumName, typesOfRdcsToBeUsedInTensorRefinement, thisSseId)) {
            System.out.println("Error: The SSE " + thisSseId.toString() + " to be computed is longer than the available data available for it. Please modify the SSE boundary.");
            System.exit(1);
        }

        // Build an ideal secondary structure element
        //System.out.println("thisSseId.numberOfResidues() = " + thisSseId.numberOfResidues());
        myProtein idealSse = myForKin.buildIdealSSE(thisSseId); //buildIdealSSE(thisSseId);
        System.out.println("Printing Ideal Helix");
        idealSse.print();

        Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMedium = thisRdcCsaTable.getRdcMapForThisMedium(mediumName);
        Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMediumBackCal = new TreeMap<myDipolarCoupling.Type, Vector<myDipolarCoupling>>();
               
        // Compute the alignment tensor and the rotation to one of the POFs        
        Matrix rotToPOF = myAlignmentTensorEstimator.bestFitUsingSVD(idealSse, rdcCsaTableForThisMedium,
                typesOfRdcsToBeUsedInTensorRefinement, rdcCsaTableForThisMediumBackCal, saupe);

//        System.out.println("The back computed RDCs are:");
//        for (Map.Entry<myDipolarCoupling.Type, Vector<myDipolarCoupling>> entry2 : rdcCsaTableForThisMediumBackCal.entrySet()) {
//            System.out.println("The RDC type is: " + entry2.getKey());
//            Vector<myDipolarCoupling> rdcVec = entry2.getValue();
//            System.out.println(rdcVec);
//        }
        
        // Rotate the SSE to the POF
        myProtein thisFragmentInPOF = new myProtein(idealSse);
        thisFragmentInPOF.rotate(new myMatrix(rotToPOF.getArrayCopy()));

//        System.out.println("************");
//        idealSse.print();
//        thisFragmentInPOF.print();
//        System.out.println("************");

        // Get the RDC types that will be used in analytic solutions.
        myDipolarCoupling.Type phiTypeRdc = null;
        for (myDipolarCoupling.Type t : typesOfRdcsToBeUsedForAnalyticSolution) {
            if (myDipolarCoupling.isPhiTypeRdc(t)) {
                phiTypeRdc = t;
                break;
            }
        }
        if (phiTypeRdc == null) {
            System.out.println("Error: There does not exist RDC type that can be used to compute phi using analytic method");
            System.exit(1);
        }

        myDipolarCoupling.Type psiTypeRdc = null;
        for (myDipolarCoupling.Type t : typesOfRdcsToBeUsedForAnalyticSolution) {
            if (myDipolarCoupling.isPsiTypeRdc(t)) {
                psiTypeRdc = t;
                break;
            }
        }
        if (psiTypeRdc == null) {
            System.out.println("Error: There does not exist RDC type that can be used to compute psi using analytic method");
            System.exit(1);
        }
        
        double phiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(idealSse, rdcCsaTableForThisMedium.get(phiTypeRdc),
            rotToPOF, -(saupe[0] + saupe[1]), saupe[0], saupe[1], myConstantsForRdcs.DmaxScaled.get(phiTypeRdc.toString()));
        System.out.println("Phi type rdc rmsd: " + phiTypeRdcRmsd);

        double psiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(idealSse, rdcCsaTableForThisMedium.get(psiTypeRdc),
            rotToPOF, -(saupe[0] + saupe[1]), saupe[0], saupe[1], myConstantsForRdcs.DmaxScaled.get(psiTypeRdc.toString()));
        System.out.println("Psi type rdc rmsd: " + psiTypeRdcRmsd);

        Writer outputSaupe = null;
        File saupeListFile = new File("saupeList.txt"); // Note that this file holds the emerging pattern of AT and finally has to be parameterized.
        if (saupeListFile.exists()) {
            saupeListFile.delete();
        }

        myProtein sseUsingRefinedTensor = null;

        try {
            outputSaupe = new BufferedWriter(new FileWriter(saupeListFile));
            System.out.println("Alignment Tensor after Round 0:    Syy: " + saupe[0] + "    Szz: " + saupe[1]);
            outputSaupe.write("Alignment Tensor after Round 0:    Syy: " + saupe[0] + "    Szz: " + saupe[1] + '\n');
            System.out.println("Rhombicity = " + computeRhombicity(saupe[0], saupe[1]));
            outputSaupe.write("Rhombicity = " + computeRhombicity(saupe[0], saupe[1]) + '\n');
            outputSaupe.close();

            for (int iter = 1; iter <= refineCycle; iter++) {
                System.out.println("Number of DFS cycles (= #Sampled RDC Sets): " + dfsCycle);

                //phiTypeRdcRmsd = sandwich(0.05, 0.25, phiTypeRdcRmsd);
                //psiTypeRdcRmsd = sandwich(0.05, 1.0, psiTypeRdcRmsd);

                //phiTypeRdcRmsd = (phiTypeRdcRmsd > 3.0)? 2.0 : phiTypeRdcRmsd; // TODO: JAN28
				//psiTypeRdcRmsd = (psiTypeRdcRmsd > 3.0)? 2.0 : psiTypeRdcRmsd; // TODO: JAN28
				
		System.out.println("Intervals set correctly");

                if (myMiscConstants.enforceRdcErrorIntervalEarly == true) {
                    if (iter > 0) {
                        phiTypeRdcRmsd = sandwich(0.1 * myConstantsForRdcs.DmaxScaled.get(phiTypeRdc.toString()),
                                2.0 * myMiscConstants.phiScale * myConstantsForRdcs.DmaxScaled.get(phiTypeRdc.toString()), phiTypeRdcRmsd);
                        psiTypeRdcRmsd = sandwich(0.1 * myConstantsForRdcs.DmaxScaled.get(psiTypeRdc.toString()),
                                2.0 * myMiscConstants.psiScale * myConstantsForRdcs.DmaxScaled.get(psiTypeRdc.toString()), psiTypeRdcRmsd);
                    }
                } else {
                    if (iter > 1) {
                        phiTypeRdcRmsd = sandwich(0.1 * myConstantsForRdcs.DmaxScaled.get(phiTypeRdc.toString()),
                                2.0 * myMiscConstants.phiScale * myConstantsForRdcs.DmaxScaled.get(phiTypeRdc.toString()), phiTypeRdcRmsd);
                        psiTypeRdcRmsd = sandwich(0.1 * myConstantsForRdcs.DmaxScaled.get(psiTypeRdc.toString()),
                                2.0 * myMiscConstants.psiScale * myConstantsForRdcs.DmaxScaled.get(psiTypeRdc.toString()), psiTypeRdcRmsd);
                    }
                }
                
                //phiTypeRdcRmsd = sandwich(0.05, 0.5, phiTypeRdcRmsd);
                //psiTypeRdcRmsd = sandwich(0.05, 1.5, psiTypeRdcRmsd);                
                
                System.out.println(phiTypeRdc.toString() + " RDC RMSD threshold: " + phiTypeRdcRmsd + "    " + psiTypeRdc.toString() + " RDC RMSD threshold: " + psiTypeRdcRmsd);

                myResidue firstResidue = thisFragmentInPOF.residueAt(thisFragmentInPOF.getBeginResidueNumber());
                myPeptidePlane pPlane = new myPeptidePlane(firstResidue.getAtom(myAtomLabel.__N).getCoordinates(),
                        firstResidue.getAtom(myAtomLabel.__H).getCoordinates(), firstResidue.getAtom(myAtomLabel.__CA).getCoordinates());

                sseUsingRefinedTensor = computeMaximumlikelihoodFragment(rdcCsaTableForThisMedium, rdcCsaTableForThisMediumBackCal,
                        typesOfRdcsToBeUsedForAnalyticSolution, thisSseId, pPlane, -(saupe[0] + saupe[1]), saupe[0], saupe[1],
                        phiTypeRdcRmsd, psiTypeRdcRmsd, dfsCycle, weightInScoringFunction, printResults);

                if (sseUsingRefinedTensor == null) {
                    System.out.println("Error: sseUsingRefinedTensor is null in refineAlignmentTensor");
                    System.exit(1);
                }

                rotToPOF = myAlignmentTensorEstimator.bestFitUsingSVD(sseUsingRefinedTensor, rdcCsaTableForThisMedium,
                        typesOfRdcsToBeUsedInTensorRefinement, rdcCsaTableForThisMediumBackCal, saupe);
                
                phiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(sseUsingRefinedTensor, rdcCsaTableForThisMedium.get(phiTypeRdc),
                        rotToPOF, -(saupe[0] + saupe[1]), saupe[0], saupe[1], myConstantsForRdcs.DmaxScaled.get(phiTypeRdc.toString()));
                System.out.println("Phi type rdc rmsd: " + phiTypeRdcRmsd);

                psiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(sseUsingRefinedTensor, rdcCsaTableForThisMedium.get(psiTypeRdc),
                        rotToPOF, -(saupe[0] + saupe[1]), saupe[0], saupe[1], myConstantsForRdcs.DmaxScaled.get(psiTypeRdc.toString()));
                System.out.println("Psi type rdc rmsd: " + psiTypeRdcRmsd);

                thisFragmentInPOF = new myProtein(sseUsingRefinedTensor);
                thisFragmentInPOF.rotate(new myMatrix(rotToPOF.getArrayCopy()));

                System.out.println("Printing the best fragment after rotation");
                thisFragmentInPOF.print();

                outputSaupe = new BufferedWriter(new FileWriter(saupeListFile, true));
                System.out.println("Alignment Tensor after Round " + iter + ":    Syy: " + saupe[0] + "    Szz: " + saupe[1]);
                outputSaupe.write("Alignment Tensor after Round " + iter + ":    Syy: " + saupe[0] + "    Szz: " + saupe[1] + '\n');
                System.out.println("Rhombicity = " + computeRhombicity(saupe[0], saupe[1]));
                outputSaupe.write("Rhombicity = " + computeRhombicity(saupe[0], saupe[1]) + '\n');
                outputSaupe.close();
            }
        } catch (IOException e) {
            e.printStackTrace();
        }

        //System.out.println("Exiting from computeAlignmentTensor...");
        System.out.println("Printing best fragment BEFORE rotation to POF");
        sseUsingRefinedTensor.print();
        System.out.println("Printing best fragment AFTER rotation to POF");
        thisFragmentInPOF.print();

        return thisFragmentInPOF;
        //return pdbVecUsingRefinedTensor;
    }

    /**
     * Given three doubles lb, ub, val, this method tests if val is between lb 
     * and ub. If so, then it returns val, else it returns either lb or ub
     * depending on which one is closer to val.
     *
     * @param lb the lower bound of the interval
     * @param ub the upper bound of the interval
     * @param val the number being tested for containment in the interval
     * @return if val is between lb and ub, then it returns val, else return
     * either lb or ub depending on which one is closer to val
     */
    private double sandwich(double lb, double ub, double val) {
        if (lb > ub) { // swap
            double temp = lb;
            lb = ub;
            ub = temp;
        }
        if (val < lb) {
            val = lb;
        } else if (val > ub) {
            val = ub;
        } else ; // do nothing
        return val;
    }

    /**
     * Compute the secondary structure element given the alignment tensor and RDCs.
     *
     * @param thisRdcCsaTable the table containing all RDCs
     * @param mediumName specification of the medium
     * @param typesOfRdcsToBeUsedInTensorRefinement specify the types of RDCs to be used for alignment tensor computation
     * @param typesOfRdcsToBeUsedForAnalyticSolution speficy the types of RDCs to be used for constructing the secondary structure element
     * @param thisSseId specification of the secondary structure element to be computed
     * @param Syy the diagonalized alignment tensor component
     * @param Szz the diagonalized alignment tensor component
     * @param refineCycle the number of iterations to be done during the structure computation
     * @param dfsCycle the number of search trees to be enumerated, i.e., the number of sampled RDC sets to be used to compute the maximum likelihood structure
     * @param weightInScoringFunction relative weight for rms dihedral deviation from standard values
     * @param resolution the angular resolution for grid search to estimate the best-fit initial peptide plane
     * @param printResults
     * @return the maximum likelihood protein fragment in one of the POFs computed over all sampled sets of RDCs
     */
    public myProtein computeSseGivenAlignmentTensorAndRdcs(final myDipolarCouplingTable thisRdcCsaTable, String mediumName,
            Vector<myDipolarCoupling.Type> typesOfRdcsToBeUsedInTensorRefinement,
            Vector<myDipolarCoupling.Type> typesOfRdcsToBeUsedForAnalyticSolution,
            final mySseInfo thisSseId, final double Syy, final double Szz,
            final int refineCycle, final int dfsCycle, final double weightInScoringFunction, final double resolution, final boolean printResults) {
        // do sanity check and build an ideal SSE model
        if (!testIfRdcsExistForBoundaryResiduesOfSse(thisRdcCsaTable, mediumName, typesOfRdcsToBeUsedInTensorRefinement, thisSseId)) {
            System.out.println("Error: The SSE " + thisSseId.toString() + " to be computed is longer than the available data available for it. Please modify the SSE boundary.");
            System.exit(1);
        }

        // Build an ideal secondary structure element
        myProtein idealSse = myForKin.buildIdealSSE(thisSseId);

        Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMedium = thisRdcCsaTable.getRdcMapForThisMedium(mediumName);
        Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMediumBackCal = new TreeMap<myDipolarCoupling.Type, Vector<myDipolarCoupling>>();

        double[] dummy = new double[2];

        Matrix rotToPOF = myAlignmentTensorEstimator.bestFitUsingGridSearchForGivenAT(idealSse, rdcCsaTableForThisMedium,
                typesOfRdcsToBeUsedInTensorRefinement, rdcCsaTableForThisMediumBackCal, Syy, Szz, resolution);

        // Rotate the SSE to the POF
        myProtein thisFragmentInPOF = new myProtein(idealSse);
        thisFragmentInPOF.rotate(new myMatrix(rotToPOF.getArrayCopy()));
        //System.out.println("************");
        //idealSse.print();
        //pdbVecInPOF.print();
        //System.out.println("************");

        // Now that we are in POF we will optimize the first peptide plane.
        //Note: generalize this for any RDCs
        thisFragmentInPOF = myAlignmentTensorEstimator.optimizeFirstPeptidePlane(thisFragmentInPOF, rdcCsaTableForThisMedium, myDipolarCoupling.Type.N_HN,
                -(Syy + Szz), Syy, Szz, -myMiscConstants.maximumDeviationFromMean, myMiscConstants.maximumDeviationFromMean, myMiscConstants.stepSizeUsedInEstimatingFirstPeptidePlane);

        // Get the RDC types that will be used in analytic solutions.
        myDipolarCoupling.Type phiTypeRdc = null;
        for (myDipolarCoupling.Type t : typesOfRdcsToBeUsedForAnalyticSolution) {
            if (myDipolarCoupling.isPhiTypeRdc(t)) {
                phiTypeRdc = t;
                break;
            }
        }
        if (phiTypeRdc == null) {
            System.out.println("Error: There does not exist RDC type that can be used to compute phi using analytic method");
            System.exit(1);
        }

        myDipolarCoupling.Type psiTypeRdc = null;
        for (myDipolarCoupling.Type t : typesOfRdcsToBeUsedForAnalyticSolution) {
            if (myDipolarCoupling.isPsiTypeRdc(t)) {
                psiTypeRdc = t;
                break;
            }
        }
        if (psiTypeRdc == null) {
            System.out.println("Error: There does not exist RDC type that can be used to compute psi using analytic method");
            System.exit(1);
        }

        // commented out on aug13
        double phiTypeRdcRmsd0 = myAlignmentTensorEstimator.computeRdcRmsd(idealSse, rdcCsaTableForThisMedium.get(phiTypeRdc),
                rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(phiTypeRdc.toString()));
        System.out.println("Phi type rdc rmsd0: " + phiTypeRdcRmsd0);

        // commented out on aug13
        double psiTypeRdcRmsd0 = myAlignmentTensorEstimator.computeRdcRmsd(idealSse, rdcCsaTableForThisMedium.get(psiTypeRdc),
                rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(psiTypeRdc.toString()));
        System.out.println("Psi type rdc rmsd0: " + psiTypeRdcRmsd0);



        double phiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(thisFragmentInPOF, rdcCsaTableForThisMedium.get(phiTypeRdc),
                Matrix.identity(3, 3), -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(phiTypeRdc.toString()));
        System.out.println("Phi type rdc rmsd: " + phiTypeRdcRmsd);

        double psiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(thisFragmentInPOF, rdcCsaTableForThisMedium.get(psiTypeRdc),
                Matrix.identity(3, 3), -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(psiTypeRdc.toString()));
        System.out.println("Psi type rdc rmsd: " + psiTypeRdcRmsd);

        Writer outputSaupe = null;
        File saupeListFile = new File("saupeList.txt"); // Note that this file holds the emerging pattern of AT and finally has to be parameterized.
        if (saupeListFile.exists()) {
            saupeListFile.delete();
        }
        
        myProtein sseUsingRefinedTensor = null;

        try {
            outputSaupe = new BufferedWriter(new FileWriter(saupeListFile));
            System.out.println("Alignment Tensor after Round 0:    Syy: " + Syy + "    Szz: " + Szz);
            outputSaupe.write("Alignment Tensor after Round 0:    Syy: " + Syy + "    Szz: " + Szz + '\n');
            System.out.println("Rhombicity = " + computeRhombicity(Syy, Szz));
            outputSaupe.write("Rhombicity = " + computeRhombicity(Syy, Szz) + '\n');
            outputSaupe.close();

            //System.out.println("refineCycle = " + refineCycle); System.exit(1);

            for (int iter = 1; iter <= refineCycle; iter++) {

                inputChar.readChar();

                System.out.println("Number of DFS cycles (= #Sampled RDC Sets): " + dfsCycle);

                //phiTypeRdcRmsd = (phiTypeRdcRmsd > 4.0)? 2.0 : phiTypeRdcRmsd; // TODO: JAN28
                //psiTypeRdcRmsd = (psiTypeRdcRmsd > 4.0)? 2.0 : psiTypeRdcRmsd; // TODO: JAN28

                if (myMiscConstants.enforceRdcErrorIntervalEarly == true) {
                    if (iter > 0) {
                        phiTypeRdcRmsd = sandwich(0.1 * myConstantsForRdcs.DmaxScaled.get(phiTypeRdc.toString()),
                                2.0 * myMiscConstants.phiScale * myConstantsForRdcs.DmaxScaled.get(phiTypeRdc.toString()), phiTypeRdcRmsd);
                        psiTypeRdcRmsd = sandwich(0.1 * myConstantsForRdcs.DmaxScaled.get(psiTypeRdc.toString()),
                                2.0 * myMiscConstants.psiScale * myConstantsForRdcs.DmaxScaled.get(psiTypeRdc.toString()), psiTypeRdcRmsd);
                    }
                } else {
                    if (iter > 1) {
                        phiTypeRdcRmsd = sandwich(0.1 * myConstantsForRdcs.DmaxScaled.get(phiTypeRdc.toString()),
                                2.0 * myMiscConstants.phiScale * myConstantsForRdcs.DmaxScaled.get(phiTypeRdc.toString()), phiTypeRdcRmsd);
                        psiTypeRdcRmsd = sandwich(0.1 * myConstantsForRdcs.DmaxScaled.get(psiTypeRdc.toString()),
                                2.0 * myMiscConstants.psiScale * myConstantsForRdcs.DmaxScaled.get(psiTypeRdc.toString()), psiTypeRdcRmsd);
                    }
                }


                System.out.println(phiTypeRdc.toString() + " RDC RMSD threshold: " + phiTypeRdcRmsd + "    " + psiTypeRdc.toString() + " RDC RMSD threshold: " + psiTypeRdcRmsd);

                myResidue firstResidue = thisFragmentInPOF.residueAt(thisFragmentInPOF.getBeginResidueNumber());
                myPeptidePlane pPlane = new myPeptidePlane(firstResidue.getAtom(myAtomLabel.__N).getCoordinates(),
                        firstResidue.getAtom(myAtomLabel.__H).getCoordinates(), firstResidue.getAtom(myAtomLabel.__CA).getCoordinates());

                sseUsingRefinedTensor = computeMaximumlikelihoodFragment(rdcCsaTableForThisMedium, rdcCsaTableForThisMediumBackCal,
                        typesOfRdcsToBeUsedForAnalyticSolution, thisSseId, pPlane, -(Syy + Szz), Syy, Szz,
                        phiTypeRdcRmsd, psiTypeRdcRmsd, dfsCycle, weightInScoringFunction, printResults);

                if (sseUsingRefinedTensor == null) {
                    System.out.println("Error: sseUsingRefinedTensor is null in computeSseGivenAlignmentTensorAndRdcs");
                    System.exit(1);
                }
/*
                rotToPOF = myAlignmentTensorEstimator.bestFitUsingSVD(sseUsingRefinedTensor, rdcCsaTableForThisMedium,
                        typesOfRdcsToBeUsedInTensorRefinement, rdcCsaTableForThisMediumBackCal, dummy);
*/
                // Note: This can be removed for longer helix in preference for the above svd
                rotToPOF = myAlignmentTensorEstimator.bestFitUsingGridSearchForGivenAT(sseUsingRefinedTensor, rdcCsaTableForThisMedium,
                        typesOfRdcsToBeUsedInTensorRefinement, rdcCsaTableForThisMediumBackCal, Syy, Szz, resolution);
/*
                phiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(sseUsingRefinedTensor, rdcCsaTableForThisMedium.get(phiTypeRdc),
                        rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(phiTypeRdc.toString()));
                System.out.println("Phi type rdc rmsd: " + phiTypeRdcRmsd);

                psiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(sseUsingRefinedTensor, rdcCsaTableForThisMedium.get(psiTypeRdc),
                        rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(psiTypeRdc.toString()));
                System.out.println("Psi type rdc rmsd: " + psiTypeRdcRmsd);
*/
                thisFragmentInPOF = new myProtein(sseUsingRefinedTensor);
                thisFragmentInPOF.rotate(new myMatrix(rotToPOF.getArrayCopy()));

                thisFragmentInPOF = myAlignmentTensorEstimator.optimizeFirstPeptidePlane(thisFragmentInPOF, rdcCsaTableForThisMedium, myDipolarCoupling.Type.N_HN,
                        -(Syy + Szz), Syy, Szz, -myMiscConstants.maximumDeviationFromMean, myMiscConstants.maximumDeviationFromMean, myMiscConstants.stepSizeUsedInEstimatingFirstPeptidePlane);

                phiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(thisFragmentInPOF, rdcCsaTableForThisMedium.get(phiTypeRdc),
                        /*rotToPOF*/Matrix.identity(3, 3), -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(phiTypeRdc.toString()));
                System.out.println("Phi type rdc rmsd: " + phiTypeRdcRmsd);

                psiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(thisFragmentInPOF, rdcCsaTableForThisMedium.get(psiTypeRdc),
                        /*rotToPOF*/Matrix.identity(3, 3), -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(psiTypeRdc.toString()));
                System.out.println("Psi type rdc rmsd: " + psiTypeRdcRmsd);

                System.out.println("Printing the best fragment after rotation (note: this is not returned, rather the fragment before rotation is returned");
                thisFragmentInPOF.print();

                outputSaupe = new BufferedWriter(new FileWriter(saupeListFile, true));
                System.out.println("Alignment Tensor after Round " + iter + ":    Syy: " + Syy + "    Szz: " + Szz);
                outputSaupe.write("Alignment Tensor after Round " + iter + ":    Syy: " + Syy + "    Szz: " + Szz + '\n');
                System.out.println("Rhombicity = " + computeRhombicity(Syy, Szz));
                outputSaupe.write("Rhombicity = " + computeRhombicity(Syy, Szz) + '\n');
                outputSaupe.close();
            }
        } catch (IOException e) {
            e.printStackTrace();
        }

        System.out.println("Done inside...");
        inputChar.readChar();
        //return thisFragmentInPOF;
        return sseUsingRefinedTensor;
    }


/*************************/

    public myProtein computeSseGivenAlignmentTensorAndRdcs(final Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMedium,
            Vector<myDipolarCoupling.Type> typesOfRdcsToBeUsedInTensorRefinement,
            Vector<myDipolarCoupling.Type> typesOfRdcsToBeUsedForAnalyticSolution,
            final mySseInfo thisSseId, final double Syy, final double Szz,
            final int refineCycle, final int dfsCycle, final double weightInScoringFunction, final double resolution, final boolean printResults) {
        // do sanity check and build an ideal SSE model
//D11        if (!testIfRdcsExistForBoundaryResiduesOfSse(thisRdcCsaTable, mediumName, typesOfRdcsToBeUsedInTensorRefinement, thisSseId)) {
//            System.out.println("Error: The SSE " + thisSseId.toString() + " to be computed is longer than the available data available for it. Please modify the SSE boundary.");
//            System.exit(1);
//        }

        // Build an ideal secondary structure element
        myProtein idealSse = myForKin.buildIdealSSE(thisSseId);

//D11        Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMedium = thisRdcCsaTable.getRdcMapForThisMedium(mediumName);
        Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMediumBackCal = new TreeMap<myDipolarCoupling.Type, Vector<myDipolarCoupling>>();

        double[] dummy = new double[2];

        Matrix rotToPOF = myAlignmentTensorEstimator.bestFitUsingGridSearchForGivenAT(idealSse, rdcCsaTableForThisMedium,
                typesOfRdcsToBeUsedInTensorRefinement, rdcCsaTableForThisMediumBackCal, Syy, Szz, resolution);

        // Rotate the SSE to the POF
        myProtein thisFragmentInPOF = new myProtein(idealSse);
        thisFragmentInPOF.rotate(new myMatrix(rotToPOF.getArrayCopy()));
        //System.out.println("************");
        //idealSse.print();
        //pdbVecInPOF.print();
        //System.out.println("************");

        // Now that we are in POF we will optimize the first peptide plane.
        //Note: generalize this for any RDCs
        thisFragmentInPOF = myAlignmentTensorEstimator.optimizeFirstPeptidePlane(thisFragmentInPOF, rdcCsaTableForThisMedium, myDipolarCoupling.Type.N_HN,
                -(Syy + Szz), Syy, Szz, -myMiscConstants.maximumDeviationFromMean, myMiscConstants.maximumDeviationFromMean, myMiscConstants.stepSizeUsedInEstimatingFirstPeptidePlane);

        // Get the RDC types that will be used in analytic solutions.
        myDipolarCoupling.Type phiTypeRdc = null;
        for (myDipolarCoupling.Type t : typesOfRdcsToBeUsedForAnalyticSolution) {
            if (myDipolarCoupling.isPhiTypeRdc(t)) {
                phiTypeRdc = t;
                break;
            }
        }
        if (phiTypeRdc == null) {
            System.out.println("Error: There does not exist RDC type that can be used to compute phi using analytic method");
            System.exit(1);
        }

        myDipolarCoupling.Type psiTypeRdc = null;
        for (myDipolarCoupling.Type t : typesOfRdcsToBeUsedForAnalyticSolution) {
            if (myDipolarCoupling.isPsiTypeRdc(t)) {
                psiTypeRdc = t;
                break;
            }
        }
        if (psiTypeRdc == null) {
            System.out.println("Error: There does not exist RDC type that can be used to compute psi using analytic method");
            System.exit(1);
        }

        // commented out on aug13
        double phiTypeRdcRmsd0 = myAlignmentTensorEstimator.computeRdcRmsd(idealSse, rdcCsaTableForThisMedium.get(phiTypeRdc),
                rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(phiTypeRdc.toString()));
        System.out.println("Phi type rdc rmsd0: " + phiTypeRdcRmsd0);

        // commented out on aug13
        double psiTypeRdcRmsd0 = myAlignmentTensorEstimator.computeRdcRmsd(idealSse, rdcCsaTableForThisMedium.get(psiTypeRdc),
                rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(psiTypeRdc.toString()));
        System.out.println("Psi type rdc rmsd0: " + psiTypeRdcRmsd0);



        double phiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(thisFragmentInPOF, rdcCsaTableForThisMedium.get(phiTypeRdc),
                Matrix.identity(3, 3), -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(phiTypeRdc.toString()));
        System.out.println("Phi type rdc rmsd: " + phiTypeRdcRmsd);

        double psiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(thisFragmentInPOF, rdcCsaTableForThisMedium.get(psiTypeRdc),
                Matrix.identity(3, 3), -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(psiTypeRdc.toString()));
        System.out.println("Psi type rdc rmsd: " + psiTypeRdcRmsd);

        Writer outputSaupe = null;
        File saupeListFile = new File("saupeList.txt"); // Note that this file holds the emerging pattern of AT and finally has to be parameterized.
        if (saupeListFile.exists()) {
            saupeListFile.delete();
        }

        myProtein sseUsingRefinedTensor = null;

        try {
            outputSaupe = new BufferedWriter(new FileWriter(saupeListFile));
            System.out.println("Alignment Tensor after Round 0:    Syy: " + Syy + "    Szz: " + Szz);
            outputSaupe.write("Alignment Tensor after Round 0:    Syy: " + Syy + "    Szz: " + Szz + '\n');
            System.out.println("Rhombicity = " + computeRhombicity(Syy, Szz));
            outputSaupe.write("Rhombicity = " + computeRhombicity(Syy, Szz) + '\n');
            outputSaupe.close();

            //System.out.println("refineCycle = " + refineCycle); System.exit(1);

            for (int iter = 1; iter <= refineCycle; iter++) {

                inputChar.readChar();

                System.out.println("Number of DFS cycles (= #Sampled RDC Sets): " + dfsCycle);

                //phiTypeRdcRmsd = sandwich(0.05, 1.5, phiTypeRdcRmsd);
                //psiTypeRdcRmsd = sandwich(0.05, 1.0, psiTypeRdcRmsd);

                if (iter > 1) {
                    //phiTypeRdcRmsd = sandwich(0.05, 1.5, phiTypeRdcRmsd);
                    //psiTypeRdcRmsd = sandwich(0.05, 1.0, psiTypeRdcRmsd);
                    phiTypeRdcRmsd = sandwich(0.1 * myConstantsForRdcs.DmaxScaled.get(phiTypeRdc.toString()),
                            2.0 * myMiscConstants.phiScale * myConstantsForRdcs.DmaxScaled.get(phiTypeRdc.toString()), phiTypeRdcRmsd);
                    psiTypeRdcRmsd = sandwich(0.1 * myConstantsForRdcs.DmaxScaled.get(psiTypeRdc.toString()),
                            2.0 * myMiscConstants.psiScale * myConstantsForRdcs.DmaxScaled.get(psiTypeRdc.toString()), psiTypeRdcRmsd);
                }

                System.out.println(phiTypeRdc.toString() + " RDC RMSD threshold: " + phiTypeRdcRmsd + "    " + psiTypeRdc.toString() + " RDC RMSD threshold: " + psiTypeRdcRmsd);

                myResidue firstResidue = thisFragmentInPOF.residueAt(thisFragmentInPOF.getBeginResidueNumber());
                myPeptidePlane pPlane = new myPeptidePlane(firstResidue.getAtom(myAtomLabel.__N).getCoordinates(),
                        firstResidue.getAtom(myAtomLabel.__H).getCoordinates(), firstResidue.getAtom(myAtomLabel.__CA).getCoordinates());

                sseUsingRefinedTensor = computeMaximumlikelihoodFragment(rdcCsaTableForThisMedium, rdcCsaTableForThisMediumBackCal,
                        typesOfRdcsToBeUsedForAnalyticSolution, thisSseId, pPlane, -(Syy + Szz), Syy, Szz,
                        phiTypeRdcRmsd, psiTypeRdcRmsd, dfsCycle, weightInScoringFunction, printResults);

                if (sseUsingRefinedTensor == null) {
                    System.out.println("Error: sseUsingRefinedTensor is null in computeSseGivenAlignmentTensorAndRdcs");
                    System.exit(1);
                }
/*
                rotToPOF = myAlignmentTensorEstimator.bestFitUsingSVD(sseUsingRefinedTensor, rdcCsaTableForThisMedium,
                        typesOfRdcsToBeUsedInTensorRefinement, rdcCsaTableForThisMediumBackCal, dummy);
*/
                // Note: This can be removed for longer helix in preference for the above svd
                rotToPOF = myAlignmentTensorEstimator.bestFitUsingGridSearchForGivenAT(sseUsingRefinedTensor, rdcCsaTableForThisMedium,
                        typesOfRdcsToBeUsedInTensorRefinement, rdcCsaTableForThisMediumBackCal, Syy, Szz, resolution);
/*
                phiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(sseUsingRefinedTensor, rdcCsaTableForThisMedium.get(phiTypeRdc),
                        rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(phiTypeRdc.toString()));
                System.out.println("Phi type rdc rmsd: " + phiTypeRdcRmsd);

                psiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(sseUsingRefinedTensor, rdcCsaTableForThisMedium.get(psiTypeRdc),
                        rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(psiTypeRdc.toString()));
                System.out.println("Psi type rdc rmsd: " + psiTypeRdcRmsd);
*/
                thisFragmentInPOF = new myProtein(sseUsingRefinedTensor);
                thisFragmentInPOF.rotate(new myMatrix(rotToPOF.getArrayCopy()));

                thisFragmentInPOF = myAlignmentTensorEstimator.optimizeFirstPeptidePlane(thisFragmentInPOF, rdcCsaTableForThisMedium, myDipolarCoupling.Type.N_HN,
                        -(Syy + Szz), Syy, Szz, -myMiscConstants.maximumDeviationFromMean, myMiscConstants.maximumDeviationFromMean, myMiscConstants.stepSizeUsedInEstimatingFirstPeptidePlane);

                phiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(thisFragmentInPOF, rdcCsaTableForThisMedium.get(phiTypeRdc),
                        /*rotToPOF*/Matrix.identity(3, 3), -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(phiTypeRdc.toString()));
                System.out.println("Phi type rdc rmsd: " + phiTypeRdcRmsd);

                psiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(thisFragmentInPOF, rdcCsaTableForThisMedium.get(psiTypeRdc),
                        /*rotToPOF*/Matrix.identity(3, 3), -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(psiTypeRdc.toString()));
                System.out.println("Psi type rdc rmsd: " + psiTypeRdcRmsd);

                System.out.println("Printing the best fragment after rotation (note: this is not returned, rather the fragment before rotation is returned");
                thisFragmentInPOF.print();

                outputSaupe = new BufferedWriter(new FileWriter(saupeListFile, true));
                System.out.println("Alignment Tensor after Round " + iter + ":    Syy: " + Syy + "    Szz: " + Szz);
                outputSaupe.write("Alignment Tensor after Round " + iter + ":    Syy: " + Syy + "    Szz: " + Szz + '\n');
                System.out.println("Rhombicity = " + computeRhombicity(Syy, Szz));
                outputSaupe.write("Rhombicity = " + computeRhombicity(Syy, Szz) + '\n');
                outputSaupe.close();
            }
        } catch (IOException e) {
            e.printStackTrace();
        }

        System.out.println("Done inside...");
        inputChar.readChar();
        //return thisFragmentInPOF;
        return sseUsingRefinedTensor;
    }




/*
    public myPair<myProtein, myProtein> getBestDirectionalPair(myProtein p1, myProtein p2, Vector<myNoe> noeVec) {
        Vector<myProtein> fourFoldFragment1 = myNewPacker.getFourFolds(p1);
        Vector<myProtein> fourFoldFragment2 = myNewPacker.getFourFolds(p2);

        Vector<myProtein> packedFragments = new Vector<myProtein>();

        // for each of the 16 pairs try to pack them and report the noe rmsd
        for (myProtein f1 : fourFoldFragment1) {
            for (myProtein f2 : fourFoldFragment2) {
                Vector<myProtein> f1f2 = new Vector<myProtein>();
                f1f2.add(f1);
                f1f2.add(f2);
                int[] packingOrder = new int[]{0,1};
                myPacker packer = new myPacker();
                myProtein packedf1f2 = packer.packAllFragmentsUsingDistanceRestraints(f1f2, packingOrder, noeVec);

                System.out.print("The noe rmsds seen so far: ");
                for (Double d : __noe_rmsds) {
                    System.out.print(d + "  ");
                }
                System.out.println();

                System.out.println("This packed Fragment is:");
                packedf1f2.print();
                packedFragments.add(packedf1f2);
            }
        }
    }
*/

    /**
     * Compute a given strand and add to the beta-sheet to which it belongs.
     *
     * @param thisRdcCsaTable the table containing all RDCs
     * @param mediumName specification of the medium
     * @param typesOfRdcsToBeUsedInTensorRefinement specify the types of RDCs to be used for alignment tensor computation
     * @param typesOfRdcsToBeUsedForAnalyticSolution speficy the types of RDCs to be used for constructing the secondary structure element
     * @param thisSseId the beta-strand boundary specification
     * @param Syy the diagonalized alignment tensor component
     * @param Szz the diagonalized alignment tensor component
     * @param refineCycle the number of iterations to be done during the structure computation
     * @param dfsCycle the number of search trees to be enumerated, i.e., the number of sampled RDC sets to be used to compute the maximum likelihood structure
     * @param weightInScoringFunction relative weight for rms dihedral deviation from standard values
     * @param resolution the angular resolution for grid search to estimate the best-fit initial peptide plane
     * @param printResults
     * @param hBondWeight
     * @param noeVec the NOE/hbond table
     * @param partialSheetThusFar the partial sheet formed thus far
     * @param parallel true if the strand to be computed is parallel to the neighboring strand already computed
     * @param prevStrandSpec specification of the neighboring strand already computed
     * @return the beta-sheet grown by one more strand
     */
    public myProtein computeThisStrandAndAddItToSheet(final myDipolarCouplingTable thisRdcCsaTable, String mediumName,
            Vector<myDipolarCoupling.Type> typesOfRdcsToBeUsedInTensorRefinement,
            Vector<myDipolarCoupling.Type> typesOfRdcsToBeUsedForAnalyticSolution,
            mySseInfo thisSseId, final double Syy, final double Szz,
            final int refineCycle, final int dfsCycle, final double weightInScoringFunction, final double resolution, final boolean printResults,
            final double hBondWeight, Vector<myNoe> noeVec, myProtein partialSheetThusFar, boolean parallel, mySseInfo prevStrandSpec) {

        myProtein workingPartialSheet = new myProtein(partialSheetThusFar);

        // do sanity check and build an ideal SSE model
        if (!testIfRdcsExistForBoundaryResiduesOfSse(thisRdcCsaTable, mediumName, typesOfRdcsToBeUsedInTensorRefinement, thisSseId)) {
            System.out.println("Error: The SSE " + thisSseId.toString() + " to be computed is longer than the available data available for it. Please modify the SSE boundary.");
            System.exit(1);
        }        

        // Build an ideal secondary structure element
        myProtein idealSse = myForKin.buildIdealSSE(thisSseId); 

        Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMedium = thisRdcCsaTable.getRdcMapForThisMedium(mediumName);
        Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMediumBackCal = new TreeMap<myDipolarCoupling.Type, Vector<myDipolarCoupling>>();

        double[] dummy = new double[2];

        //System.out.println("?????????????Note: warning! Incorporate HScore in the bestFit function"); // already done
        /*Matrix rotToPOF*/
        myTriple<myProtein, myProtein, Matrix> protProtMatTriple =
                myAlignmentTensorEstimator.bestFitUsingGridSearchForGivenATAndNoes__(idealSse, rdcCsaTableForThisMedium,
                typesOfRdcsToBeUsedInTensorRefinement, rdcCsaTableForThisMediumBackCal, Syy, Szz, resolution, hBondWeight, 
                noeVec, workingPartialSheet, parallel, prevStrandSpec);
        
        myProtein bestWorkingPartialSheet = protProtMatTriple.first();
        myProtein bestPackedFragment = protProtMatTriple.second();
        Matrix rotToPOF = protProtMatTriple.third();
        
        // Rotate the SSE to the POF
        myProtein thisFragmentInPOF = bestPackedFragment; //new myProtein(idealSse);  
        //thisFragmentInPOF.rotate(new myMatrix(rotToPOF.getArrayCopy())); 
        //System.out.println("************");
        //idealSse.print();
        //pdbVecInPOF.print();
        //System.out.println("************");

        // Now that we are in POF we will optimize the first peptide plane.
        //Note: generalize this for any RDCs
        thisFragmentInPOF = myAlignmentTensorEstimator.optimizeFirstPeptidePlane(thisFragmentInPOF, rdcCsaTableForThisMedium, myDipolarCoupling.Type.N_HN,
                -(Syy + Szz), Syy, Szz, -myMiscConstants.maximumDeviationFromMean, myMiscConstants.maximumDeviationFromMean, myMiscConstants.stepSizeUsedInEstimatingFirstPeptidePlane);

        // Get the RDC types that will be used in analytic solutions.
        myDipolarCoupling.Type phiTypeRdc = null;
        for (myDipolarCoupling.Type t : typesOfRdcsToBeUsedForAnalyticSolution) {
            if (myDipolarCoupling.isPhiTypeRdc(t)) {
                phiTypeRdc = t;
                break;
            }
        }
        if (phiTypeRdc == null) {
            System.out.println("Error: There does not exist RDC type that can be used to compute phi using analytic method");
            System.exit(1);
        }

        myDipolarCoupling.Type psiTypeRdc = null;
        for (myDipolarCoupling.Type t : typesOfRdcsToBeUsedForAnalyticSolution) {
            if (myDipolarCoupling.isPsiTypeRdc(t)) {
                psiTypeRdc = t;
                break;
            }
        }
        if (psiTypeRdc == null) {
            System.out.println("Error: There does not exist RDC type that can be used to compute psi using analytic method");
            System.exit(1);
        }

        double phiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(thisFragmentInPOF, rdcCsaTableForThisMedium.get(phiTypeRdc),
                Matrix.identity(3, 3), -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(phiTypeRdc.toString()));
        System.out.println("Phi type rdc rmsd: " + phiTypeRdcRmsd);

        double psiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(thisFragmentInPOF, rdcCsaTableForThisMedium.get(psiTypeRdc),
                Matrix.identity(3, 3), -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(psiTypeRdc.toString()));
        System.out.println("Psi type rdc rmsd: " + psiTypeRdcRmsd);

        Writer outputSaupe = null;
        File saupeListFile = new File("saupeList.txt"); // this file holds the emerging pattern of AT and finally has to be parameterized
        if (saupeListFile.exists()) {
            saupeListFile.delete();
        }

        myProtein sseUsingRefinedTensor = null;

        try {
            outputSaupe = new BufferedWriter(new FileWriter(saupeListFile));
            System.out.println("Alignment Tensor after Round 0:    Syy: " + Syy + "    Szz: " + Szz);
            outputSaupe.write("Alignment Tensor after Round 0:    Syy: " + Syy + "    Szz: " + Szz + '\n');
            System.out.println("Rhombicity = " + computeRhombicity(Syy, Szz));
            outputSaupe.write("Rhombicity = " + computeRhombicity(Syy, Szz) + '\n');
            outputSaupe.close();

            //System.out.println("refineCycle = " + refineCycle); System.exit(1);

            for (int iter = 1; iter <= refineCycle; iter++) {

                inputChar.readChar();

                System.out.println("Number of DFS cycles (= #Sampled RDC Sets): " + dfsCycle);

                //phiTypeRdcRmsd = sandwich(0.05, 1.5, phiTypeRdcRmsd);
                //psiTypeRdcRmsd = sandwich(0.05, 1.0, psiTypeRdcRmsd);
                
                phiTypeRdcRmsd = sandwich(0.1 * myConstantsForRdcs.DmaxScaled.get(phiTypeRdc.toString()),
                        2.0 * myMiscConstants.phiScale * myConstantsForRdcs.DmaxScaled.get(phiTypeRdc.toString()), phiTypeRdcRmsd);
                psiTypeRdcRmsd = sandwich(0.1 * myConstantsForRdcs.DmaxScaled.get(psiTypeRdc.toString()),
                        2.0 * myMiscConstants.psiScale * myConstantsForRdcs.DmaxScaled.get(psiTypeRdc.toString()), psiTypeRdcRmsd);

                System.out.println(phiTypeRdc.toString() + " RDC RMSD threshold: " + phiTypeRdcRmsd + "    " + psiTypeRdc.toString() + " RDC RMSD threshold: " + psiTypeRdcRmsd);

                //make sure that you are aligned properly doing 4*4 fold degeneracy test.
                //System.out.println("The things below need to be designed-----------------");
                //myPair<myProtein, myProtein> bestDirectionalPair = getBestDirectionalPair(workingPartialSheet, thisFragmentInPOF, noeVec);
                System.out.println("Iteration Number (while computing second strand onwards) is: " + iter);
                inputChar.readChar();
                //System.exit(1);

                myResidue firstResidue = thisFragmentInPOF.residueAt(thisFragmentInPOF.getBeginResidueNumber());

                myPeptidePlane pPlane = new myPeptidePlane(firstResidue.getAtom(myAtomLabel.__N).getCoordinates(),
                        firstResidue.getAtom(myAtomLabel.__H).getCoordinates(), firstResidue.getAtom(myAtomLabel.__CA).getCoordinates());

                sseUsingRefinedTensor = computeMaximumlikelihoodBetaStrand(rdcCsaTableForThisMedium, rdcCsaTableForThisMediumBackCal,
                        typesOfRdcsToBeUsedForAnalyticSolution, thisSseId, pPlane, -(Syy + Szz), Syy, Szz,
                        phiTypeRdcRmsd, psiTypeRdcRmsd, dfsCycle, weightInScoringFunction, printResults,
                        hBondWeight, noeVec, bestWorkingPartialSheet);

                System.out.println("Note: We returned from computeMaximumlikelihoodBetaStrand__");

                if (sseUsingRefinedTensor == null) {
                    System.out.println("Error: sseUsingRefinedTensor is null in computeMaximumlikelihoodBetaStrand__");
                    System.exit(1);
                }
/*
                rotToPOF = myAlignmentTensorEstimator.bestFitUsingSVD(sseUsingRefinedTensor, rdcCsaTableForThisMedium,
                        typesOfRdcsToBeUsedInTensorRefinement, rdcCsaTableForThisMediumBackCal, dummy);
*/
                // we can invoke packer again to estimate the first pp once again, not done for efficiency

                rotToPOF = Matrix.identity(3, 3); // since we are not changing the orientation of the first pp

                thisFragmentInPOF = new myProtein(sseUsingRefinedTensor);
                //thisFragmentInPOF.rotate(new myMatrix(rotToPOF.getArrayCopy())); // commented out as the matrix is identity

                phiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(thisFragmentInPOF, rdcCsaTableForThisMedium.get(phiTypeRdc),
                        /*rotToPOF*/Matrix.identity(3, 3), -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(phiTypeRdc.toString()));
                System.out.println("Phi type rdc rmsd: " + phiTypeRdcRmsd);

                psiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(thisFragmentInPOF, rdcCsaTableForThisMedium.get(psiTypeRdc),
                        /*rotToPOF*/Matrix.identity(3, 3), -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(psiTypeRdc.toString()));
                System.out.println("Psi type rdc rmsd: " + psiTypeRdcRmsd);

                System.out.println("Printing the best fragment after rotation (for now rotation matrix in identity!!");
                thisFragmentInPOF.print();

                outputSaupe = new BufferedWriter(new FileWriter(saupeListFile, true));
                System.out.println("Alignment Tensor after Round " + iter + ":    Syy: " + Syy + "    Szz: " + Szz);
                outputSaupe.write("Alignment Tensor after Round " + iter + ":    Syy: " + Syy + "    Szz: " + Szz + '\n');
                System.out.println("Rhombicity = " + computeRhombicity(Syy, Szz));
                outputSaupe.write("Rhombicity = " + computeRhombicity(Syy, Szz) + '\n');
                outputSaupe.close();

                System.out.println("Completing computeThisStrandAndAddItToSheet's iteration# " + iter);
                inputChar.readChar();
                //System.exit(1);
            }
        } catch (IOException e) {
            e.printStackTrace();
        }

        System.out.println("Done inside computing second (or third etc.) strand");
        //System.out.println("Warning: Inside computeThisStrandAndAddItToSheet__() the return value has to be fixed, for example see computeSseGivenAlignmentTensorAndRdcs__()");
        inputChar.readChar();
        
        //return sseUsingRefinedTensor;
        myProtein mergedProtein = myProtein.cloneAndMergeTwoProteinFragments(bestWorkingPartialSheet, sseUsingRefinedTensor);
        return mergedProtein;
    }

    /**
     * Compute a beta-sheet given the strand specifications for the strands belonging to the beta-sheet.
     *
     * @param thisRdcCsaTable the table containing all RDCs
     * @param mediumName specification of the medium
     * @param typesOfRdcsToBeUsedInTensorRefinement specify the types of RDCs to be used for alignment tensor computation
     * @param typesOfRdcsToBeUsedForAnalyticSolution speficy the types of RDCs to be used for constructing the secondary structure element
     * @param Syy the diagonalized alignment tensor component
     * @param Szz the diagonalized alignment tensor component
     * @param refineCycle the number of iterations to be done during the structure computation
     * @param dfsCycle the number of search trees to be enumerated, i.e., the number of sampled RDC sets to be used to compute the maximum likelihood structure
     * @param weightInScoringFunction relative weight for rms dihedral deviation from standard values
     * @param resolution the angular resolution for grid search to estimate the best-fit initial peptide plane
     * @param printResults
     * @param hBondWeight
     * @param hBondVec the NOE/hbond table
     * @param strandAdjInfo the specification of the beta-sheet (strand adjacency, relative direction and boundary)
     * @return the beta-sheet conformation in one of the principal order frames
     */
    public myProtein computeBetaSheetGivenAlignmentTensorAndRdcs(final myDipolarCouplingTable thisRdcCsaTable, String mediumName,
            Vector<myDipolarCoupling.Type> typesOfRdcsToBeUsedInTensorRefinement,
            Vector<myDipolarCoupling.Type> typesOfRdcsToBeUsedForAnalyticSolution,
            /*Vector<mySseInfo> thisBetaSheetSseIds,*/ final double Syy, final double Szz,
            final int refineCycle, final int dfsCycle, final double weightInScoringFunction, final double resolution, final boolean printResults,
            final double hBondWeight, Vector<myNoe> hBondVec, Vector<myTriple<mySseInfo, mySseInfo, Boolean>> strandAdjInfo) {
            //System.out.println("I am inside computeBetaSheetGivenAlignmentTensorAndRdcs...");

        // do sanity check and build an ideal SSE model
        for (myTriple<mySseInfo, mySseInfo, Boolean> thisStrandPair : strandAdjInfo) {
            mySseInfo firstStrandId = thisStrandPair.first();
            mySseInfo secondStrandId = thisStrandPair.second();

            if (!testIfRdcsExistForBoundaryResiduesOfSse(thisRdcCsaTable, mediumName, typesOfRdcsToBeUsedInTensorRefinement, firstStrandId)) {
                System.out.println("Error: The strand " + firstStrandId.toString() + " to be computed is longer than the available data available for it. Please modify the strand boundary.");
                System.exit(1);
            }

            if (!testIfRdcsExistForBoundaryResiduesOfSse(thisRdcCsaTable, mediumName, typesOfRdcsToBeUsedInTensorRefinement, secondStrandId)) {
                System.out.println("Error: The strand " + secondStrandId.toString() + " to be computed is longer than the available data available for it. Please modify the strand boundary.");
                System.exit(1);
            }
        }

        // Compute the first strand
        myProtein firstStrand = computeSseGivenAlignmentTensorAndRdcs(thisRdcCsaTable, mediumName,
                typesOfRdcsToBeUsedInTensorRefinement, typesOfRdcsToBeUsedForAnalyticSolution,
                strandAdjInfo.firstElement().first(), Syy, Szz, refineCycle, dfsCycle, weightInScoringFunction, resolution, printResults);

        // Read the First Strand from the already computed PDB to test the program.        
        System.out.println("Printing the first strand: ");
        firstStrand.print();        
        myProtein currSheet = new myProtein(firstStrand);       

        // Compute and pack the second strand onwards
        for (myTriple<mySseInfo, mySseInfo, Boolean> thisStrandPair : strandAdjInfo) {
            mySseInfo thisSseId = thisStrandPair.second();
            mySseInfo prevStrandSpec = thisStrandPair.first();
            boolean areParallel = thisStrandPair.third();

            // The values below are preset to good ones as found empirically
            myProtein newSheet = computeThisStrandAndAddItToSheet(thisRdcCsaTable, mediumName,
                    typesOfRdcsToBeUsedInTensorRefinement, typesOfRdcsToBeUsedForAnalyticSolution,
                    thisSseId, Syy, Szz, /*refineCycle*/1, /*dfsCycle*/400000, weightInScoringFunction, /*resolution*/5.0, printResults,
                    hBondWeight, hBondVec, currSheet, areParallel, prevStrandSpec);

            currSheet = newSheet;
        }

        return currSheet;
    }

    /**
     * Compute the maximum likelihood fragment conformation in one of the principal
     * order frames so that it satisfies the RDCs.
     *
     * @param rdcCsaTableForThisMedium the table containing all RDCs in one medium
     * @param rdcCsaTableForThisMediumBackCal the table containing all back-computed RDCs in one medium
     * @param typesOfRdcsToBeUsedForAnalyticSolution speficy the types of RDCs to be used for constructing the secondary structure element
     * @param thisSseId the specification of the SSE boundary
     * @param pPlane the first peptide plane of the SSE in a principal order frame
     * @param Sxx the diagonalized alignment tensor component
     * @param Syy the diagonalized alignment tensor component
     * @param Szz the diagonalized alignment tensor component
     * @param phiTypeRdcRmsd the inverval used for sampling the RDCs
     * @param psiTypeRdcRmsd the inverval used for sampling the RDCs
     * @param dfsCycle the number of search trees to be enumerated, i.e., the number of sampled RDC sets to be used to compute the maximum likelihood structure
     * @param weightInScoringFunction relative weight for rms dihedral deviation from standard values
     * @param printResults
     * @return the conformation of the maximum likelihood fragment in one of the principal order frames
     */
    public myProtein computeMaximumlikelihoodFragment(Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMedium,
            Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMediumBackCal,
            Vector<myDipolarCoupling.Type> typesOfRdcsToBeUsedForAnalyticSolution, final mySseInfo thisSseId,
            final myPeptidePlane pPlane, final double Sxx, final double Syy, final double Szz,
            final double phiTypeRdcRmsd, final double psiTypeRdcRmsd, final int dfsCycle, final double weightInScoringFunction, boolean printResults) {

        long startTime = System.currentTimeMillis();

        // Get the RDC types that will be used in analytic solutions.
        myDipolarCoupling.Type phiTypeRdc = null;
        for (myDipolarCoupling.Type t : typesOfRdcsToBeUsedForAnalyticSolution) {
            if (myDipolarCoupling.isPhiTypeRdc(t)) {
                phiTypeRdc = t;
                break;
            }
        }
        if (phiTypeRdc == null) {
            System.out.println("Error: There does not exist RDC type that can be used to compute phi using analytic method");
            System.exit(1);
        }

        myDipolarCoupling.Type psiTypeRdc = null;
        for (myDipolarCoupling.Type t : typesOfRdcsToBeUsedForAnalyticSolution) {
            if (myDipolarCoupling.isPsiTypeRdc(t)) {
                psiTypeRdc = t;
                break;
            }
        }
        if (psiTypeRdc == null) {
            System.out.println("Error: There does not exist RDC type that can be used to compute psi using analytic method");
            System.exit(1);
        }
        
        Vector<myDipolarCoupling> phiTypeRdcVecWithMissingRdcsFilled = new Vector<myDipolarCoupling>();
        Vector<myDipolarCoupling> psiTypeRdcVecWithMissingRdcsFilled = new Vector<myDipolarCoupling>();

        int[] phiTypeExpOrBackCal = fillInMissingRdcs(thisSseId, rdcCsaTableForThisMedium.get(phiTypeRdc), rdcCsaTableForThisMediumBackCal.get(phiTypeRdc), phiTypeRdcVecWithMissingRdcsFilled);
        int[] psiTypeExpOrBackCal = fillInMissingRdcs(thisSseId, rdcCsaTableForThisMedium.get(psiTypeRdc), rdcCsaTableForThisMediumBackCal.get(psiTypeRdc), psiTypeRdcVecWithMissingRdcsFilled);
        

        // Clone the vectors to be used as place holders for sampled RDCs
        Vector<myDipolarCoupling> phiTypeRdcSampled = new Vector<myDipolarCoupling>(phiTypeRdcVecWithMissingRdcsFilled.size());
        for (myDipolarCoupling dc : phiTypeRdcVecWithMissingRdcsFilled) {
            phiTypeRdcSampled.add(new myDipolarCoupling(dc));
        }
        
        Vector<myDipolarCoupling> psiTypeRdcSampled = new Vector<myDipolarCoupling>(psiTypeRdcVecWithMissingRdcsFilled.size());
        for (myDipolarCoupling dc : psiTypeRdcVecWithMissingRdcsFilled) {
            psiTypeRdcSampled.add(new myDipolarCoupling(dc));
        }
        
        //Matrix mat = myPeptidePlane.RotationWrtGlobalFrame(pPlane);
        Matrix Rg = myForKin.getRotationWrtGlobalFrame(pPlane);

        boolean rightHand = true;

        System.out.println("Number of DFS cycles (= #Sampled RDC Sets): " + dfsCycle);
        System.out.println(phiTypeRdc.toString() + " RDC rmsd threshold: " + phiTypeRdcRmsd +  "    " + psiTypeRdc.toString() + " RDC rmsd threshold: " + psiTypeRdcRmsd);

        myPhiPsiSolutionTree thisPhiPsiSolutionTree = new myPhiPsiSolutionTree();

        long prevT = System.currentTimeMillis(), currT = System.currentTimeMillis();

        int maxSearchDepthThusFar = 0;

        for (int numberOfSamplingCycle = 0; numberOfSamplingCycle < dfsCycle; numberOfSamplingCycle++) {

            if (numberOfSamplingCycle % 10000 == 0) {
                System.out.println("Number of Sampling cycles completed so far: " + numberOfSamplingCycle / 1000 + "k");
                currT = System.currentTimeMillis();
                System.out.println("Time taken for 10K iterations: " + (double) (currT - prevT) / 1000.0);
                prevT = currT;
                System.out.println("Maximum search depth: " + thisPhiPsiSolutionTree.getMaxSearchDepth());
            }

            /*double phiTypeSquaredRdcDev = */generateSampledRdcVector(phiTypeRdcVecWithMissingRdcsFilled, phiTypeExpOrBackCal, phiTypeRdcRmsd, phiTypeRdcSampled, rr);
            /*double psiTypeSquaredRdcDev = */generateSampledRdcVector(psiTypeRdcVecWithMissingRdcsFilled, psiTypeExpOrBackCal, psiTypeRdcRmsd, psiTypeRdcSampled, rr);

            try {
                Writer phiPsiSolutionFile = null;
                File file = new File("phiPsiSolutionSets.txt");
                if (file.exists() && numberOfSamplingCycle == 0) {
                    file.delete();
                }
                phiPsiSolutionFile = new BufferedWriter(new FileWriter(file, true)); //true for append mode

                long oldTimeStamp = file.lastModified();

                //thisPhiPsiSolutionTree.__sampled_rdc_set_number = numberOfSamplingCycle;

                thisPhiPsiSolutionTree.phiPsiSolutionTreeRootCaller(thisSseId, phiTypeRdcSampled, psiTypeRdcSampled,
                        phiTypeRdcVecWithMissingRdcsFilled, psiTypeRdcVecWithMissingRdcsFilled,
                        Rg, Syy, Szz, pPlane, phiPsiSolutionFile, weightInScoringFunction, rightHand);

                if (thisPhiPsiSolutionTree.getMaxSearchDepth() > maxSearchDepthThusFar) {
                    System.out.println("Max Search Depth of " + thisPhiPsiSolutionTree.getMaxSearchDepth() + " achieved for the sampled set: " + numberOfSamplingCycle);
                    maxSearchDepthThusFar = thisPhiPsiSolutionTree.getMaxSearchDepth();
                }

                phiPsiSolutionFile.close();
                long newTimeStamp = file.lastModified();

                if (newTimeStamp - oldTimeStamp != 0) {
                    Writer finalOutputFile = null;
                    File finalFile = new File("phiPsiSolutionSets.txt");
                    //if (finalFile.exists()/* && mmm == 0*/)
                    //	finalFile.delete();
                    finalOutputFile = new BufferedWriter(new FileWriter(finalFile, true)); //true for append mode

                    System.out.println("Above solution is for the sampled set: " + (numberOfSamplingCycle + 1) + '\n');
                    finalOutputFile.write("Above solution is for the sampled set: " + (numberOfSamplingCycle + 1) + '\n');
                    finalOutputFile.close();
                }
            } catch (IOException e) {
                System.out.println("An IOException is thrown while writing to the outputfile inside CRTMinHelix");
                e.printStackTrace();
                System.exit(1);
            }
        }

        long endTime = System.currentTimeMillis();
        double totalTime = (double) ((endTime - startTime) / 60000.0); // We have the total time spent in doing DFS and analyzing solutions
        System.out.println("Total time spent in doing depth-first searches over all sampled set RDCs to compute the maximum likelihood solution is " + totalTime + " minutes");

        return thisPhiPsiSolutionTree.getBestFragment();
    }
            
    /**
     * Compute the maximum likelihood beta strand so that it satisfies the RDCs
     * and NOEs and forms hydrogen bonds with the previous strand of the sheet
     * to which it belongs.
     *
     * @param rdcCsaTableForThisMedium the table containing all RDCs in one medium
     * @param rdcCsaTableForThisMediumBackCal the table containing all back-computed RDCs in one medium
     * @param typesOfRdcsToBeUsedForAnalyticSolution speficy the types of RDCs to be used for constructing the secondary structure element
     * @param thisSseId the specification of the beta strand boundary
     * @param pPlane the first peptide plane of the beta strand
     * @param Sxx the diagonalized alignment tensor component
     * @param Syy the diagonalized alignment tensor component
     * @param Szz the diagonalized alignment tensor component
     * @param phiTypeRdcRmsd the inverval used for sampling the RDCs
     * @param psiTypeRdcRmsd the interval used for sampling the RDCs
     * @param dfsCycle the number of search trees to be enumerated, i.e., the number of sampled RDC sets to be used to compute the maximum likelihood structure
     * @param weightInScoringFunction relative weight for rms dihedral deviation from standard values
     * @param printResults
     * @param hBondWeight
     * @param noeVec the NOE/hbond table
     * @param partialSheetThusFar the partial sheet computed thus far
     * @return the conformation of the beta strand that can be added to grow the beta sheet by one more strand
     */
    public myProtein computeMaximumlikelihoodBetaStrand(Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMedium,
            Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMediumBackCal,
            Vector<myDipolarCoupling.Type> typesOfRdcsToBeUsedForAnalyticSolution, final mySseInfo thisSseId,
            final myPeptidePlane pPlane, final double Sxx, final double Syy, final double Szz,
            final double phiTypeRdcRmsd, final double psiTypeRdcRmsd, final int dfsCycle, final double weightInScoringFunction, boolean printResults,
            final double hBondWeight, Vector<myNoe> noeVec, myProtein partialSheetThusFar) {

        long startTime = System.currentTimeMillis();

        // Get the RDC types that will be used in analytic solutions.
        myDipolarCoupling.Type phiTypeRdc = null;
        for (myDipolarCoupling.Type t : typesOfRdcsToBeUsedForAnalyticSolution) {
            if (myDipolarCoupling.isPhiTypeRdc(t)) {
                phiTypeRdc = t;
                break;
            }
        }
        if (phiTypeRdc == null) {
            System.out.println("Error: There does not exist RDC type that can be used to compute phi using analytic method");
            System.exit(1);
        }

        myDipolarCoupling.Type psiTypeRdc = null;
        for (myDipolarCoupling.Type t : typesOfRdcsToBeUsedForAnalyticSolution) {
            if (myDipolarCoupling.isPsiTypeRdc(t)) {
                psiTypeRdc = t;
                break;
            }
        }
        if (psiTypeRdc == null) {
            System.out.println("Error: There does not exist RDC type that can be used to compute psi using analytic method");
            System.exit(1);
        }

        Vector<myDipolarCoupling> phiTypeRdcVecWithMissingRdcsFilled = new Vector<myDipolarCoupling>();
        Vector<myDipolarCoupling> psiTypeRdcVecWithMissingRdcsFilled = new Vector<myDipolarCoupling>();

        int[] phiTypeExpOrBackCal = fillInMissingRdcs(thisSseId, rdcCsaTableForThisMedium.get(phiTypeRdc), rdcCsaTableForThisMediumBackCal.get(phiTypeRdc), phiTypeRdcVecWithMissingRdcsFilled);
        int[] psiTypeExpOrBackCal = fillInMissingRdcs(thisSseId, rdcCsaTableForThisMedium.get(psiTypeRdc), rdcCsaTableForThisMediumBackCal.get(psiTypeRdc), psiTypeRdcVecWithMissingRdcsFilled);


        // Clone the vectors to be used as place holders for sampled RDCs
        Vector<myDipolarCoupling> phiTypeRdcSampled = new Vector<myDipolarCoupling>(phiTypeRdcVecWithMissingRdcsFilled.size());
        for (myDipolarCoupling dc : phiTypeRdcVecWithMissingRdcsFilled) {
            phiTypeRdcSampled.add(new myDipolarCoupling(dc));
        }

        Vector<myDipolarCoupling> psiTypeRdcSampled = new Vector<myDipolarCoupling>(psiTypeRdcVecWithMissingRdcsFilled.size());
        for (myDipolarCoupling dc : psiTypeRdcVecWithMissingRdcsFilled) {
            psiTypeRdcSampled.add(new myDipolarCoupling(dc));
        }

        //Matrix mat = myPeptidePlane.RotationWrtGlobalFrame(pPlane);
        Matrix Rg = myForKin.getRotationWrtGlobalFrame(pPlane);

        boolean rightHand = true;

        System.out.println("Number of DFS cycles (= #Sampled RDC Sets): " + dfsCycle);
        System.out.println(phiTypeRdc.toString() + " RDC rmsd threshold: " + phiTypeRdcRmsd +  "    " + psiTypeRdc.toString() + " RDC rmsd threshold: " + psiTypeRdcRmsd);

        myPhiPsiSolutionTree thisPhiPsiSolutionTree = new myPhiPsiSolutionTree();

        long prevT = System.currentTimeMillis(), currT = System.currentTimeMillis();

        int maxSearchDepthThusFar = 0;

        for (int numberOfSamplingCycle = 0; numberOfSamplingCycle < dfsCycle; numberOfSamplingCycle++) {

            if (numberOfSamplingCycle % 10000 == 0) {
                System.out.println("Number of Sampling cycles completed so far: " + numberOfSamplingCycle / 1000 + "k");
                currT = System.currentTimeMillis();
                System.out.println("Time taken for 10K iterations: " + (double) (currT - prevT) / 1000.0);
                prevT = currT;
                System.out.println("Maximum search depth: " + thisPhiPsiSolutionTree.getMaxSearchDepth());
            }

            /*double phiTypeSquaredRdcDev = */generateSampledRdcVector(phiTypeRdcVecWithMissingRdcsFilled, phiTypeExpOrBackCal, phiTypeRdcRmsd, phiTypeRdcSampled, rr);
            /*double psiTypeSquaredRdcDev = */generateSampledRdcVector(psiTypeRdcVecWithMissingRdcsFilled, psiTypeExpOrBackCal, psiTypeRdcRmsd, psiTypeRdcSampled, rr);

            try {
                Writer phiPsiSolutionFile = null;
                File file = new File("phiPsiSolutionSets.txt");
                if (file.exists() && numberOfSamplingCycle == 0) {
                    file.delete();
                }
                phiPsiSolutionFile = new BufferedWriter(new FileWriter(file, true)); //true for append mode

                long oldTimeStamp = file.lastModified();

                //thisPhiPsiSolutionTree.__sampled_rdc_set_number = numberOfSamplingCycle;

                thisPhiPsiSolutionTree.phiPsiSolutionTreeRootCallerForBetaStrand(thisSseId, phiTypeRdcSampled, psiTypeRdcSampled,
                        phiTypeRdcVecWithMissingRdcsFilled, psiTypeRdcVecWithMissingRdcsFilled,
                        Rg, Syy, Szz, pPlane, phiPsiSolutionFile, weightInScoringFunction, rightHand,
                        hBondWeight, noeVec, partialSheetThusFar);

                // Note: this piece of code is only to hasten the search
                if (thisPhiPsiSolutionTree.getSolutionCounter() > myMiscConstants.maxSolutionStrands) {
                    numberOfSamplingCycle = dfsCycle - 1;
                }


                if (thisPhiPsiSolutionTree.getMaxSearchDepth() > maxSearchDepthThusFar) {
                    System.out.println("Max Search Depth of " + thisPhiPsiSolutionTree.getMaxSearchDepth() + " achieved for the sampled set: " + numberOfSamplingCycle);
                    maxSearchDepthThusFar = thisPhiPsiSolutionTree.getMaxSearchDepth();
                }

                phiPsiSolutionFile.close();
                long newTimeStamp = file.lastModified();

                if (newTimeStamp - oldTimeStamp != 0) {
                    Writer finalOutputFile = null;
                    File finalFile = new File("phiPsiSolutionSets.txt");
                    //if (finalFile.exists()/* && mmm == 0*/)
                    //	finalFile.delete();
                    finalOutputFile = new BufferedWriter(new FileWriter(finalFile, true)); //true for append mode

                    System.out.println("Above solution is for the sampled set: " + (numberOfSamplingCycle + 1) + '\n');
                    finalOutputFile.write("Above solution is for the sampled set: " + (numberOfSamplingCycle + 1) + '\n');
                    finalOutputFile.close();
                }
            } catch (IOException e) {
                System.out.println("An IOException is thrown while writing to the outputfile inside CRTMinHelix");
                e.printStackTrace();
                System.exit(1);
            }
        }

        long endTime = System.currentTimeMillis();
        double totalTime = (double) ((endTime - startTime) / 60000.0); // We have the total time spent in doing DFS and analyzing solutions
        System.out.println("Total time spent in doing depth-first searches over all sampled set RDCs to compute the maximum likelihood solution is " + totalTime + " minutes");

        System.out.println("Looking into what is being returned:");
        if (thisPhiPsiSolutionTree.getBestFragment() == null) {
            System.out.println("NOTE: null is being returned from computeMaximumlikelihoodBetaStrand__");
        }
        return thisPhiPsiSolutionTree.getBestFragment();
    }




    /**
     * This method tests the various methods in this class.
     * 
     * @param args
     */
    public static void main(String... args) {

        mySseInfo thisSseId = new mySseInfo(24, 34, "H");

         myPeptidePlane pPlane = new myPeptidePlane(new myPoint(0.0, 0.0, 0.0), new myPoint(0.0, 0.0, -Const.dN2H),
                new myPoint(0.0, Const.dN2CA * Math.cos(Const.theta1), Const.dN2CA * Math.sin(Const.theta1)));

        Vector<myPhiPsi> phiPsiVec = new Vector<myPhiPsi>();

        for (int i = thisSseId.getBeginResidueNumber(); i <= thisSseId.getEndResidueNumber(); i++) {
            if (thisSseId.isHelix()) {
                phiPsiVec.add(new myPhiPsi(i, "ALA", Const.phiAveHelix, Const.psiAveHelix));
            } else if (thisSseId.isStrand()) {
                phiPsiVec.add(new myPhiPsi(i, "ALA", Const.phiAveBeta, Const.psiAveBeta));
            } else {
                System.out.println("Error: " + thisSseId.toString() + " is not an SSE");
                System.exit(1);
            }
        }
        myProtein idealHelix1 =  myForKin.buildIdealSSE(thisSseId); //buildIdealSSE(thisSseId);

        myProtein idealHelix2 = myProtein.buildProteinBackboneFromDihedrals(phiPsiVec, pPlane);

        System.out.println("ideal helix 1");
        idealHelix1.print();
        System.out.println("ideal helix 2");
        idealHelix2.print();
        
    }

    /****************************Deprecated Methods****************************/

    /**
     * Bootstrap the alignment tensor computation and refine the alignment tensor
     * iteratively to finally obtain a good estimate of the alignment tensor.
     *
     * @param thisSseId the secondary structure element to be used to bootstrap the alignment tensor
     * @param nhRdcVec the vector of H-HN RDCs for the secondary structure element
     * @param chRdcVec the vector of CA-HA RDCs for the secondary structure element
     * @param refineCycle the number of refinements to be done during the alignment tensor computation
     * @param dfsCycle the number of depth-first search cycles to be used,
     * i.e., the number of sampled RDC sets to be used to compute the maximum likelihood structure
     * @param weightInScoringFunction relative weight of PHI-PSI deviation from ideal values wrt. the RDC rmsds (weight = 1) in the scoring function
     * @param resolution
     * @param printResults
     * @param saupe this array of size four holds {Syy, Szz, nhRdcRmsd, chRdcRmsd} and these values are to be passed back to the caller
     * @return the protein fragment for the secondary structure element in the principal order frame
     */
    @Deprecated public myProtein computeAlignmentTensor(final mySseInfo thisSseId, final Vector<myDipolarCoupling> nhRdcVec,
            final Vector<myDipolarCoupling> chRdcVec, final int refineCycle, final int dfsCycle, final double weightInScoringFunction,
            final double resolution, final boolean printResults, double[] saupe) {
        // Do sanity check and build an ideal SSE model
        int beginResNum = Math.min(nhRdcVec.firstElement().getResidueNumber(), chRdcVec.firstElement().getResidueNumber());
        int endResNum = Math.max(nhRdcVec.lastElement().getResidueNumber(), chRdcVec.lastElement().getResidueNumber());
        if (!(beginResNum == thisSseId.getBeginResidueNumber() && endResNum == thisSseId.getEndResidueNumber())) {
            System.out.println("Error: The SSE " + thisSseId.toString() + " to be computed is longer than the available data available for it. Please modify the SSE boundary.");
            System.exit(1);
        }

        myProtein idealSse =  myForKin.buildIdealSSE(thisSseId);
        System.out.println("Printing Ideal Helix");
        idealSse.print();

        Vector<myDipolarCoupling> nhRdcBackCal = new Vector<myDipolarCoupling>();
        Vector<myDipolarCoupling> chRdcBackCal = new Vector<myDipolarCoupling>();
        //System.out.println("thisSseId.numberOfResidues() = " + thisSseId.numberOfResidues());
        //System.out.println("size = " + nhRdcBackCal.size() + chRdcBackCal.size());

        // Compute the alignment tensor and the rotation to one of the POFs
        Matrix rotToPOF = myAlignmentTensorEstimator.bestFitUsingSVD(idealSse, nhRdcVec, chRdcVec, nhRdcBackCal, chRdcBackCal, saupe);
        //System.out.println("size = " + nhRdcBackCal.size() + chRdcBackCal.size());

        // Rotate the SSE to the POF
        myProtein thisFragmentInPOF = new myProtein(idealSse);
        thisFragmentInPOF.rotate(new myMatrix(rotToPOF.getArrayCopy()));
        //System.out.println("************");
        //idealSse.print();
        //pdbVecInPOF.print();
        //System.out.println("************");

        Writer outputSaupe = null;
        File saupeListFile = new File("saupeList.txt"); // Note that this file holds the emerging pattern of AT and finally has to be parameterized.
        if (saupeListFile.exists()) {
            saupeListFile.delete();
        }

        myProtein sseUsingRefinedTensor = null;

        try {
            outputSaupe = new BufferedWriter(new FileWriter(saupeListFile));
            System.out.println("Alignment Tensor after Round 0:    Syy: " + saupe[0] + "    Szz: " + saupe[1]);
            outputSaupe.write("Alignment Tensor after Round 0:    Syy: " + saupe[0] + "    Szz: " + saupe[1] + '\n');
            System.out.println("Rhombicity = " + computeRhombicity(saupe[0], saupe[1]));
            outputSaupe.write("Rhombicity = " + computeRhombicity(saupe[0], saupe[1]) + '\n');
            outputSaupe.close();

            for (int iter = 1; iter <= refineCycle; iter++) {
                System.out.println("Number of DFS cycles (= #Sampled RDC Sets): " + dfsCycle);

                myResidue firstResidue = thisFragmentInPOF.residueAt(thisFragmentInPOF.getBeginResidueNumber());
                myPeptidePlane pPlane = new myPeptidePlane(firstResidue.getAtom(myAtomLabel.__N).getCoordinates(),
                        firstResidue.getAtom(myAtomLabel.__H).getCoordinates(), firstResidue.getAtom(myAtomLabel.__CA).getCoordinates());

                sseUsingRefinedTensor = computeMaximumlikelihoodFragment(thisSseId, nhRdcVec, chRdcVec, nhRdcBackCal, chRdcBackCal,
                        pPlane, saupe[0], saupe[1], saupe[2], saupe[3], dfsCycle, weightInScoringFunction, printResults);
                rotToPOF = myAlignmentTensorEstimator.bestFitUsingSVD(sseUsingRefinedTensor, nhRdcVec, chRdcVec, nhRdcBackCal, chRdcBackCal, saupe);

                thisFragmentInPOF = new myProtein(sseUsingRefinedTensor);
                thisFragmentInPOF.rotate(new myMatrix(rotToPOF.getArrayCopy()));

                System.out.println("Printing the best fragment after rotation");
                thisFragmentInPOF.print();

                outputSaupe = new BufferedWriter(new FileWriter(saupeListFile, true));
                System.out.println("Alignment Tensor after Round " + iter + ":    Syy: " + saupe[0] + "    Szz: " + saupe[1]);
                outputSaupe.write("Alignment Tensor after Round " + iter + ":    Syy: " + saupe[0] + "    Szz: " + saupe[1] + '\n');
                System.out.println("Rhombicity = " + computeRhombicity(saupe[0], saupe[1]));
                outputSaupe.write("Rhombicity = " + computeRhombicity(saupe[0], saupe[1]) + '\n');
                outputSaupe.close();
            }
        } catch (IOException e) {
            e.printStackTrace();
        }

        System.out.println("Printing best fragment BEFORE rotation to POF");
        sseUsingRefinedTensor.print();
        System.out.println("Printing best fragment AFTER rotation to POF");
        thisFragmentInPOF.print();

        return thisFragmentInPOF;
        //return pdbVecUsingRefinedTensor;
    }

    /**
     * Compute the maximum likelihood protein fragment from the RDC data, using
     * the exact solution based approach. Return the protein fragment for which
     * the scoring function has the minimum value.
     *
     * @param thisSseId the secondary structure element boundary specification
     * @param nhRdcVec the vector containing the experimental N-HN RDCs for the SSE
     * @param chRdcVec the vector containing the experimental CA-HA RDCs for the SSE
     * @param nhRdcBackCal the back-computed N-HN RDC vector for the SSE
     * @param chRdcBackCal the back-computed CA-HA RDC vector for the SSE
     * @param pPlane the first peptide plane of the SSE
     * @param Syy the diagonalized alignment tensor component
     * @param Szz the diagonalized alignment tensor component
     * @param nhRmsd the N-HN RDC rmsd threshold to be used to generate sampled RDC set from experimental RDC set
     * @param chRmsd the CA-HA RDC rmsd threshold to be used to generate sampled RDC set from experimental RDC set
     * @param dfsCycle the number of sampled RDC sets for each of which a depth-first search is done to compute the solution
     * @param weightInScoringFunction the relative weight used in scoring function
     * @param printResults
     * @return the maximum likelihood protein fragment computed over all sampled sets of RDCs
     */
    @Deprecated public myProtein computeMaximumlikelihoodFragment(final mySseInfo thisSseId, Vector<myDipolarCoupling> nhRdcVec, final Vector<myDipolarCoupling> chRdcVec,
            final Vector<myDipolarCoupling> nhRdcBackCal, final Vector<myDipolarCoupling> chRdcBackCal, final myPeptidePlane pPlane,
            final double Syy, final double Szz, final double nhRmsd, final double chRmsd, final int dfsCycle,
            final double weightInScoringFunction, boolean printResults) {

        long startTime = System.currentTimeMillis();

        Vector<myDipolarCoupling> nhRdcVecWithMissingRdcsFilled = new Vector<myDipolarCoupling>();
        Vector<myDipolarCoupling> chRdcVecWithMissingRdcsFilled = new Vector<myDipolarCoupling>();
        int[] nhExpOrBackCal = fillInMissingRdcs(thisSseId, nhRdcVec, nhRdcBackCal, nhRdcVecWithMissingRdcsFilled);
        int[] chExpOrBackCal = fillInMissingRdcs(thisSseId, chRdcVec, chRdcBackCal, chRdcVecWithMissingRdcsFilled);

        // Clone the vectors to be used as place holders for sampled RDCs
        Vector<myDipolarCoupling> nhRdcSampled = new Vector<myDipolarCoupling>();
        for (myDipolarCoupling dc : nhRdcVecWithMissingRdcsFilled) {
            nhRdcSampled.add(new myDipolarCoupling(dc));
        }

        Vector<myDipolarCoupling> chRdcSampled = new Vector<myDipolarCoupling>();
        for (myDipolarCoupling dc : chRdcVecWithMissingRdcsFilled) {
            chRdcSampled.add(new myDipolarCoupling(dc));
        }

        Matrix mat = myPeptidePlane.RotationWrtGlobalFrame(pPlane);

        boolean rightHand = true;

        System.out.println("Number of DFS cycles (= #Sampled RDC Sets): " + dfsCycle);
        System.out.println("N-HN RDC rmsd threshold: " + nhRmsd + "    CA-HA RDC rmsd threshold: " + chRmsd);

        myPhiPsiSolutionTree thisPhiPsiSolutionTree = new myPhiPsiSolutionTree();

        long prevT = System.currentTimeMillis(), currT = System.currentTimeMillis();

        int maxSearchDepthThusFar = 0;

        for (int numberOfSamplingCycle = 0; numberOfSamplingCycle < dfsCycle; numberOfSamplingCycle++) {

            if (numberOfSamplingCycle % 10000 == 0) {
                System.out.println("Number of Sampling cycles completed so far: " + numberOfSamplingCycle / 1000 + "k");
                currT = System.currentTimeMillis();
                System.out.println("Time taken for 10K iterations: " + (double) (currT - prevT) / 1000.0);
                prevT = currT;
                System.out.println("Maximum search depth: " + thisPhiPsiSolutionTree.getMaxSearchDepth());
            }

            /*double nhSquaredRdcDev = */generateSampledRdcVector(nhRdcVecWithMissingRdcsFilled, nhExpOrBackCal, nhRmsd, nhRdcSampled, rr); // NH
            /*double chSquaredRdcDev = */generateSampledRdcVector(chRdcVecWithMissingRdcsFilled, chExpOrBackCal, chRmsd, chRdcSampled, rr); // CH

            try {
                Writer phiPsiSolutionFile = null;
                File file = new File("phiPsiSolutionSets.txt");
                if (file.exists() && numberOfSamplingCycle == 0) {
                    file.delete();
                }
                phiPsiSolutionFile = new BufferedWriter(new FileWriter(file, true)); //true for append mode

                long oldTimeStamp = file.lastModified();

                //thisPhiPsiSolutionTree.__sampled_rdc_set_number = numberOfSamplingCycle;

                thisPhiPsiSolutionTree.phiPsiSolutionTreeRootCaller(thisSseId, nhRdcSampled, chRdcSampled, nhRdcVecWithMissingRdcsFilled, chRdcVecWithMissingRdcsFilled,
                        mat, Syy, Szz, pPlane, phiPsiSolutionFile, weightInScoringFunction, rightHand);

                if (thisPhiPsiSolutionTree.getMaxSearchDepth() > maxSearchDepthThusFar) {
                    System.out.println("Max Search Depth of " + thisPhiPsiSolutionTree.getMaxSearchDepth() + " achieved for the sampled set: " + numberOfSamplingCycle);
                    maxSearchDepthThusFar = thisPhiPsiSolutionTree.getMaxSearchDepth();
                }

                phiPsiSolutionFile.close();
                long newTimeStamp = file.lastModified();

                if (newTimeStamp - oldTimeStamp != 0) {
                    Writer finalOutputFile = null;
                    File finalFile = new File("phiPsiSolutionSets.txt");
                    //if (finalFile.exists()/* && mmm == 0*/)
                    //	finalFile.delete();
                    finalOutputFile = new BufferedWriter(new FileWriter(finalFile, true)); //true for append mode

                    System.out.println("Above solution is for the sampled set: " + (numberOfSamplingCycle + 1) + '\n');
                    finalOutputFile.write("Above solution is for the sampled set: " + (numberOfSamplingCycle + 1) + '\n');
                    finalOutputFile.close();
                }
            } catch (IOException e) {
                System.out.println("An IOException is thrown while writing to the outputfile inside CRTMinHelix");
                e.printStackTrace();
                System.exit(1);
            }
        }

        long endTime = System.currentTimeMillis();
        double totalTime = (double) ((endTime - startTime) / 60000.0); // We have the total time spent in doing DFS and analyzing solutions
        System.out.println("Total time spent in doing depth-first searches over all sampled set RDCs to compute the maximum likelihood solution is " + totalTime + " minutes");

        return thisPhiPsiSolutionTree.getBestFragment();
    }

    /**
     * Compute the backbone structure of the protein fragment given the alignment
     * tensor and the RDCs using the exact solution and systematic search based
     * algorithm.
     *
     * @param thisSseId the secondary structure element boundary specification
     * @param nhRdcVec the vector containing the experimental N-HN RDCs for the SSE
     * @param chRdcVec the vector containing the experimental CA-HA RDCs for the SSE
     * @param refineCycle the number of refinements (to the initial peptide plane)
     * to be done during the structure computation
     * @param dfsCycle the number of sampled RDC sets for each of which a depth-first search is done to compute the solution
     * @param weightInScoringFunction the relative weight used in scoring function
     * @param resolution the angular resolution for grid search to estimate the best-fit initial peptide plane
     * @param printResults
     * @param Syy the diagonalized alignment tensor component
     * @param Szz the diagonalized alignment tensor component
     * @return the maximum likelihood protein fragment in the POF computed over all sampled sets of RDCs
     */
    @Deprecated public myProtein computeSseGivenAlignmentTensorAndRdcs(final mySseInfo thisSseId, final Vector<myDipolarCoupling> nhRdcVec,
            final Vector<myDipolarCoupling> chRdcVec, final int refineCycle, final int dfsCycle, final double weightInScoringFunction,
            final double resolution, final boolean printResults, final double Syy, final double Szz) {
//        System.out.println("I am inside computeSseGivenAlignmentTensorAndRdcs...");

        // do sanity check and build an ideal SSE model
        int beginResNum = Math.min(nhRdcVec.firstElement().getResidueNumber(), chRdcVec.firstElement().getResidueNumber());
        int endResNum = Math.max(nhRdcVec.lastElement().getResidueNumber(), chRdcVec.lastElement().getResidueNumber());
        if (!(beginResNum == thisSseId.getBeginResidueNumber() && endResNum == thisSseId.getEndResidueNumber())) {
            System.out.println("Error: The SSE " + thisSseId.toString() + " to be computed is longer than the available data available for it. Please modify the SSE boundary");
            System.exit(1);
        }

        myProtein idealSse =  myForKin.buildIdealSSE(thisSseId); //buildIdealSSE(thisSseId);
        Vector<myDipolarCoupling> nhRdcBackCal = new Vector<myDipolarCoupling>();
        Vector<myDipolarCoupling> chRdcBackCal = new Vector<myDipolarCoupling>();
        //System.out.println("thisSseId.numberOfResidues() = " + thisSseId.numberOfResidues());
        //System.out.println("size = " + nhRdcBackCal.size() + chRdcBackCal.size());

        // Compute the alignment tensor and the rotation to one of the POFs
        double[] dummy = new double[4];
        //Matrix rotToPOF = myAlignmentTensorEstimator.bestFitUsingSVD(idealSse, nhRdcVec, chRdcVec, nhRdcBackCal, chRdcBackCal, saupeNeverUsed);
        //System.out.println("size = " + nhRdcBackCal.size() + chRdcBackCal.size());

        Matrix rotToPOF = myAlignmentTensorEstimator.bestFitUsingGridSearchForGivenAT(idealSse, nhRdcVec, chRdcVec, Syy, Szz, resolution, dummy);
        double nhRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(idealSse, nhRdcVec, rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.nhRatio);
        double chRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(idealSse, chRdcVec, rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.cahaRatio);

        nhRdcRmsd = sandwich(0.05, 1.5, nhRdcRmsd);
        chRdcRmsd = sandwich(0.05, 2.0, chRdcRmsd);

        nhRdcBackCal = myAlignmentTensorEstimator.backComputeRdcs(idealSse, nhRdcVec.elementAt(0), rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.nhRatio);
        chRdcBackCal = myAlignmentTensorEstimator.backComputeRdcs(idealSse, chRdcVec.elementAt(0), rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.cahaRatio);

        // Rotate the SSE to the POF
        myProtein thisFragmentInPOF = new myProtein(idealSse);
        thisFragmentInPOF.rotate(new myMatrix(rotToPOF.getArrayCopy()));
        //System.out.println("************");
        //idealSse.print();
        //pdbVecInPOF.print();
        //System.out.println("************");

        Writer outputSaupe = null;
        File saupeListFile = new File("saupeList.txt"); // Note that this file holds the emerging pattern of AT and finally has to be parameterized.
        if (saupeListFile.exists()) {
            saupeListFile.delete();
        }

        myProtein sseUsingRefinedTensor = null;

        try {
            outputSaupe = new BufferedWriter(new FileWriter(saupeListFile));
            System.out.println("Alignment Tensor after Round 0:    Syy: " + Syy + "    Szz: " + Szz);
            outputSaupe.write("Alignment Tensor after Round 0:    Syy: " + Syy + "    Szz: " + Szz + '\n');
            System.out.println("Rhombicity = " + computeRhombicity(Syy, Szz));
            outputSaupe.write("Rhombicity = " + computeRhombicity(Syy, Szz) + '\n');
            outputSaupe.close();

            //System.out.println("refineCycle = " + refineCycle); System.exit(1);

            for (int iter = 1; iter <= refineCycle; iter++) {
                System.out.println("Number of DFS cycles (= #Sampled RDC Sets): " + dfsCycle);
                System.out.println("N-HN RDC rmsd threshold: " + nhRdcRmsd + "    CA-HA RDC rmsd threshold: " + chRdcRmsd);

                myResidue firstResidue = thisFragmentInPOF.residueAt(thisFragmentInPOF.getBeginResidueNumber());
                myPeptidePlane pPlane = new myPeptidePlane(firstResidue.getAtom(myAtomLabel.__N).getCoordinates(),
                        firstResidue.getAtom(myAtomLabel.__H).getCoordinates(), firstResidue.getAtom(myAtomLabel.__CA).getCoordinates());

                sseUsingRefinedTensor = computeMaximumlikelihoodFragment(thisSseId, nhRdcVec, chRdcVec, nhRdcBackCal, chRdcBackCal, pPlane,
                        Syy, Szz, nhRdcRmsd, chRdcRmsd, dfsCycle, weightInScoringFunction, printResults);

                if (sseUsingRefinedTensor == null) {
                    System.out.println("Error: sseUsingRefinedTensor is null in computeSseGivenAlignmentTensorAndRdcs");
                    System.exit(1);
                }
                //rotToPOF = myAlignmentTensorEstimator.bestFitUsingSVD(sseUsingRefinedTensor, nhRdcVec, chRdcVec, nhRdcBackCal, chRdcBackCal, saupeNeverUsed);
                rotToPOF = myAlignmentTensorEstimator.bestFitUsingGridSearchForGivenAT(sseUsingRefinedTensor, nhRdcVec, chRdcVec, Syy, Szz, resolution, dummy);
                nhRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(sseUsingRefinedTensor, nhRdcVec, rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.nhRatio);
                chRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(sseUsingRefinedTensor, chRdcVec, rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.cahaRatio);

                nhRdcRmsd = sandwich(0.05, 1.5, nhRdcRmsd);
                chRdcRmsd = sandwich(0.05, 2.0, chRdcRmsd);

                nhRdcBackCal = myAlignmentTensorEstimator.backComputeRdcs(sseUsingRefinedTensor, nhRdcVec.elementAt(0), rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.nhRatio);
                chRdcBackCal = myAlignmentTensorEstimator.backComputeRdcs(sseUsingRefinedTensor, chRdcVec.elementAt(0), rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.cahaRatio);

                thisFragmentInPOF = new myProtein(sseUsingRefinedTensor);
                thisFragmentInPOF.rotate(new myMatrix(rotToPOF.getArrayCopy()));

                System.out.println("Printing the best fragment after rotation");
                thisFragmentInPOF.print();

                outputSaupe = new BufferedWriter(new FileWriter(saupeListFile, true));
                System.out.println("Alignment Tensor after Round " + iter + ":    Syy: " + Syy + "    Szz: " + Szz);
                outputSaupe.write("Alignment Tensor after Round " + iter + ":    Syy: " + Syy + "    Szz: " + Szz + '\n');
                System.out.println("Rhombicity = " + computeRhombicity(Syy, Szz));
                outputSaupe.write("Rhombicity = " + computeRhombicity(Syy, Szz) + '\n');
                outputSaupe.close();
            }
        } catch (IOException e) {
            e.printStackTrace();
        }

        //return thisFragmentInPOF;
        return sseUsingRefinedTensor;
    }

    /**
     * Compute the SSE given the alignment tensor components and the experimental RDCs.
     *
     * @param thisSseId
     * @param nhRdcVec
     * @param chRdcVec
     * @param refineCycle
     * @param dfsCycle
     * @param weight4Angles
     * @param resolution
     * @param printResults
     * @param Syy
     * @param Szz
     * @return
     * @deprecated
     */
    @Deprecated public myProtein CRTComputeSseFromGivenAT(final mySseInfo thisSseId, final Vector<myDipolarCoupling> nhRdcVec,
            final Vector<myDipolarCoupling> chRdcVec, final int refineCycle, final int dfsCycle, final double weight4Angles,
            final double resolution, final boolean debugDFS, final boolean printResults, final double Syy, final double Szz) {
        System.out.println("I am inside CRTComputeSseFromGivenAT...");

        // do sanity check and build an ideal SSE model
        int beginResNum = Math.min(nhRdcVec.firstElement().getResidueNumber(), chRdcVec.firstElement().getResidueNumber());
        int endResNum = Math.max(nhRdcVec.lastElement().getResidueNumber(), chRdcVec.lastElement().getResidueNumber());
        if (!(beginResNum == thisSseId.getBeginResidueNumber() && endResNum == thisSseId.getEndResidueNumber())) {
            System.out.println("Error: The SSE " + thisSseId.toString() + " to be computed is longer than the available data available for it. Please modify the SSE boundary");
            System.exit(1);
        }

        myProtein idealSse =  myForKin.buildIdealSSE(thisSseId); //buildIdealSSE(thisSseId);
        Vector<myDipolarCoupling> nhRdcBackCal = new Vector<myDipolarCoupling>();
        Vector<myDipolarCoupling> chRdcBackCal = new Vector<myDipolarCoupling>();
        //System.out.println("thisSseId.numberOfResidues() = " + thisSseId.numberOfResidues());
        //System.out.println("size = " + nhRdcBackCal.size() + chRdcBackCal.size());

        // Compute the alignment tensor and the rotation to one of the POFs
        double[] saupeNeverUsed = new double[4];
        Matrix rotToPOF = myAlignmentTensorEstimator.bestFitUsingSVD(idealSse, nhRdcVec, chRdcVec, nhRdcBackCal, chRdcBackCal, saupeNeverUsed);
        //System.out.println("size = " + nhRdcBackCal.size() + chRdcBackCal.size());

        // Rotate the SSE to the POF
        myProtein thisFragmentInPOF = new myProtein(idealSse);
        thisFragmentInPOF.rotate(new myMatrix(rotToPOF.getArrayCopy()));
        //System.out.println("************");
        //idealSse.print();
        //pdbVecInPOF.print();
        //System.out.println("************");

        Writer outputSaupe = null;
        File saupeListFile = new File("saupeList.txt"); // Note that this file holds the emerging pattern of AT and finally has to be parameterized.
        if (saupeListFile.exists()) {
            saupeListFile.delete();
        }

        myProtein sseUsingRefinedTensor = null;

        try {
            outputSaupe = new BufferedWriter(new FileWriter(saupeListFile));
            System.out.println("Alignment Tensor after Round 0:    Syy: " + Syy + "    Szz: " + Szz);
            outputSaupe.write("Alignment Tensor after Round 0:    Syy: " + Syy + "    Szz: " + Szz + '\n');
            System.out.println("Rhombicity = " + computeRhombicity(Syy, Szz));
            outputSaupe.write("Rhombicity = " + computeRhombicity(Syy, Szz) + '\n');
            outputSaupe.close();

            //System.out.println("refineCycle = " + refineCycle); System.exit(1);

            for (int iter = 1; iter <= refineCycle; iter++) {
                System.out.println("dfsCycle is equal to: " + dfsCycle);

                System.out.println("saupeNeverUsed[2]: " +  saupeNeverUsed[2] + "    saupeNeverUsed[3]: " + saupeNeverUsed[3]);

                myResidue firstResidue = thisFragmentInPOF.residueAt(thisFragmentInPOF.getBeginResidueNumber());
                myPeptidePlane pPlane = new myPeptidePlane(firstResidue.getAtom(myAtomLabel.__N).getCoordinates(),
                        firstResidue.getAtom(myAtomLabel.__H).getCoordinates(), firstResidue.getAtom(myAtomLabel.__CA).getCoordinates());

                sseUsingRefinedTensor = computeMaximumlikelihoodFragment(thisSseId, nhRdcVec, chRdcVec, nhRdcBackCal, chRdcBackCal, pPlane,
                        Syy, Szz, saupeNeverUsed[2], saupeNeverUsed[3], dfsCycle, weight4Angles, printResults);

                if (sseUsingRefinedTensor == null) {
                    System.out.println("Error: sseUsingRefinedTensor == null in CRTComputeSseFromGivenAT");
                    System.exit(1);
                } /*else {
                    System.out.println("Debugging................");
                    sseUsingRefinedTensor.print();
                    System.out.println(thisSseId.toString());
                }*/
                rotToPOF = myAlignmentTensorEstimator.bestFitUsingSVD(sseUsingRefinedTensor, nhRdcVec, chRdcVec, nhRdcBackCal, chRdcBackCal, saupeNeverUsed);

                thisFragmentInPOF = new myProtein(sseUsingRefinedTensor);
                thisFragmentInPOF.rotate(new myMatrix(rotToPOF.getArrayCopy()));

                System.out.println("Printing the best fragment after rotation");
                thisFragmentInPOF.print();

                outputSaupe = new BufferedWriter(new FileWriter(saupeListFile, true));
                System.out.println("Alignment Tensor after Round " + iter + ":    Syy: " + Syy + "    Szz: " + Szz);
                outputSaupe.write("Alignment Tensor after Round " + iter + ":    Syy: " + Syy + "    Szz: " + Szz + '\n');
                System.out.println("Rhombicity = " + computeRhombicity(Syy, Szz));
                outputSaupe.write("Rhombicity = " + computeRhombicity(Syy, Szz) + '\n');
                outputSaupe.close();
                //nCycle /= 10;
                //nCycle -= 10000;
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        System.out.println("exiting from ModelRdc->refineSaupe...");
        //return thisFragmentInPOF;

        return sseUsingRefinedTensor;
    }
}
