/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2009 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, 01 December, 2009
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.0, Nov 18, 2009
 * @author        Chittaranjan Tripathy (2007-2009)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */
import java.util.*;
import java.io.*;

/**
 * Description of the class
 */
public class myMiscConstants {
    /**
     * Default constructor which creates an instance of this class. This
     * constructor is private to enforce the singleton property of the class in
     * future.
     */
    private myMiscConstants() {
    }

    /**
     * This class parses the advanced configuration parameters from a file and
     * stores them as constants.
     */
    private static class advancedParameterParser {

        /**
         * The parameters read from the parameter file is stored in this map.
         */
        private Map<String, String> __parameter_map = new TreeMap<String, String>();

        /**
         * This is the default constructor which reads from a file the RDC and
         * CSA parameters from a file and stores them as constants.
         */
        public advancedParameterParser(String parameterFile) {
            try {
                Scanner scanner = new Scanner(new File(parameterFile));
                scanner.useDelimiter(System.getProperty("line.separator"));
                while (scanner.hasNextLine()) {
                    String line = myMiscUtilities.stripLineComment(scanner.nextLine()).trim();
                    if (line.length() != 0) {
                        String[] words = line.split("\\s+");
                        if (words.length == 2) {
                            //System.out.println("words: " + Arrays.toString(words));
                            __parameter_map.put(words[0], words[1]);
                        }
                    }
                }
                scanner.close();
            } catch (FileNotFoundException e) {
                System.out.println("Error: Input file " + parameterFile + " not found");
                e.printStackTrace();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * This object when constructed contains the set of constants read from the
     * file.
     */
    private static advancedParameterParser __parser = new advancedParameterParser(System.getProperty("user.dir") + System.getProperty("file.separator") +
                    "mytoppar" + System.getProperty("file.separator") + "advanced_configuration.txt");

    private static advancedParameterParser __parser2 = new advancedParameterParser(/*String programParameterFile = */System.getProperty("user.dir") + System.getProperty("file.separator") +
                myInputDataAndParameterManager.getInputDirectory("dirArch.txt") + System.getProperty("file.separator") +
                myInputDataAndParameterManager.getInputParameterDirectory("dirArch.txt") + System.getProperty("file.separator") +
                myInputDataAndParameterManager.getProgramParameterFile("dirArch.txt"));
    
    public static final String delimiter = " ";
    public static final String undefined = "UNDEFINED";
    public static final String illegal = "ILLEGAL";
    public static final char illegalAAName = 'X';

    // These constants are defined by C. The can later be moved to the parameter file if decided.
    static final double CRTGaussianEpsilon = 1.0; // can be used to squeeze the Gaussian sampling interval
    static final int CRTRandomSeed = 98098;
    static final double CRTRelativeWeightOfAngles = 1; // relative weight of phirmsd+psirmsd wrt rdc rmsd

    static final double eps = 1.0E-10; //for comparing two float numbers

    static final double maximumDeviationFromMean = Math.toRadians(30.0); // the maximum deviation on either side of phi for helix and strand while computing the first peptide plane

    static final double stepSizeUsedInEstimatingFirstPeptidePlane = Math.toRadians(0.001); // 0.001 degrees

    static final double scoreTermForRdcThreshold = 6.0;

    static final double scoreTermForSheetPackingThreshold = 6.0;

    static final double clashScoreThresholdForSheetPacking = 200;

    static final double noeRmsdUpperBoundForSheetPacking = 0.7;

    static final double genericThreshold = 9999.0;

    static final double genericIllegalDouble = 99999.0;

    static final double phiScale = 1.0;
    static final double psiScale = 1.0;
    
    //static final int maxSolutionStrands = 400;
    static final int maxSolutionStrands = Integer.parseInt(__parser.__parameter_map.get("maxSolutionStrands"));

    static final double sphericalCapAngle = Math.toRadians(Double.parseDouble(__parser.__parameter_map.get("sphericalCapAngle")));

    static final int optimizeFirstPeptidePlane =
            (__parser2.__parameter_map.get("optimizeFirstPeptidePlane") != null) ?
                Integer.parseInt(__parser2.__parameter_map.get("optimizeFirstPeptidePlane")) : 2; // 0 = noOpt; 1 = optWhenRdcsAvailable; 2 = forceOptAlways => ifRdcunavialable then raise error

    static {
        if (!(optimizeFirstPeptidePlane == 0 || optimizeFirstPeptidePlane == 1 || optimizeFirstPeptidePlane == 2)) {
            System.out.println("Error: Invalid value for optimizeFirstPeptidePlane (allowed: 0, 1, 2)");
            System.exit(1);
        }
    }

    static final boolean testIfRdcsExistForBoundaryResiduesOfSse =
            (__parser2.__parameter_map.get("testIfRdcsExistForBoundaryResiduesOfSse") != null) ?
                Boolean.parseBoolean(__parser2.__parameter_map.get("testIfRdcsExistForBoundaryResiduesOfSse")) : true;

    static final boolean enforceRdcErrorIntervalEarly =
            (__parser2.__parameter_map.get("enforceRdcErrorIntervalEarly") != null) ?
                Boolean.parseBoolean(__parser2.__parameter_map.get("enforceRdcErrorIntervalEarly")) : false;

    static final boolean computeAlignmentTensor =
            (__parser2.__parameter_map.get("computeAlignmentTensor") != null) ?
                Boolean.parseBoolean(__parser2.__parameter_map.get("computeAlignmentTensor")) : true;

    static final double Syy = (__parser2.__parameter_map.get("Syy") != null) ?
                Double.parseDouble(__parser2.__parameter_map.get("Syy")) : genericIllegalDouble;

    static final double Szz = (__parser2.__parameter_map.get("Szz") != null) ?
                Double.parseDouble(__parser2.__parameter_map.get("Szz")) : genericIllegalDouble;

    public static void main(String... args) {

        myVector3D v1 = new myVector3D(new myPoint(51.939, -78.574, -10.158), new myPoint(49.893, -74.332,   2.411));

        myVector3D v2 = new myVector3D(new myPoint(52.516, -69.949,   1.042), new myPoint(53.882, -71.653,  -2.089));

        System.out.println(Math.toDegrees(Math.PI - myVector3D.angle(v1, v2)));
        System.out.println("sphericalCapAngle: " + Math.toDegrees(sphericalCapAngle));
        System.out.println("enforceRdcErrorIntervalEarly: " + enforceRdcErrorIntervalEarly);

        System.out.println("testIfRdcsExistForBoundaryResiduesOfSse: " + testIfRdcsExistForBoundaryResiduesOfSse);

         System.out.println("optimizeFirstPeptidePlane: " + optimizeFirstPeptidePlane);
    }
}
