/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2009 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, 01 December, 2009
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.0, Nov 18, 2009
 * @author        Chittaranjan Tripathy (2007-2009)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */
import java.util.*;
import java.io.*;

/**
 * Description of the class
 */
public class myNoe {
    private int __proton_A_residue_number;
    private int __proton_B_residue_number;
    private String __proton_A_residue_name = null;
    private String __proton_B_residue_name = null;
    private String __proton_A_atom_name = null;
    private String __proton_B_atom_name = null;
    private double __lower_bound;
    private double __upper_bound;

    public void swap() {
//        if (__proton_A_residue_number <= __proton_B_residue_number) {
//            return;
//        }

        // Swap residue numbers
        int temp = __proton_A_residue_number;
        __proton_A_residue_number = __proton_B_residue_number;
        __proton_B_residue_number = temp;

        // Swap residue names
        String tempS = __proton_A_residue_name;
        __proton_A_residue_name = __proton_B_residue_name;
        __proton_B_residue_name = tempS;

        // Swap atom names
        String tempN = __proton_A_atom_name;
        __proton_A_atom_name = __proton_B_atom_name;
        __proton_B_atom_name = tempN;
    }
    
    public myNoe() {
        __proton_A_residue_number = 0;
        __proton_B_residue_number = 0;        
        __lower_bound = 0.0;
        __upper_bound = 0.0;
    }

    public myNoe(final myNoe ee) {
        __proton_A_residue_number = ee.__proton_A_residue_number;
        __proton_B_residue_number = ee.__proton_B_residue_number;
        __proton_A_residue_name = ee.__proton_A_residue_name;
        __proton_B_residue_name = ee.__proton_B_residue_name;
        __proton_A_atom_name = ee.__proton_A_atom_name;
        __proton_B_atom_name = ee.__proton_B_atom_name;
        __lower_bound = ee.__lower_bound;
        __upper_bound = ee.__upper_bound;
    }

    public myNoe(int resNumA, int resNumB) {
        __proton_A_residue_number = resNumA;
        __proton_B_residue_number = resNumB;        
        __lower_bound = 0.0;
        __upper_bound = 0.0;
        //swap();
    }

    public myNoe(int resNumA, int resNumB, String res1, String res2, String atom1, String atom2, double lb, double ub) {
        __proton_A_residue_number = resNumA;
        __proton_B_residue_number = resNumB;
        __proton_A_residue_name = res1;
        __proton_B_residue_name = res2;
        __proton_A_atom_name = atom1;
        __proton_B_atom_name = atom2;
        __lower_bound = lb;
        __upper_bound = ub;
        //swap();
    }

    public myNoe(int resNumA, String atom1, int resNumB, String atom2, double lb, double ub) {
        __proton_A_residue_number = resNumA;
        __proton_B_residue_number = resNumB;
        __proton_A_atom_name = atom1;
        __proton_B_atom_name = atom2;
        __lower_bound = lb;
        __upper_bound = ub;
        //swap();
    }

    public int getResidueNoA() {
        return __proton_A_residue_number;
    }

    public int getResidueNoB() {
        return __proton_B_residue_number;
    }

    public String getResidueA() {
        return __proton_A_residue_name;
    }

    public String getResidueB() {
        return __proton_B_residue_name;
    }

    public String getAtomA() {
        return __proton_A_atom_name;
    }

    public String getAtomB() {
        return __proton_B_atom_name;
    }

    public double[] getRange() {
        return new double[]{__lower_bound, __upper_bound};
    }

    public double getLower() {
        return __lower_bound;
    }

    public double getUpper() {
        return __upper_bound;
    }
    //  set the values

    public void setDistUpper(double up) {
        __upper_bound = up;
    }

    public void setDistLower(double lw) {
        __lower_bound = lw;
    }

    @Override public String toString() {
        String desc = null;
        if (__proton_A_residue_name != null && __proton_B_residue_name != null) {
            desc = __proton_A_residue_number + "  "+ __proton_A_residue_name + "  " + __proton_A_atom_name + "  " + __proton_B_residue_number + "  " + __proton_B_residue_name + "  " + __proton_B_atom_name + "  " + __lower_bound + "  " + __upper_bound;
        } else {
            desc = __proton_A_residue_number + "  "+ __proton_A_atom_name + "  " + __proton_B_residue_number + "  " + __proton_B_atom_name + "  " + __lower_bound + "  " + __upper_bound;
        }
        return desc;
    }

    /**
     * This is overriden to make sure that one does not attempt to clone the
     * object. We have provided copy constructor for that.
     *
     * @return
     * @throws CloneNotSupportedException
     */
    @Override public Object clone() throws CloneNotSupportedException {
        throw new CloneNotSupportedException();
    }

    public static class noeComparator implements Comparator<myNoe> {
        public int compare(myNoe ee1, myNoe ee2) {
            int r1 = ee1.getResidueNoA();
            int r2 = ee2.getResidueNoA();
            if (r1 < r2) {
                return -1;
            } else if (r1 > r2) {
                return 1;
            } else {
                return 0;
            }
        }
    }

    public static class noeComparatorProtonA implements Comparator<myNoe> {
        public int compare(myNoe ee1, myNoe ee2) {            
            int r1 = ee1.getResidueNoA();
            int r2 = ee2.getResidueNoA();
            if (r1 < r2) {
                return -1;
            } else if (r1 > r2) {
                return 1;
            } else {
                return 0;
            }
        }
    }

    public static class noeComparatorProtonB implements Comparator<myNoe> {
        public int compare(myNoe ee1, myNoe ee2) {
            int r1 = ee1.getResidueNoB();
            int r2 = ee2.getResidueNoB();
            if (r1 < r2) {
                return -1;
            } else if (r1 > r2) {
                return 1;
            } else {
                return 0;
            }
        }
    }

    public void print() {
        System.out.println(this.toString());
    }
    
    public static void printNoeVector(Vector<myNoe> noeVec) {
        for (myNoe ee : noeVec) {
            System.out.println(ee.toString());
        }
    }

    /**
     * Read the noe file in Xplor format.
     *
     * @param noeFileName the file containing the dipolar coupling information
     * @return a vector of noe object
     */
    public static Vector<myNoe> parseNoeFileInXplorFormat(String noeFileName) {
        Vector<myNoe> myNoeVec = new Vector<myNoe>();

        try {
            Scanner scanner = new Scanner(new File(noeFileName));
            scanner.useDelimiter(System.getProperty("line.separator"));            
            while (scanner.hasNextLine()) {
                String line = scanner.nextLine();                
                line = line.replaceAll("\\(|\\)", " ").trim();
                //System.out.println(line);
                if (line.length() != 0) {
                    String[] words = line.split("\\s+");
                    if ( words.length >= 14 && words[0].trim().matches("assign") &&
                            words[2].matches("[0-9]+") && words[5].matches("[HNCO].*") &&
                            words[7].matches("[0-9]+") && words[10].matches("[HNCO].*")) {
                        int protonAResidueNumber = Integer.parseInt(words[2]);
                        String protonAAtomName = words[5].trim().toUpperCase();
                        if (protonAAtomName.equalsIgnoreCase("HN")) {
                            protonAAtomName = myAtomLabel.__H;
                        }
                        int protonBResidueNumber = Integer.parseInt(words[7]);
                        String protonBAtomName = words[10].trim().toUpperCase();
                        if (protonBAtomName.equalsIgnoreCase("HN")) {
                            protonBAtomName = myAtomLabel.__H;
                        }
                        //double ub = Double.parseDouble(words[11]);
                        //double lb = ub - Double.parseDouble(words[12]);
                        double ub = Double.parseDouble(words[11]) + Double.parseDouble(words[13]);
                        double lb = 1.8;//Double.parseDouble(words[11]) - Double.parseDouble(words[13]);

                        myNoe ee = new myNoe(protonAResidueNumber, protonAAtomName, protonBResidueNumber, protonBAtomName, lb, ub);
                        myNoeVec.add(ee);
                    }
                }
            }
            scanner.close();
        } catch (FileNotFoundException e) {
            System.out.println("Error: Input file " + noeFileName + " not found");
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }
        
        return myNoeVec;
    }

    /**
     * The main method tests this class.
     *
     * @param args
     */
    public static void main(String... args) {

        String userDir = System.getProperty("user.dir");
        String fileSeparator = System.getProperty("file.separator");
        String inputDirectory = userDir + fileSeparator + "input_files_1d3z" + fileSeparator + "input_data_files" + fileSeparator;
        String inputFile = inputDirectory + "noe.txt";
        System.out.println("inputFile: " + inputFile);

        Vector<myNoe> myNoeVec = parseNoeFileInXplorFormat(inputFile);

        System.out.println("Printing the noes");
        for (myNoe ee : myNoeVec) {
            ee.print();
        }

        // Testing myNewPacker here.
        String strand1File = userDir + fileSeparator + /*"btt1.pdb";//*/"ubqStrand111.pdb";
        String strand2File = userDir + fileSeparator + /*"btt2.pdb";//*/"ubqStrand222.pdb";

        myProtein strand1 = null;
        myProtein strand2 = null;

        getFirstStrand:
        {
            Vector<myProtein> vp = new Vector<myProtein>();
            myPdbParser pParser = new myPdbParser(strand1File);
            while (pParser.hasNextProtein()) {
                vp.add(pParser.nextProtein());
            }
            myProtein pp = vp.elementAt(0);
            strand1 = pp;
        }

        getSecondStrand:
        {
            Vector<myProtein> vp = new Vector<myProtein>();
            myPdbParser pParser = new myPdbParser(strand2File);
            while (pParser.hasNextProtein()) {
                vp.add(pParser.nextProtein());
            }
            myProtein pp = vp.elementAt(0);
            strand2 = pp;

            strand2.translate(100, 200, 374);
        }

        strand1.print();
        strand2.print();

        myNewPacker pk = new myNewPacker();
        
        myTriple<myProtein, myProtein, Double> score = pk./*centerFit__*/__packUsingNoes__(strand1, strand2, myNoeVec, null);

        

//        myTriple<myProtein, myProtein, Double> scr = pk.centerFit__(strand1, strand2, myNoeVec);
//        scr.second().print();
//        scr.first().print();

/*
            myNoe[] noes = new myNoe[] {
                // new ones
                new myNoe(3, "H", 15, "H", 2.79, 2.79),
                new myNoe(4, "H", 15, "H", 4.99, 4.99),
                //new myNoe(5, "H", 13, "H", 2.71, 2.71),
                new myNoe(6, "H", 13, "H", 4.78, 4.78),
                new myNoe(5, "H", 14, "HA", 3.79, 3.79),

                new myNoe(4, "H", 67, "H", 3.62, 3.62),
                new myNoe(6, "H", 68, "H", 4.81, 4.81),
               // new myNoe(6, "H", 69, "H", 3.59, 3.59),
                new myNoe(4, "H", 66, "H", 4.82, 4.82),
                new myNoe(5, "HA", 68, "HA", 4.15, 4.15),

                new myNoe(44, "HA", 68, "H", 4.83, 4.83),
                new myNoe(43, "HA", 69, "H", 5.17, 5.17),
                new myNoe(43, "HA", 69, "HA", 2.82, 2.82),
                new myNoe(42, "H", 70, "H", 2.78, 2.78),

                new myNoe(45, "H", 49, "H", 5.00, 5.00),
                new myNoe(44, "HA", 49, "HA", 2.52, 2.52),
                new myNoe(43, "H", 50, "H", 3.96, 3.96),
                //new myNoe(, "H", , "H", , ),
            };
        myNoeVec = new Vector<myNoe>(Arrays.asList(noes));
        myTriple<myProtein, myProtein, Double> scr = pk.centerFit__(strand1, strand2, myNoeVec);
        //scr.second().print();
        //scr.first().print();
*/
    }
}

