/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2009 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, 01 December, 2009
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.0, Nov 18, 2009
 * @author        Chittaranjan Tripathy (2007-2009)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */
import java.util.Formatter;
import java.util.Locale;

/**
 * Description of the class
 */
public class myPdbResidueRow {
    // Data structure to parse a row representing an atom in the PDB file.
    //typedef int index_type;
    //typedef string element_type;

    String __record_name = myMiscConstants.undefined; // Record name, e.g. ATOM.
    int __serial_number; // Atom serial number or Index in the PDB file.
    String __atom_name = myMiscConstants.undefined; // Atom name upto 4 characters in IUPAC format.
    char __alt_loc_indicator; // Alternate location indicator.
    String __residue_name = myMiscConstants.undefined; // Residue name in 3 letter IUPAC format.
    char __chain_id; // Chain ID.
    int __residue_number; // Residue sequence number.
    char __insertion_code; // Insertion code for residue.
    myPoint __coordinates; // x, y and z coordinates of the atom.
    double __occupancy; // Occupancy.
    double __temperature_factor; // Temperature factor.
    String __segment_id = myMiscConstants.undefined; // Segment ID.
    String __element = myMiscConstants.undefined; // Element Symbol as in the Periodic Table.
    String __charge = myMiscConstants.undefined; // Charge.

    /**
     * Set the fields. This method can be thought as a copy constructor.
     *
     * @param r an instance of this type
     */
    private void assignFields(final myPdbResidueRow r) {
        __record_name = r.__record_name;
        __serial_number = r.__serial_number;
        __atom_name = r.__atom_name;
        __alt_loc_indicator = r.__alt_loc_indicator;
        __residue_name = r.__residue_name;
        __chain_id = r.__chain_id;
        __residue_number = r.__residue_number;
        __insertion_code = r.__insertion_code;
        __coordinates = new myPoint(r.__coordinates);
        __occupancy = r.__occupancy;
        __temperature_factor = r.__temperature_factor;
        __segment_id = r.__segment_id;
        __element = r.__element;
        __charge = r.__charge;
    }

    /**
     * Default constructor that constructs a default object.
     */
    private myPdbResidueRow() {
    }

    /**
     * Construct an object of PDB row type from the given set of arguments.
     *
     * @param record_name the name of the record
     * @param serial_number the serial number
     * @param atom_name the name of the atom
     * @param alt_loc_indicator the alternative location indicator
     * @param residue_name the name of the residue
     * @param chain_id the chain id
     * @param residue_number the residue number
     * @param insertion_code the insertion code
     * @param coordinates the coordinates of the atom
     * @param occupancy the occupancy factor
     * @param temperature_factor the temperature factor
     * @param segment_id the segment id
     * @param element the element type as in the Periodic table
     * @param charge the charge of the atom
     */
    public myPdbResidueRow(String record_name, int serial_number, String atom_name, char alt_loc_indicator, String residue_name,
            char chain_id, int residue_number, char insertion_code, final myPoint coordinates, double occupancy,
            double temperature_factor, String segment_id, String element, String charge) {
        __record_name = record_name;
        __serial_number = serial_number;
        __atom_name = atom_name;
        __alt_loc_indicator = alt_loc_indicator;
        __residue_name = residue_name;
        __chain_id = chain_id;
        __residue_number = residue_number;
        __insertion_code = insertion_code;
        __coordinates = new myPoint(coordinates);
        __occupancy = occupancy;
        __temperature_factor = temperature_factor;
        __segment_id = segment_id;
        __element = element;
        __charge = charge;
    }

    /**
     * Construct an object of PDB row type from a string representing a PDB row.
     *
     * @param row a string representing a PDB row
     */
    public myPdbResidueRow(String row) {
        __record_name = row.substring(0, 6).trim();
        __serial_number = Integer.parseInt(row.substring(6, 11).trim());
        __atom_name = row.substring(12, 16).trim();
        __alt_loc_indicator = row.charAt(16);
        __residue_name = row.substring(17, 20).trim();
        __chain_id = row.charAt(21);
        __residue_number = Integer.parseInt(row.substring(22, 26).trim());
        __insertion_code = row.charAt(26);
        __coordinates = new myPoint(Double.parseDouble(row.substring(30, 38).trim()),
                Double.parseDouble(row.substring(38, 46).trim()), Double.parseDouble(row.substring(46, 54).trim()));
        __occupancy = Double.parseDouble(row.substring(54, 60).trim());

        if (row.length() > 62) {
            __temperature_factor = Double.parseDouble(row.substring(60, 66).trim());
        }

        if (row.length() > 78) {
            __segment_id = row.substring(72, 76).trim();
            __element = row.substring(76, 78).trim();            
            __charge = row.substring(78, 80).trim();
        }
    }

    /**
     * Copy constructor that constructs new object from an object passed as parameter.
     *
     * @param r the object which is to be cloned
     */
    public myPdbResidueRow(final myPdbResidueRow r) {
        assignFields(r);
    }

    /**
     * Return the PDB row string representation of the object.
     *
     * @return the PDB row string representation of the object
     */
    public String getResidueRow() {
        StringBuilder sb = new StringBuilder(80);
        Formatter formatter = new Formatter(sb, Locale.US);

        formatter.format("%-6s", __record_name);
        formatter.format("%5d", __serial_number);
        formatter.format("%c", ' '); // This is a blank space according to the PDB format.
        if (__record_name.equals("HETATM")) {
            formatter.format("%-4s", __atom_name);
        } else { // record is an ATOM
            if (__atom_name.length() < 4) {
                formatter.format("%c%-3s", ' ', __atom_name);
            } else {
                formatter.format("%-4s", __atom_name);
            }
        }    
        formatter.format("%c%3s %c%4d%c%s", __alt_loc_indicator, __residue_name, __chain_id, __residue_number, __insertion_code, "   ");
        formatter.format("%8.3f%8.3f%8.3f", __coordinates.getX(), __coordinates.getY(), __coordinates.getZ());
        formatter.format("%6.2f%6.2f%s", __occupancy, __temperature_factor, "      ");
        formatter.format("%-4s%2s%2s", __segment_id, __element, __charge);        

        return sb.toString();
    }

    /**
     * Print the row in PDB format.
     */
    public void print() {
        System.out.println(getResidueRow());
    }

    /**
     * Return the residue name of the row which represents an ATOM or HETATM.
     *
     * @return the residue name of the row which represents an ATOM or HETATM
     */
    public String getResidueName() {
        return __residue_name;
    }

    /**
     * Return the residue name of the row which represents an ATOM or HETATM.
     * 
     * @return the residue name of the row which represents an ATOM or HETATM
     */
    public int getResidueNumber() {
        return __residue_number;
    }    

    /**
     * Return a newly constructed atom with the parameters supplied as arguments.
     *
     * @return a newly constructed atom with the parameters supplied as arguments
     */
    public myAtom toAtom() {
        return new myAtom(__record_name, __serial_number, __atom_name, __residue_name,
                __chain_id, __residue_number, __coordinates, __occupancy, 
                __temperature_factor, __segment_id, __element, __charge);  
    }

    /**
     * The main method tests this class.
     *
     * @param args
     */
    public static void main(String[] args) {
        myPdbResidueRow pr = new myPdbResidueRow();
        pr.print();
    }
}

