/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2009 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, 01 December, 2009
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.0, Nov 18, 2009
 * @author        Chittaranjan Tripathy (2007-2009)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 * @author        Lincong Wang (2001-2005)
 * @email         wlincong@cs.dartmouth.edu
 * @organization  Dartmouth College
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */

/**
 * Description of the class
 */

/*
 *  A simple class for processing information about Hydrogen bonds
 * @version 0.1
 * @author Lincong Wang
 * Wrote by Lincong Wang At Bruce's Lab at Dartmouth College
 */
import java.io. *;
import java.util. *;
 
public class Hbond {
    private int residueNoA;   //the Sequence No of one of the two partners of the NOE
    private int residueNoB;   //the Sequence No of Another partner of the NOE
    private String atomA;     //the name of Atom A
    private String atomB;     //the name of Atom B
    private String id;        //an id assigned to the H-bond
    private String type;      //the name of Atom B

    /** Constructors */
    public Hbond() {
        residueNoA = 0;
        residueNoB = 0;
        atomA = null;
        atomB = null;
        id    = null;        
	type  = null;
    }

   public Hbond(int Ano, String A, int Bno,  String B,  String id, String t) {
        residueNoA = Ano;
        residueNoB = Bno;
        this.atomA = A;
        this.atomB = B;
        this.id   = id;
        this.type = t;
    }
 
    public int getResidueNoA() {
        return residueNoA;
    }
    public int getResidueNoB() {
        return residueNoB;
    }
    public String getAtomA()   {
        return atomA;
    }
    public String getAtomB()   {
        return atomB;
    }
    public String getID()   {
        return id;
    }
    public String getType()   {
        return type;
    }
  
    public String toString()    {
        String desc = residueNoA + atomA + "__"+residueNoB+atomB+"  "+id+"   "+type;
        return desc;
    }
    protected Object clone(){
        try{
            Object s = super.clone();     // clone the stack
            return s;                           // return the clone
        } catch (CloneNotSupportedException e)  {
            throw new InternalError();
        }
    }
 
    public static class HbondComparatorA implements Comparator{
        public int compare(Object o1, Object o2){
            Hbond n1 = (Hbond)o1;
            Hbond n2 = (Hbond)o2;
            int d1 = n1.getResidueNoA();
            int d2 = n2.getResidueNoA();
            if (d1 < d2)
                return -1;
            else if (d1 > d2)
                return 1;
            else return 0;
        }
    }
 
    public static class HbondComparatorB implements Comparator{
        public int compare(Object o1, Object o2){
            Hbond n1 = (Hbond)o1;
            Hbond n2 = (Hbond)o2;
            int d1 = n1.getResidueNoB();
            int d2 = n2.getResidueNoB();
            if (d1 < d2)
                return -1;
            else if (d1 > d2)
                return 1;
            else return 0;
        }
    }
    /*
     * A very simple reader for H-bond data: see the attched file for the format
     @param hbFile the name of the file
     @return a vector of Hbond object
     */
    public Vector HbondReader(String hbFile){
        Vector inputs = new Vector();
        int noA, noB;
        String atomA, atomB, id, type;
        try{
            StreamTokenizer in =
                new StreamTokenizer(
                                    new FileReader(hbFile));
            while( in.nextToken() != StreamTokenizer.TT_EOF ){ //ignore the comments
                if (in.ttype == StreamTokenizer.TT_WORD)
                    ;
                else if (in.ttype == StreamTokenizer.TT_NUMBER){
                    noA = (int)in.nval;
                    in.nextToken();
                    atomA = in.sval;
                    in.nextToken();
                    noB = (int)in.nval;
                    in.nextToken();
                    atomB = in.sval;
                    in.nextToken();
                    id = in.sval;
                    in.nextToken();
                    type = in.sval;
                    inputs.add(new Hbond(noA, atomA, noB, atomB, id, type));
                }
            }
        }catch (FileNotFoundException e){
            System.out.println("File not found: " + hbFile);
        }catch (IOException e) {
            System.out.println("IOException: the stack trace is:");
            e.printStackTrace();
        }
        return inputs;
    }

    public Vector hbOfTwoStrands(Vector hbVec, String ids){
	String id;        //an id assigned to the H-bond
	Hbond hb = new  Hbond();
	Vector hb2Strands = new Vector();
	for (int i = 0; i < hbVec.size(); i++){
	    hb =  (Hbond)hbVec.elementAt(i);
	    if (hb.getID().equals(ids))
		hb2Strands.add(hb);
	}
	return hb2Strands;
    }
}
