/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2009 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, 01 December, 2009
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.0, Nov 18, 2009
 * @author        Chittaranjan Tripathy (2007-2009)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */
import java.util.*;
import java.io.*;

/**
 * Description of the class
 */
public class RDCAnalytic {

    public static void main(String... args) {
        long startTime = System.currentTimeMillis();

        //System.out.println("maxSolutionStrands: " + myMiscConstants.maxSolutionStrands);
        //System.exit(1);

        myInputDataAndParameterManager thisParameterManager = new myInputDataAndParameterManager("dirArch.txt"); // The file name should not be changed else RDC Constants File might also break as this file name is hardwired there
        String inputFile = thisParameterManager.getSeqFile();

        myProteinSequence thisProteinSeq = new myProteinSequence(inputFile);
        System.out.println(thisProteinSeq);

        myDipolarCouplingTable thisRdcCsaTable = thisParameterManager.getDipolarCouplingTable();
        thisRdcCsaTable.print();

        Vector<mySseInfo> idVec = thisParameterManager.parseSseIdFile();
        System.out.println("Printing the SSE boundaries");
        for (int i = 0; i < idVec.size(); i++) {
            mySseInfo id2ary = idVec.elementAt(i);
            System.out.println(id2ary);
        }

        Vector<Vector<myTriple<mySseInfo, mySseInfo, Boolean>>> betaSheetSpecs = thisParameterManager.parseSseIdFileAndReturnSheetSpecifications();
        System.out.println("Beta sheet specifications: ");
        for (Vector<myTriple<mySseInfo, mySseInfo, Boolean>> bsp : betaSheetSpecs) {
            System.out.println("Sheet: ");
            for (myTriple<mySseInfo, mySseInfo, Boolean> sp : bsp) {
                System.out.println(sp.first() + " " + sp.second() + " " + sp.third());
            }
        }

        mySseInfo sseForAtComputation = thisParameterManager.parseSseIdFileAndReturnSseForAlignmentTensorComputation();
        System.out.println("Printing the SSE to be used for alignment tensor computation");
        System.out.println(sseForAtComputation);

        Map<String, Map<String, String>> paramMaps = thisParameterManager.parseParameterFile();
        System.out.println("Printing the Parameter map");
        for (Map.Entry<String, Map<String, String>> entry : paramMaps.entrySet()) {            
            for (Map.Entry<String, String> entry2 : entry.getValue().entrySet()) {
                System.out.println(entry2.getKey() + "    " + entry2.getValue());
            }
            System.out.println();
        }

        ReadNoes: {
            Vector<myNoe> thisNoeVec = null; // It actually stores noes or hbond info
            System.out.println("Printing the NOEs");
            // Read the NOEs
            NoeReader: {
                String inputNoeFile = thisParameterManager.getNoeFile();
                thisNoeVec = myNoe.parseNoeFileInXplorFormat(inputNoeFile); //new Vector<myNoe>(Arrays.asList(noes));
                myNoe.printNoeVector(thisNoeVec);
            }
        }   
//System.exit(1);
   
        File file = new File("phiPsiSolutionSets.txt");
        if (file.exists()) {
            file.delete();
        }

        file = new File("bestFragment.pdb");
        if (file.exists()) {
            file.delete();
        }

        file = new File("mostRecentFragment.pdb");
        if (file.exists()) {
            file.delete();
        }

        file = new File("bestSetOfFragments.pdb");
        if (file.exists()) {
            file.delete();
        }

        Vector<myProtein> computedSseFragments = new Vector<myProtein>();        

        myBackboneModeler bbModeler =  myBackboneModeler.getInstance();

        // Select the RDCs to be used to compute the structure.
        String mediumName = "medium1";
        Vector<myDipolarCoupling.Type> typesOfRdcsToBeUsedForRefinement = new Vector<myDipolarCoupling.Type>();
        typesOfRdcsToBeUsedForRefinement.add(myDipolarCoupling.Type.CA_HA);
        typesOfRdcsToBeUsedForRefinement.add(myDipolarCoupling.Type.N_HN);

        Vector<myDipolarCoupling.Type> typesOfRdcsToBeUsedForAnalyticSolution = new Vector<myDipolarCoupling.Type>();
        typesOfRdcsToBeUsedForAnalyticSolution.add(myDipolarCoupling.Type.CA_HA);
        typesOfRdcsToBeUsedForAnalyticSolution.add(myDipolarCoupling.Type.N_HN);
        
        double[] saupe = new double[2]; // for returning Syy and Szz

        boolean computeAT = myMiscConstants.computeAlignmentTensor;
        
        if (computeAT) {
            computationOfAlignmentTensor:
            {
                // Estimate the alignment tensor components Syy and Szz from a helix.
                // At the end of AT computation we will have a helix computed and
                // rotated to the POF along with the alignment tensor components.

                // Get the parameters to be used for AT computation
                Map<String, String> paramMapForAT = paramMaps.get("parametersForAlignmentTensorComputation"); // Parameters for computing the alignment tensor

                int dfsCycle = 0;
                if (paramMapForAT.get("numberOfSearchTrees").equalsIgnoreCase("auto")) {
                    // Inteligently pick the number of search trees you want to enumerate
                    int numberOfResiduesInThisSse = Math.abs(sseForAtComputation.getEndResidueNumber() - sseForAtComputation.getBeginResidueNumber() + 1);
                    if (numberOfResiduesInThisSse < 10) {
                        dfsCycle = numberOfResiduesInThisSse * 100000;
                    } else {
                        dfsCycle = numberOfResiduesInThisSse * 100000; //1000000; // if the number of residues is >= 10 then cap this to 1 million in auto mode
                    }
                } else {
                    dfsCycle = Integer.parseInt(paramMapForAT.get("numberOfSearchTrees"));
                }

                //int dfsCycle = Integer.parseInt(paramMapForAT.get("numberOfSearchTrees"));
                int refineCycle = Integer.parseInt(paramMapForAT.get("numberOfOptimizationsOfPP1"));
                double weight4Angles = new Double(paramMapForAT.get("weightForRmsDihedralDeviationTerm")).doubleValue();
                System.out.println("Parameters for alignment tensor computation: " + paramMapForAT.toString());

                double resolution = 1.0; // for alignment tensor actually we don't use this
                boolean printResults = true; // for alignment tensor actually we don't use this

                myProtein thisSse = bbModeler.computeAlignmentTensor(thisRdcCsaTable, mediumName,
                        typesOfRdcsToBeUsedForRefinement, typesOfRdcsToBeUsedForAnalyticSolution,
                        sseForAtComputation, refineCycle, dfsCycle, weight4Angles, resolution, printResults, saupe);
                for (myResidue r : thisSse) {
                    r.mutate(thisProteinSeq.getResidueType(r.getResidueNumber()));
                }
                computedSseFragments.add(thisSse);
                computedSseFragments.lastElement().print("bestSetOfFragments.pdb", true);
                
                // Code to print the AT helix in a separate file
                file = new File("at.pdb");
                if (file.exists()) {
                    file.delete();
                }
                computedSseFragments.lastElement().print("at.pdb", false);
            }
        } else { // load the Syy and Szz values from the parameter file
            saupe[0] = myMiscConstants.Syy;
            saupe[1] = myMiscConstants.Szz;

            if (saupe[0] == myMiscConstants.genericIllegalDouble || saupe[1] == myMiscConstants.genericIllegalDouble) {
                System.out.println("Error: The alignment tensor values (Syy: " + saupe[0] + ", Szz: " + saupe[1] + ") are not set properly");
                System.exit(1);
            }
        }
        //System.exit(1);

        // the alingnment tensor components
        double Syy = saupe[0]; double Szz = saupe[1];
        System.out.println("Syy: " + Syy + "  Szz: " + Szz);

        ComputationOfAllHelices:
        {
            int helixCounter = 1;

            // Use the alignment tensor computed above to compute rest of the SSEs.
            for (int i = 0; i < idVec.size(); i++) {
                mySseInfo thisSseId = idVec.elementAt(i);

                if (thisSseId.isHelix()) {
                    // do everything below
                } else if (thisSseId.isStrand()) {
                    continue; // means do not compute a beta strand, as it is going to be done in the BLOCK below.
                    // get the parameters for the strand
                } else {
                    System.out.println("Error: invalid SSE");
                    System.exit(1);
                }

                Map<String, String> thisParameterMap = paramMaps.get("parametersForHelixComputation"); // Parameters for computing the helices
                int thisDfsCycle = 0;
                if (thisParameterMap.get("numberOfSearchTrees").equalsIgnoreCase("auto")) {
                    // Inteligently pick the number of search trees you want to enumerate
                    int numberOfResiduesInThisSse = Math.abs(thisSseId.getEndResidueNumber() - thisSseId.getBeginResidueNumber() + 1);
                    if (numberOfResiduesInThisSse < 10) {
                        thisDfsCycle = numberOfResiduesInThisSse * 100000;
                    } else {
                        thisDfsCycle = numberOfResiduesInThisSse * 100000; // 1000000; // if the number of residues is >= 10 then cap this to 1 million in auto mode
                    }
                } else {
                    thisDfsCycle = Integer.parseInt(thisParameterMap.get("numberOfSearchTrees"));
                }
                int thisRefineCycle = Integer.parseInt(thisParameterMap.get("numberOfOptimizationsOfPP1"));
                double thisWeight4Angles = new Double(thisParameterMap.get("weightForRmsDihedralDeviationTerm")).doubleValue();
                double thisResolution = new Double(thisParameterMap.get("gridResolutionInDegrees")).doubleValue();
                System.out.println("Parameters for " + thisSseId.toString() + " computation: " + thisParameterMap.toString());

                boolean thisPrintResults = true; // To be used in future

                myProtein thisSse = bbModeler.computeSseGivenAlignmentTensorAndRdcs(thisRdcCsaTable, mediumName, typesOfRdcsToBeUsedForRefinement, typesOfRdcsToBeUsedForAnalyticSolution,
                        thisSseId, Syy, Szz, thisRefineCycle, thisDfsCycle, thisWeight4Angles, thisResolution, thisPrintResults);
                for (myResidue r : thisSse) {
                    r.mutate(thisProteinSeq.getResidueType(r.getResidueNumber()));
                }
                computedSseFragments.add(thisSse);

                System.out.println("**************Begin Printing the Protein Computed Thus Far****************");
                for (myProtein p : computedSseFragments) {
                    p.print();
                }                
                computedSseFragments.lastElement().print("bestSetOfFragments.pdb", true);
                System.out.println("***************End Printing the Protein Computed Thus Far*****************");

                // Code to print the helices in separate files
                String helixPdbFileName = "helix" + helixCounter + ".pdb";
                helixCounter++;
                file = new File(helixPdbFileName);
                if (file.exists()) {
                    file.delete();
                }
                computedSseFragments.lastElement().print(helixPdbFileName, false);
            }
        }
        //System.exit(1);

        ComputationOfAllSheets:
        {
            int sheetCounter = 1;

            for (int i = 0; i < betaSheetSpecs.size(); i++) {
                Map<String, String> thisParameterMap = paramMaps.get("parametersForStrandComputation"); // Parameters for computing the sheets

                int thisDfsCycle = 0;
                if (thisParameterMap.get("numberOfSearchTrees").equalsIgnoreCase("auto")) {
                    // Inteligently pick the number of search trees you want to enumerate
                    thisDfsCycle = 10000; // scales well
                } else {
                    thisDfsCycle = Integer.parseInt(thisParameterMap.get("numberOfSearchTrees"));
                }

                //int thisDfsCycle = Integer.parseInt(thisParameterMap.get("numberOfSearchTrees"));
                int thisRefineCycle = Integer.parseInt(thisParameterMap.get("numberOfOptimizationsOfPP1"));
                double thisWeight4Angles = new Double(thisParameterMap.get("weightForRmsDihedralDeviationTerm")).doubleValue();
                double thisResolution = new Double(thisParameterMap.get("gridResolutionInDegrees")).doubleValue();
                System.out.println("Parameters for sheet computation: " + thisParameterMap.toString());

                boolean thisPrintResults = true; // To be used in future
                double hbWeight = 1.0; // This is not used at this point of time, and will be used in future

                Vector<myNoe> thisNoeVec = null; // It actually stores noes or hbond info

                // Read the NOEs
                NoeReader:
                {
                    String inputNoeFile = thisParameterManager.getNoeFile();
                    thisNoeVec = myNoe.parseNoeFileInXplorFormat(inputNoeFile); //new Vector<myNoe>(Arrays.asList(noes));
                    myNoe.printNoeVector(thisNoeVec);
                }

                Vector<myTriple<mySseInfo, mySseInfo, Boolean>> thisBetaSheetSseIds = betaSheetSpecs.elementAt(i);

                myProtein thisSse = bbModeler.computeBetaSheetGivenAlignmentTensorAndRdcs(thisRdcCsaTable, mediumName, typesOfRdcsToBeUsedForRefinement,
                        typesOfRdcsToBeUsedForAnalyticSolution, /*thisBetaSheetSseIds,*/ Syy, Szz, thisRefineCycle, thisDfsCycle, thisWeight4Angles,
                        thisResolution, thisPrintResults, hbWeight, thisNoeVec, thisBetaSheetSseIds);
                computedSseFragments.add(thisSse);

                System.out.println("**************Begin Printing the Protein Computed Thus Far****************");
                for (myProtein p : computedSseFragments) {
                    p.print();
                }
                computedSseFragments.lastElement().print("bestSetOfFragments.pdb", true);
                System.out.println("***************End Printing the Protein Computed Thus Far*****************");

                // Code to print the helices in separate files
                String sheetPdbFileName = "sheet" + sheetCounter + ".pdb";
                sheetCounter++;
                file = new File(sheetPdbFileName);
                if (file.exists()) {
                    file.delete();
                }
                myProtein sh = myProtein.cloneAndSortResidues(computedSseFragments.lastElement());
                sh.print(sheetPdbFileName, false);
            }
        }

        // Finally, print the vector of SSEs.
        System.out.println("\n\n");
        System.out.println("******Printing the set of computed fragments********");
        System.out.println("****************************************************");
        for (myProtein p : computedSseFragments) {
            p.print();
        }

        long endTime = System.currentTimeMillis();
        double totalTime = (double) ((endTime - startTime) / 60000.0); //in minutes
        System.out.println("Time= " + totalTime + " minutes");
    }
}

