/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2009 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, 01 December, 2009
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.0, Nov 18, 2009
 * @author        Chittaranjan Tripathy (2007-2009)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */
import java.util.*;

/**
 * Description of the class
 */
public class myAtom implements Comparable<myAtom> {

    /**
     * The fields defined below are for the corresponding columns in a PDB row.
     * Taken from: http://www.wwpdb.org/documentation/format23/sect9.html
     *
     * COLUMNS      DATA TYPE        FIELD      DEFINITION
     * ------------------------------------------------------
     * 1 -  6      Record name      "ATOM    "
     * 7 - 11      Integer          serial     Atom serial number.
     * 13 - 16      Atom             name       Atom name.
     * 17           Character        altLoc     Alternate location indicator.
     * 18 - 20      Residue name     resName    Residue name.
     * 22           Character        chainID    Chain identifier.
     * 23 - 26      Integer          resSeq     Residue sequence number.
     * 27           AChar            iCode      Code for insertion of residues.
     * 31 - 38      Real(8.3)        x          Orthogonal coordinates for X in Angstroms
     * 39 - 46      Real(8.3)        y          Orthogonal coordinates for Y in Angstroms
     * 47 - 54      Real(8.3)        z          Orthogonal coordinates for Z in Angstroms
     * 55 - 60      Real(6.2)        occupancy  Occupancy.
     * 61 - 66      Real(6.2)        tempFactor Temperature factor.
     * 77 - 78      LString(2)       element    Element symbol, right-justified.
     * 79 - 80      LString(2)       charge     Charge on the atom.
     */
    protected String __record_name = myMiscConstants.undefined; // Record name, e.g. ATOM, HETATM.
    protected int __serial_number = 0; // Atom serial number or Index in the PDB file.
    protected String __atom_name = myMiscConstants.undefined;	// Atom name upto 4 characters in IUPAC format.
    char __alternate_location_indicator = ' '; // Alternate locations of the same atom.
    protected String __residue_name = myMiscConstants.undefined; // Residue name in 3 letter IUPAC format.
    protected char __chain_id = ' ';
    protected int __residue_number = 0; // Residue sequence number.
    char __insertion_code = ' '; // Code for insertion of residues.
    protected myPoint __coordinates = new myPoint(0, 0, 0); // x, y and z coordinates of the atom.
    protected double __occupancy = 1.0;
    protected double __temperature_factor = 0.0;
    protected String __segment_id = myMiscConstants.undefined;
    protected String __element = myMiscConstants.undefined; // Element Symbol as in the Periodic Table.
    protected String __charge = myMiscConstants.undefined;
    
    /**
     * Other atom properties that are usually not there in PDB.
     */
    protected String __force_field_type = myMiscConstants.undefined; // the force field type symbol
    protected int __atomic_number = -1;
    protected double __vdw_radius = -1000000;
    protected double __atomic_mass = -1;
    
    /**
     * We use a set to hold the connectivity (bond) information. This is
     * interpreted as an adjacency list of atoms. Using a set ensures that we
     * do not have duplicate information contained in the list. In other words,
     * we stick to the graph (instead of a multigraph) representation of a
     * protein.
     */
    protected Set<myAtom> __adjacency_list = new TreeSet<myAtom>(); //This ensures that no duplicates are there.

    /**
     * Default constructor that constructs the default atom.
     */    
    public myAtom() {
    }

    /**
     * Construct an atom given the PDB atom parameters.
     * 
     * @param recName name of the record (e.g., ATOM, HETATM, etc.)
     * @param serialNum the serial number of the atom
     * @param atomName the atom name
     * @param resName the name of the residue
     * @param chainId chain identifier
     * @param resNum residue number
     * @param coordinates the x, y, z coordinates of the atom
     * @param occupancy occupancy
     * @param tempFactor temperature factor
     * @param segmentId segment id
     * @param element the element type of the atom
     * @param charge charge of the atom
     */
    public myAtom(String recName, int serialNum, String atomName, String resName, char chainId,
            int resNum, myPoint coordinates, double occupancy, double tempFactor,
            String segmentId, String element, String charge) {
        __record_name = recName;
        __serial_number = serialNum;
        __atom_name = atomName;
        __residue_name = resName;
        __chain_id = chainId;
        __residue_number = resNum;
        __coordinates = new myPoint(coordinates);
        __occupancy = occupancy;
        __temperature_factor = tempFactor;
        __segment_id = segmentId;
        __element = element;
        __charge = charge;
        setAtomProperties();
    }

    /**
     * Construct an atom with the atom name specified in the argument.
     *
     * @param atomName the name of the atom
     */
    public myAtom(String atomName) {
        __atom_name = atomName;
        setAtomProperties();
    }

    /**
     * Construct an atom with the atom name and the coordinates.
     *
     * @param atomName the name of the atom
     * @param coordinates the coordinates of the atom
     */
    public myAtom(String atomName, myPoint coordinates) {
        __atom_name = atomName;
        __coordinates = new myPoint(coordinates);
        setAtomProperties();
    }

    /**
     * A copy constructor which takes an instance of an atom and constructs
     * another atom with the same parameters.
     *
     * @param r the atom to be cloned
     */
    public myAtom(final myAtom r) {
        __record_name = r.__record_name;
        __serial_number = r.__serial_number;
        __atom_name = r.__atom_name;
        __residue_name = r.__residue_name;
        __chain_id = r.__chain_id;
        __residue_number = r.__residue_number;
        __coordinates = new myPoint(r.__coordinates);
        __occupancy = r.__occupancy;
        __temperature_factor = r.__temperature_factor;
        __segment_id = r.__segment_id;
        __element = r.__element;
        __charge = r.__charge;

        __force_field_type = r.__force_field_type;

        __atomic_number = r.__atomic_number;
        __vdw_radius = r.__vdw_radius;
        __atomic_mass = r.__atomic_mass;
    }

    /**
     * Compares if this atom has the same name as the atom in the argument.
     *
     * @param a this atom
     * @return the value <code>0</code> if the argument string is equal to
     * this string; a value less than <code>0</code> if this string
     * is lexicographically less than the string argument; and a
     * value greater than <code>0</code> if this string is
     * lexicographically greater than the string argument.
     */
    public int compareTo(myAtom a) {
        return __atom_name.compareTo(a.__atom_name);
    }

    /**
     * Compares if the two atom has the same name, the same residue number, and
     * the same coordinates.
     *
     * @param a the atom that is being compared to this atom
     * @return true if the comparison succeeds
     */
    public boolean isEqualTo(myAtom a) {
        if ((__atom_name.compareTo(a.__atom_name) == 0) && (__residue_name.compareTo(a.__residue_name) == 0) &&
                (__residue_number == a.__residue_number) && (__coordinates.compareTo(a.__coordinates) == 0)) {
            return true;
        }
        return false;
    }

    /**
     * Return the record name (e.g., ATOM, HETATM, etc.).
     * 
     * @return the record name
     */
    public String getRecordName() {
        return __record_name;
    }

    /**
     * Modify the record name (e.g., ATOM, HETATM, etc.).
     * 
     * @param recName
     */
    public void setRecordName(String recName) {
        __record_name = recName;
    }

    /**
     *
     * @return
     */
    public int getSerialNumber() {
        return __serial_number;
    }

    /**
     *
     * @param serialNum
     */
    public void setSerialNumber(int serialNum) {
        __serial_number = serialNum;
    }

    /**
     *  Return the atom name in 4 letter IUPAC format.
     */
    public String getAtomName() {
        return __atom_name;
    }

    /**
     * Set the atom name.
     */
    void setAtomName(String atomName) {
        __atom_name = atomName;
        setAtomProperties();
    }

    /**
     *
     * @return alternate location indicator
     */
    public char getAlternateLocationIndicator() {
        return __alternate_location_indicator;
    }

    /**
     *
     * @param ch alternate location indicator
     */
    public void setAlternateLocationIndicator(char ch) {
        __alternate_location_indicator = ch;
    }

    /**
     * @return the residue name of the residue to which this atom belongs
     */
    public String getResidueName() {
        return __residue_name;
    }

    /**
     * Set the residue name.
     */
    public void setResidueName(String resName) {
        __residue_name = resName;
    }

    /**
     *
     * @return chain id.
     */
    public char getChainId() {
        return __chain_id;
    }

    /**
     *
     * @param ch
     */
    public void setChainId(char ch) {
        __chain_id = ch;
    }

    /**
     *
     * @return the residue number of the residue to which this atom belongs.
     */
    public int getResidueNumber() {
        return __residue_number;
    }

    /**
     * Set the residue number
     * @param resNum
     */
    public void setResidueNumber(int resNum) {
        __residue_number = resNum;
    }

    /**
     *
     * @return insertion code
     */
    public char getInsertionCode() {
        return __insertion_code;
    }

    /**
     *
     * @param ch insertion code
     */
    public void setInsertionCode(char ch) {
        __insertion_code = ch;
    }

    /**
     *
     * @return the point (x, y, z) representing the position of the atom.
     */
    public myPoint getCoordinates() {
        return __coordinates;
    }

    /**
     * Set the coordinates.
     * @param p
     */
    public void setCoordinates(final myPoint p) {
        __coordinates = new myPoint(p);
    }

    /**
     *
     * @return the occupancy value.
     */
    public double getOccupancy() {
        return __occupancy;
    }

    /**
     * Set the occupancy field.
     * @param occ
     */
    public void setOccupancy(double occ) {
        __occupancy = occ;
    }

    /**
     *
     * @return the temperature factor
     */
    public double getTemperatureFactor() {
        return __temperature_factor;
    }

    /**
     * Set the temperature factor field.
     * @param t
     */
    public void setTemperatureFactor(double tf) {
        __temperature_factor = tf;
    }

    /**
     *
     * @return the segment id.
     */
    public String getSegmentId() {
        return __segment_id;
    }

    /**
     * Set the segment id.
     * @param s
     */
    public void setSegmentId(String s) {
        __segment_id = s;
    }

    /**
     *
     * @return the element type.
     */
    public String getElementType() {
        return __element;
    }

    /**
     * Sets the periodic table atom type iff the atom name field has been set
     * previously correctly. Actually, this method does not have any use since
     * the method setAtomName() always sets this field correctly.
     *
     * @param elementType the periodic table element type
     */
    public void setElementType(String elementType) {
        if (__atom_name.startsWith(elementType.trim().toUpperCase())) {
            __element = elementType;
            setAtomProperties();
        } else {
            System.out.println("Error: atom name and atom type mismatch while setting the atom type");
            System.exit(1);
        }
    }

    /**
     *
     * @return the charge.
     */
    public String getCharge() {
        return __charge;
    }

    /**
     * Set the charge.
     * @param charge
     */
    public void setCharge(String charge) {
        __charge = charge;
    }

    /**
     *
     * @return
     */
    public double getVdwRadius() {
        return __vdw_radius;
    }

    /**
     *
     * @param a
     * @return
     */
    @Deprecated public boolean addBond_old(myAtom a) {// TODO: this seems incorrect to to be removed eventually
        // remove old entries
        if (__adjacency_list.contains(a)) {
            __adjacency_list.remove(a);
        }
        if (a.__adjacency_list.contains(this)) {
            a.__adjacency_list.remove(this);
        }

        // add new entries
        return this.__adjacency_list.add(a) && a.__adjacency_list.add(this);
    }

    public boolean addBond(myAtom a) {
        // remove old entries
        myAtom ac = null;
        for (myAtom temp : __adjacency_list) {
            if (temp.getAtomName().equalsIgnoreCase(a.getAtomName())) {
                ac = temp;
            }
        }
        if (ac != null)
            __adjacency_list.remove(ac);

        ac = null;
        for (myAtom temp : a.__adjacency_list) {
            if (temp.getAtomName().equalsIgnoreCase(this.getAtomName())) {
                ac = temp;
            }
        }
        if (ac != null)
            a.__adjacency_list.remove(ac);

        // add new entries
        return this.__adjacency_list.add(a) && a.__adjacency_list.add(this);
    }

    /**
     *
     * @return
     */
    public int numberOfBonds() {
        return __adjacency_list.size();
    }

    /**
     * Print the atoms this atom bonded to
     */
    public void printBonds() {
        Iterator<myAtom> iter = bondIterator();
        while (iter.hasNext()) {
            myAtom a = iter.next();
            System.out.print(a.getAtomName() + ' ' + a.getResidueName() + ' ' + a.getResidueNumber() + ' ');
        }
        System.out.println();
    }

    /**
     *
     * @return
     */
    public Set<myAtom> getBondedAtoms() { // Note: does is expose a protected field to be modified by others? Check if that is OK.
        return __adjacency_list;
    }

    /**
     * Test if this atom is bonded to the atom in the argument.
     *
     * @param a the atom which is to be tested for a bond with this object
     * @return true if there is a bond between this atom and the atom a
     */
    public boolean isBondedTo(myAtom a) {
        Iterator<myAtom> iter = bondIterator();
        while (iter.hasNext()) {
            myAtom a2 = iter.next();
            if (a2.getAtomName().equals(a.getAtomName()) &&
                    a2.getResidueName().equals(a.getResidueName()) &&
                    a2.getResidueNumber() == a.getResidueNumber()) {
                return true;
            }
        }
        return false;
    }

    /**
     * Delete the bond to the atom in the argument.
     *
     * @param b the atom to which the bond (if any) is to be deleted
     * @return true if the bond existed previously, and deleted now
     */
    public boolean removeBondTo(myAtom b) {
        return b.__adjacency_list.remove(this) && this.__adjacency_list.remove(b);
    }

    /**
     * Delete all the bonds to any of the atoms.
     */
    public void clearAllBonds() {
        __adjacency_list.clear();
    }

    /**
     * Print the atom in PDB row format.
     */
    //ATOM   5212  OG  SER A 337      -0.684  11.846 -11.394  0.50 13.13      5ECD 123
    //ATOM    212  O   SER D  12     -27.633   4.997  14.478  1.00 22.22      6FAB 132
    //static String atom_line_iformat = "ATOM  %5d%*1c%4c%1c%3c%*c%1c%4d%1c%*1c%*1c%*1c%8f%8f%8f%6f%6f%*1c%*1c%*1c%*1c%*1c%*1c%4c%2c%2c";
    public void print() {
        myPdbResidueRow r = new myPdbResidueRow(getRecordName(), getSerialNumber(),
                getAtomName(), getAlternateLocationIndicator(), getResidueName(),
                getChainId(), getResidueNumber(), getInsertionCode(), getCoordinates(),
                getOccupancy(), getTemperatureFactor(), getSegmentId(), getElementType(), getCharge());
        r.print();
    }

    /**
     * Returns the BondIterator object that can iterate the bonds of the atom.
     *
     * @return the BondIterator object
     */
    public BondIterator bondIterator() {
        return new BondIterator();
    }

// The following method should not be implemented as the semantics is confusing.
//    public Iterator<myAtom> Iterator() {
//        return new BondIterator();
//    }
    
    /**
     * This class provides an implementation of the bond iterator.
     */
    public class BondIterator implements Iterator<myAtom>, Iterable<myAtom> {

        private Iterator<myAtom> iter = __adjacency_list.iterator();

        public boolean hasNext() {
            return iter.hasNext();
        }

        public myAtom next() throws NoSuchElementException {
            if (!iter.hasNext()) {
                throw new NoSuchElementException();
            }
            return iter.next();
        }

        public void remove() {
            iter.remove();
        }

        public Iterator<myAtom> iterator() {
            return __adjacency_list.iterator();
        }
    }

    /**
     * Sets default properties such as radii, element number, mass, element type
     * Note: RHL I don't know where the radii values come from but they are good estimates
     * usually atoms will be assigned better information from a forcefield
     */
    public void setAtomProperties() {
        __force_field_type = "?";

        if (__element.equals("ILLEGAL") || __element.trim().length() == 0) { // means the field is yet to be set  
            if (__record_name.equals("ATOM")) {
                //The radii for C, H, N, O, P, S were changed to those used by the Richardsons' PROBE;
                //The old radii are commented out to the side
                if (__atom_name.toUpperCase().indexOf("C") == 0) {
                    if (isCarbonylC()) { // means this atom is a carbonyl C
                        __vdw_radius = 165; //__vdw_radius = 180;
                        __atomic_number = 6;
                        __atomic_mass = 12.01;
                        __element = "C";
                    } else {
                        __vdw_radius = 175; //__vdw_radius = 180;
                        __atomic_number = 6;
                        __atomic_mass = 12.01;
                        __element = "C";
                    }
                } else if (__atom_name.toUpperCase().indexOf("H") == 0) {
                    if (isAromaticH() || isPolarH()) {
                        __vdw_radius = 100; //__vdw_radius = 80;
                        __atomic_number = 1;
                        __atomic_mass = 1;
                        __element = "H";
                    } else {
                        __vdw_radius = 117; //__vdw_radius = 80;
                        __atomic_number = 1;
                        __atomic_mass = 1;
                        __element = "H";
                    }
                } else if (__atom_name.toUpperCase().indexOf("N") == 0) {
                    __vdw_radius = 155; //__vdw_radius = 170;
                    __atomic_number = 7;
                    __atomic_mass = 14.01;
                    __element = "N";
                } else if (__atom_name.toUpperCase().indexOf("O") == 0) {
                    __vdw_radius = 140; //__vdw_radius = 170;
                    __atomic_number = 8;
                    __atomic_mass = 16;
                    __element = "O";
                } else if (__atom_name.toUpperCase().indexOf("P") == 0) {
                    __vdw_radius = 180; //__vdw_radius = 259;
                    __atomic_number = 15;
                    __atomic_mass = 30.97;
                    __element = "P";
                } else if (__atom_name.toUpperCase().indexOf("S") == 0) {
                    __vdw_radius = 180; //__vdw_radius = 255;
                    __atomic_number = 16;
                    __atomic_mass = 32.07;
                    __element = "S";
                }
            } else { // __record_name.equals("HETATM")
                if (__atom_name.indexOf("AC") != -1) {
                    __atomic_number = 89;
                    __vdw_radius = 295;
                    __atomic_mass = 227;
                    __element = "AC";
                } else if (__atom_name.indexOf("AG") != -1) {
                    __vdw_radius = 398;
                    __atomic_number = 47;
                    __atomic_mass = 107.9;
                    __element = "AG";
                } else if (__atom_name.indexOf("AL") != -1) {
                    __vdw_radius = 338;
                    __atomic_number = 13;
                    __atomic_mass = 26.98;
                    __element = "AL";
                } else if (__atom_name.indexOf("AM") != -1) {
                    __atomic_number = 95;
                    __vdw_radius = 230;
                    __atomic_mass = 243;
                    __element = "AM";
                } else if (__atom_name.indexOf("AR") != -1) {
                    __atomic_number = 18;
                    __vdw_radius = 392;
                    __atomic_mass = 39.95;
                    __element = "AR";
                } else if (__atom_name.indexOf("AS") != -1) {
                    __atomic_number = 33;
                    __vdw_radius = 302;
                    __atomic_mass = 74.92;
                    __element = "AS";
                } else if (__atom_name.indexOf("AT") != -1) {
                    __atomic_number = 85;
                    __vdw_radius = 302;
                    __atomic_mass = 210;
                    __element = "AT";
                } else if (__atom_name.indexOf("AU") != -1) {
                    __vdw_radius = 375;
                    __atomic_number = 79;
                    __atomic_mass = 197;
                    __element = "AU";
                } else if (__atom_name.indexOf("BA") != -1) {
                    __vdw_radius = 335;
                    __atomic_number = 56;
                    __atomic_mass = 137.3;
                    __element = "BA";
                } else if (__atom_name.indexOf("BE") != -1) {
                    __atomic_number = 4;
                    __vdw_radius = 88;
                    __atomic_mass = 9.012;
                    __element = "BE";
                } else if (__atom_name.indexOf("BI") != -1) {
                    __atomic_number = 83;
                    __vdw_radius = 385;
                    __atomic_mass = 209;
                    __element = "BI";
                } else if (__atom_name.indexOf("BK") != -1) {
                    __atomic_number = 97;
                    __vdw_radius = 225;
                    __atomic_mass = 247;
                    __element = "BK";
                } else if (__atom_name.indexOf("BR") != -1) {
                    __vdw_radius = 302;
                    __atomic_number = 97;
                    __atomic_mass = 247;
                    __element = "BR";
                } else if (__atom_name.indexOf("CA") != -1) {
                    __vdw_radius = 248;
                    __atomic_number = 20;
                    __atomic_mass = 40.08;
                    __element = "CA";
                } else if (__atom_name.indexOf("CD") != -1) {
                    __atomic_number = 48;
                    __vdw_radius = 422;
                    __atomic_mass = 112.4;
                    __element = "CD";
                } else if (__atom_name.indexOf("CE") != -1) {
                    __atomic_number = 58;
                    __vdw_radius = 458;
                    __atomic_mass = 140.1;
                    __element = "CE";
                } else if (__atom_name.indexOf("CF") != -1) {
                    __atomic_number = 98;
                    __vdw_radius = 222;
                    __atomic_mass = 249;
                    __element = "CF";
                } else if (__atom_name.indexOf("CL") != -1) {
                    __atomic_number = 17;
                    __vdw_radius = 250;
                    __atomic_mass = 35.45;
                    __element = "CL";
                } else if (__atom_name.indexOf("CM") != -1) {
                    __atomic_number = 96;
                    __vdw_radius = 228;
                    __atomic_mass = 247;
                    __element = "CM";
                } else if (__atom_name.indexOf("CO") != -1) {
                    __vdw_radius = 332;
                    __atomic_number = 27;
                    __atomic_mass = 58.93;
                    __element = "CO";
                } else if (__atom_name.indexOf("CR") != -1) {
                    __vdw_radius = 338;
                    __atomic_number = 24;
                    __atomic_mass = 52;
                    __element = "CR";
                } else if (__atom_name.indexOf("CS") != -1) {
                    __atomic_number = 55;
                    __vdw_radius = 418;
                    __atomic_mass = 132.9;
                    __element = "CS";
                } else if (__atom_name.indexOf("CU") != -1) {
                    __vdw_radius = 380;
                    __atomic_number = 29;
                    __atomic_mass = 63.55;
                    __element = "CU";
                } else if (__atom_name.indexOf("DY") != -1) {
                    __atomic_number = 66;
                    __vdw_radius = 438;
                    __atomic_mass = 162.5;
                    __element = "DY";
                } else if (__atom_name.indexOf("ER") != -1) {
                    __atomic_number = 68;
                    __vdw_radius = 432;
                    __atomic_mass = 167.3;
                    __element = "ER";
                } else if (__atom_name.indexOf("ES") != -1) {
                    __atomic_number = 99;
                    __vdw_radius = 220;
                    __atomic_mass = 254;
                    __element = "ES";
                } else if (__atom_name.indexOf("EU") != -1) {
                    __atomic_number = 63;
                    __vdw_radius = 498;
                    __atomic_mass = 152;
                    __element = "EU";
                } else if (__atom_name.indexOf("FE") != -1) {
                    __vdw_radius = 335;
                    __atomic_number = 26;
                    __atomic_mass = 55.85;
                    __element = "FE";
                } else if (__atom_name.indexOf("FM") != -1) {
                    __atomic_number = 100;
                    __vdw_radius = 218;
                    __atomic_mass = 250;
                    __element = "FM";
                } else if (__atom_name.indexOf("FR") != -1) {
                    __atomic_number = 87;
                    __vdw_radius = 450;
                    __atomic_mass = 223;
                    __element = "FR";
                } else if (__atom_name.indexOf("GA") != -1) {
                    __vdw_radius = 305;
                    __atomic_number = 31;
                    __atomic_mass = 69.72;
                    __element = "GA";
                } else if (__atom_name.indexOf("GD") != -1) {
                    __atomic_number = 64;
                    __vdw_radius = 448;
                    __atomic_mass = 157.3;
                    __element = "GD";
                } else if (__atom_name.indexOf("GE") != -1) {
                    __atomic_number = 32;
                    __vdw_radius = 292;
                    __atomic_mass = 72.59;
                    __element = "GE";
                } else if (__atom_name.indexOf("HE") != -1) {
                    __vdw_radius = 400;
                    __atomic_number = 2;
                    __atomic_mass = 4.003;
                    __element = "HE";
                } else if (__atom_name.indexOf("HF") != -1) {
                    __atomic_number = 72;
                    __vdw_radius = 392;
                    __atomic_mass = 178.5;
                    __element = "HF";
                } else if (__atom_name.indexOf("HG") != -1) {
                    __atomic_number = 80;
                    __vdw_radius = 425;
                    __atomic_mass = 200.6;
                    __element = "HG";
                } else if (__atom_name.indexOf("HO") != -1) {
                    __atomic_number = 67;
                    __vdw_radius = 435;
                    __atomic_mass = 164.9;
                    __element = "HO";
                } else if (__atom_name.indexOf("IN") != -1) {
                    __atomic_number = 49;
                    __vdw_radius = 408;
                    __atomic_mass = 114.8;
                    __element = "IN";
                } else if (__atom_name.indexOf("IR") != -1) {
                    __atomic_number = 77;
                    __vdw_radius = 330;
                    __atomic_mass = 192.2;
                    __element = "IR";
                } else if (__atom_name.indexOf("KR") != -1) {
                    __atomic_number = 36;
                    __vdw_radius = 400;
                    __atomic_mass = 83.8;
                    __element = "KR";
                } else if (__atom_name.indexOf("LA") != -1) {
                    __atomic_number = 57;
                    __vdw_radius = 468;
                    __atomic_mass = 138.9;
                    __element = "LA";
                } else if (__atom_name.indexOf("LI") != -1) {
                    __vdw_radius = 170;
                    __atomic_number = 3;
                    __atomic_mass = 6.941;
                    __element = "LI";
                } else if (__atom_name.indexOf("LR") != -1) {
                    __atomic_number = 103;
                    __vdw_radius = 210;
                    __atomic_mass = 257;
                    __element = "LR";
                } else if (__atom_name.indexOf("LU") != -1) {
                    __atomic_number = 71;
                    __vdw_radius = 430;
                    __atomic_mass = 175;
                    __element = "LU";
                } else if (__atom_name.indexOf("MD") != -1) {
                    __atomic_number = 101;
                    __vdw_radius = 215;
                    __atomic_mass = 256;
                    __element = "MD";
                } else if (__atom_name.indexOf("MG") != -1) {
                    __vdw_radius = 275;
                    __atomic_number = 12;
                    __atomic_mass = 24.31;
                    __element = "MG";
                } else if (__atom_name.indexOf("MN") != -1) {
                    __vdw_radius = 338;
                    __atomic_number = 25;
                    __atomic_mass = 54.94;
                    __element = "MN";
                } else if (__atom_name.indexOf("MO") != -1) {
                    __atomic_number = 42;
                    __vdw_radius = 368;
                    __atomic_mass = 95.94;
                    __element = "MO";
                } else if (__atom_name.indexOf("NA") != -1) {
                    __atomic_number = 11;
                    __vdw_radius = 243;
                    __atomic_mass = 22.99;
                    __element = "NA";
                } else if (__atom_name.indexOf("NB") != -1) {
                    __atomic_number = 41;
                    __vdw_radius = 370;
                    __atomic_mass = 92.91;
                    __element = "NB";
                } else if (__atom_name.indexOf("ND") != -1) {
                    __atomic_number = 60;
                    __vdw_radius = 452;
                    __atomic_mass = 144.2;
                    __element = "ND";
                } else if (__atom_name.indexOf("NE") != -1) {
                    __atomic_number = 10;
                    __vdw_radius = 280;
                    __atomic_mass = 20.18;
                    __element = "NE";
                } else if (__atom_name.indexOf("NI") != -1) {
                    __vdw_radius = 405;
                    __atomic_number = 28;
                    __atomic_mass = 58.69;
                    __element = "NI";
                } else if (__atom_name.indexOf("NO") != -1) {
                    __atomic_number = 102;
                    __vdw_radius = 212;
                    __atomic_mass = 253;
                    __element = "NO";
                } else if (__atom_name.indexOf("NP") != -1) {
                    __atomic_number = 93;
                    __vdw_radius = 238;
                    __atomic_mass = 237;
                    __element = "NP";
                } else if (__atom_name.indexOf("OS") != -1) {
                    __atomic_number = 76;
                    __vdw_radius = 342;
                    __atomic_mass = 190.2;
                    __element = "OS";
                } else if (__atom_name.indexOf("PA") != -1) {
                    __atomic_number = 91;
                    __vdw_radius = 222;
                    __atomic_mass = 231;
                    __element = "PA";
                } else if (__atom_name.indexOf("PB") != -1) {
                    __atomic_number = 82;
                    __vdw_radius = 385;
                    __atomic_mass = 207.2;
                    __element = "PB";
                } else if (__atom_name.indexOf("PD") != -1) {
                    __atomic_number = 46;
                    __vdw_radius = 375;
                    __atomic_mass = 106.4;
                    __element = "PD";
                } else if (__atom_name.indexOf("PM") != -1) {
                    __atomic_number = 61;
                    __vdw_radius = 450;
                    __atomic_mass = 147;
                    __element = "PM";
                } else if (__atom_name.indexOf("PO") != -1) {
                    __atomic_number = 84;
                    __vdw_radius = 420;
                    __atomic_mass = 210;
                    __element = "PO";
                } else if (__atom_name.indexOf("PR") != -1) {
                    __atomic_number = 59;
                    __vdw_radius = 455;
                    __atomic_mass = 140.9;
                    __element = "PR";
                } else if (__atom_name.indexOf("PT") != -1) {
                    __atomic_number = 78;
                    __vdw_radius = 375;
                    __atomic_mass = 195.1;
                    __element = "PT";
                } else if (__atom_name.indexOf("PU") != -1) {
                    __atomic_number = 94;
                    __vdw_radius = 232;
                    __atomic_mass = 242;
                    __element = "PU";
                } else if (__atom_name.indexOf("RA") != -1) {
                    __atomic_number = 88;
                    __vdw_radius = 358;
                    __atomic_mass = 226;
                    __element = "RA";
                } else if (__atom_name.indexOf("RB") != -1) {
                    __atomic_number = 37;
                    __vdw_radius = 368;
                    __atomic_mass = 85.47;
                    __element = "RB";
                } else if (__atom_name.indexOf("RE") != -1) {
                    __atomic_number = 75;
                    __vdw_radius = 338;
                    __atomic_mass = 186.2;
                    __element = "RE";
                } else if (__atom_name.indexOf("RH") != -1) {
                    __atomic_number = 45;
                    __vdw_radius = 362;
                    __atomic_mass = 102.9;
                    __element = "RH";
                } else if (__atom_name.indexOf("RN") != -1) {
                    __atomic_number = 88;
                    __vdw_radius = 475;
                    __atomic_mass = 222;
                    __element = "RN";
                } else if (__atom_name.indexOf("RU") != -1) {
                    __atomic_number = 44;
                    __vdw_radius = 350;
                    __atomic_mass = 101.1;
                    __element = "RU";
                } else if (__atom_name.indexOf("SB") != -1) {
                    __atomic_number = 51;
                    __vdw_radius = 365;
                    __atomic_mass = 121.8;
                    __element = "SB";
                } else if (__atom_name.indexOf("SC") != -1) {
                    __vdw_radius = 360;
                    __atomic_number = 21;
                    __atomic_mass = 44.96;
                    __element = "SC";
                } else if (__atom_name.indexOf("SE") != -1) {
                    __atomic_number = 34;
                    __vdw_radius = 305;
                    __atomic_mass = 78.96;
                    __element = "SE";
                } else if (__atom_name.indexOf("SI") != -1) {
                    __vdw_radius = 300;
                    __atomic_number = 14;
                    __atomic_mass = 28.09;
                    __element = "SI";
                } else if (__atom_name.indexOf("SM") != -1) {
                    __atomic_number = 62;
                    __vdw_radius = 450;
                    __atomic_mass = 150.4;
                    __element = "SM";
                } else if (__atom_name.indexOf("SN") != -1) {
                    __atomic_number = 50;
                    __vdw_radius = 365;
                    __atomic_mass = 118.7;
                    __element = "SN";
                } else if (__atom_name.indexOf("SR") != -1) {
                    __atomic_number = 38;
                    __vdw_radius = 280;
                    __atomic_mass = 87.62;
                    __element = "SR";
                } else if (__atom_name.indexOf("TA") != -1) {
                    __atomic_number = 73;
                    __vdw_radius = 358;
                    __atomic_mass = 180.9;
                    __element = "TA";
                } else if (__atom_name.indexOf("TB") != -1) {
                    __atomic_number = 65;
                    __vdw_radius = 440;
                    __atomic_mass = 158.9;
                    __element = "TB";
                } else if (__atom_name.indexOf("TC") != -1) {
                    __atomic_number = 43;
                    __vdw_radius = 338;
                    __atomic_mass = 99;
                    __element = "TC";
                } else if (__atom_name.indexOf("TE") != -1) {
                    __atomic_number = 52;
                    __vdw_radius = 368;
                    __atomic_mass = 127.6;
                    __element = "TE";
                } else if (__atom_name.indexOf("TH") != -1) {
                    __atomic_number = 90;
                    __vdw_radius = 255;
                    __atomic_mass = 232;
                    __element = "TH";
                } else if (__atom_name.indexOf("TI") != -1) {
                    __vdw_radius = 368;
                    __atomic_number = 22;
                    __atomic_mass = 47.88;
                    __element = "TI";
                } else if (__atom_name.indexOf("TL") != -1) {
                    __atomic_number = 81;
                    __vdw_radius = 388;
                    __atomic_mass = 204.4;
                    __element = "TL";
                } else if (__atom_name.indexOf("TM") != -1) {
                    __atomic_number = 69;
                    __vdw_radius = 430;
                    __atomic_mass = 168.9;
                    __element = "TM";
                } else if (__atom_name.indexOf("UNE") != -1) {
                    __atomic_number = 109;
                    __atomic_mass = 266;
                    __element = "UNE";
                } else if (__atom_name.indexOf("UNH") != -1) {
                    __atomic_number = 106;
                    __atomic_mass = 263;
                    __element = "UNH";
                } else if (__atom_name.indexOf("UNO") != -1) {
                    __atomic_number = 108;
                    __atomic_mass = 265;
                    __element = "UNO";
                } else if (__atom_name.indexOf("UNP") != -1) {
                    __atomic_number = 105;
                    __atomic_mass = 260;
                    __element = "UNP";
                } else if (__atom_name.indexOf("UNQ") != -1) {
                    __atomic_number = 104;
                    __atomic_mass = 257;
                    __element = "UNQ";
                } else if (__atom_name.indexOf("UNS") != -1) {
                    __atomic_number = 107;
                    __atomic_mass = 262;
                    __element = "UNS";
                } else if (__atom_name.indexOf("XE") != -1) {
                    __atomic_number = 54;
                    __vdw_radius = 425;
                    __atomic_mass = 131.3;
                    __element = "XE";
                } else if (__atom_name.indexOf("YB") != -1) {
                    __atomic_number = 70;
                    __vdw_radius = 485;
                    __atomic_mass = 173;
                    __element = "YB";
                } else if (__atom_name.indexOf("ZN") != -1) {
                    __vdw_radius = 362;
                    __atomic_number = 30;
                    __atomic_mass = 65.39;
                    __element = "ZN";
                } else if (__atom_name.indexOf("ZR") != -1) {
                    __atomic_number = 40;
                    __vdw_radius = 390;
                    __atomic_mass = 91.22;
                    __element = "ZR";
                } //The radii for C, H, N, O, P, S were changed to those used by the Richardsons' PROBE;
                //The old radii are commented out to the side
                else if (__atom_name.toUpperCase().indexOf("C") == 0) {
                    if (isCarbonylC()) { // means this atom is a carbonyl C
                        __vdw_radius = 165; //__vdw_radius = 180;
                        __atomic_number = 6;
                        __atomic_mass = 12.01;
                        __element = "C";
                    } else {
                        __vdw_radius = 175; //__vdw_radius = 180;
                        __atomic_number = 6;
                        __atomic_mass = 12.01;
                        __element = "C";
                    }
                } else if (__atom_name.toUpperCase().indexOf("H") == 0) {
                    if (isAromaticH() || isPolarH()) {
                        __vdw_radius = 100; //__vdw_radius = 80;
                        __atomic_number = 1;
                        __atomic_mass = 1;
                        __element = "H";
                    } else {
                        __vdw_radius = 117; //__vdw_radius = 80;
                        __atomic_number = 1;
                        __atomic_mass = 1;
                        __element = "H";
                    }
                } else if (__atom_name.toUpperCase().indexOf("N") == 0) {
                    __vdw_radius = 155; //__vdw_radius = 170;
                    __atomic_number = 7;
                    __atomic_mass = 14.01;
                    __element = "N";
                } else if (__atom_name.toUpperCase().indexOf("O") == 0) {
                    __vdw_radius = 140; //__vdw_radius = 170;
                    __atomic_number = 8;
                    __atomic_mass = 16;
                    __element = "O";
                } else if (__atom_name.toUpperCase().indexOf("B") == 0) {
                    __vdw_radius = 208;
                    __atomic_number = 5;
                    __atomic_mass = 10.81;
                    __element = "B";
                } else if (__atom_name.toUpperCase().indexOf("I") == 0) {
                    __vdw_radius = 350;
                    __atomic_number = 53;
                    __atomic_mass = 126.9;
                    __element = "I";
                } else if (__atom_name.toUpperCase().indexOf("F") == 0) {
                    __vdw_radius = 160;
                    __atomic_number = 9;
                    __atomic_mass = 19.0;
                    __element = "F";
                } else if (__atom_name.toUpperCase().indexOf("P") == 0) {
                    __vdw_radius = 180; //__vdw_radius = 259;
                    __atomic_number = 15;
                    __atomic_mass = 30.97;
                    __element = "P";
                } else if (__atom_name.toUpperCase().indexOf("K") == 0) {
                    __vdw_radius = 332;
                    __atomic_number = 19;
                    __atomic_mass = 39.1;
                    __element = "K";
                } else if (__atom_name.toUpperCase().indexOf("S") == 0) {
                    __vdw_radius = 180; //__vdw_radius = 255;
                    __atomic_number = 16;
                    __atomic_mass = 32.07;
                    __element = "S";
                } else if (__atom_name.toUpperCase().indexOf("U") == 0) {
                    __vdw_radius = 242;
                    __atomic_number = 92;
                    __atomic_mass = 238;
                    __element = "U";
                } else if (__atom_name.toUpperCase().indexOf("V") == 0) {
                    __vdw_radius = 332;
                    __atomic_number = 23;
                    __atomic_mass = 50.94;
                    __element = "V";
                } else if (__atom_name.toUpperCase().indexOf("W") == 0) {
                    __vdw_radius = 342;
                    __atomic_number = 74;
                    __atomic_mass = 183.9;
                    __element = "W";
                } else if (__atom_name.toUpperCase().indexOf("Y") == 0) {
                    __vdw_radius = 445;
                    __atomic_number = 39;
                    __atomic_mass = 88.91;
                    __element = "Y";
                } else {  //unrecognized atom type
                    __vdw_radius = 10000;
                    __element = "ILLEGAL";
                }
            }
        } else { // Just use the same __element field
            if (__element.indexOf("AC") != -1) {
                __atomic_number = 89;
                __vdw_radius = 295;
                __atomic_mass = 227;
                __element = "AC";
            } else if (__element.indexOf("AG") != -1) {
                __vdw_radius = 398;
                __atomic_number = 47;
                __atomic_mass = 107.9;
                __element = "AG";
            } else if (__element.indexOf("AL") != -1) {
                __vdw_radius = 338;
                __atomic_number = 13;
                __atomic_mass = 26.98;
                __element = "AL";
            } else if (__element.indexOf("AM") != -1) {
                __atomic_number = 95;
                __vdw_radius = 230;
                __atomic_mass = 243;
                __element = "AM";
            } else if (__element.indexOf("AR") != -1) {
                __atomic_number = 18;
                __vdw_radius = 392;
                __atomic_mass = 39.95;
                __element = "AR";
            } else if (__element.indexOf("AS") != -1) {
                __atomic_number = 33;
                __vdw_radius = 302;
                __atomic_mass = 74.92;
                __element = "AS";
            } else if (__element.indexOf("AT") != -1) {
                __atomic_number = 85;
                __vdw_radius = 302;
                __atomic_mass = 210;
                __element = "AT";
            } else if (__element.indexOf("AU") != -1) {
                __vdw_radius = 375;
                __atomic_number = 79;
                __atomic_mass = 197;
                __element = "AU";
            } else if (__element.indexOf("BA") != -1) {
                __vdw_radius = 335;
                __atomic_number = 56;
                __atomic_mass = 137.3;
                __element = "BA";
            } else if (__element.indexOf("BE") != -1) {
                __atomic_number = 4;
                __vdw_radius = 88;
                __atomic_mass = 9.012;
                __element = "BE";
            } else if (__element.indexOf("BI") != -1) {
                __atomic_number = 83;
                __vdw_radius = 385;
                __atomic_mass = 209;
                __element = "BI";
            } else if (__element.indexOf("BK") != -1) {
                __atomic_number = 97;
                __vdw_radius = 225;
                __atomic_mass = 247;
                __element = "BK";
            } else if (__element.indexOf("BR") != -1) {
                __vdw_radius = 302;
                __atomic_number = 97;
                __atomic_mass = 247;
                __element = "BR";
            } else if (__element.indexOf("CA") != -1) {
                __vdw_radius = 248;
                __atomic_number = 20;
                __atomic_mass = 40.08;
                __element = "CA";
            } else if (__element.indexOf("CD") != -1) {
                __atomic_number = 48;
                __vdw_radius = 422;
                __atomic_mass = 112.4;
                __element = "CD";
            } else if (__element.indexOf("CE") != -1) {
                __atomic_number = 58;
                __vdw_radius = 458;
                __atomic_mass = 140.1;
                __element = "CE";
            } else if (__element.indexOf("CF") != -1) {
                __atomic_number = 98;
                __vdw_radius = 222;
                __atomic_mass = 249;
                __element = "CF";
            } else if (__element.indexOf("CL") != -1) {
                __atomic_number = 17;
                __vdw_radius = 250;
                __atomic_mass = 35.45;
                __element = "CL";
            } else if (__element.indexOf("CM") != -1) {
                __atomic_number = 96;
                __vdw_radius = 228;
                __atomic_mass = 247;
                __element = "CM";
            } else if (__element.indexOf("CO") != -1) {
                __vdw_radius = 332;
                __atomic_number = 27;
                __atomic_mass = 58.93;
                __element = "CO";
            } else if (__element.indexOf("CR") != -1) {
                __vdw_radius = 338;
                __atomic_number = 24;
                __atomic_mass = 52;
                __element = "CR";
            } else if (__element.indexOf("CS") != -1) {
                __atomic_number = 55;
                __vdw_radius = 418;
                __atomic_mass = 132.9;
                __element = "CS";
            } else if (__element.indexOf("CU") != -1) {
                __vdw_radius = 380;
                __atomic_number = 29;
                __atomic_mass = 63.55;
                __element = "CU";
            } else if (__element.indexOf("DY") != -1) {
                __atomic_number = 66;
                __vdw_radius = 438;
                __atomic_mass = 162.5;
                __element = "DY";
            } else if (__element.indexOf("ER") != -1) {
                __atomic_number = 68;
                __vdw_radius = 432;
                __atomic_mass = 167.3;
                __element = "ER";
            } else if (__element.indexOf("ES") != -1) {
                __atomic_number = 99;
                __vdw_radius = 220;
                __atomic_mass = 254;
                __element = "ES";
            } else if (__element.indexOf("EU") != -1) {
                __atomic_number = 63;
                __vdw_radius = 498;
                __atomic_mass = 152;
                __element = "EU";
            } else if (__element.indexOf("FE") != -1) {
                __vdw_radius = 335;
                __atomic_number = 26;
                __atomic_mass = 55.85;
                __element = "FE";
            } else if (__element.indexOf("FM") != -1) {
                __atomic_number = 100;
                __vdw_radius = 218;
                __atomic_mass = 250;
                __element = "FM";
            } else if (__element.indexOf("FR") != -1) {
                __atomic_number = 87;
                __vdw_radius = 450;
                __atomic_mass = 223;
                __element = "FR";
            } else if (__element.indexOf("GA") != -1) {
                __vdw_radius = 305;
                __atomic_number = 31;
                __atomic_mass = 69.72;
                __element = "GA";
            } else if (__element.indexOf("GD") != -1) {
                __atomic_number = 64;
                __vdw_radius = 448;
                __atomic_mass = 157.3;
                __element = "GD";
            } else if (__element.indexOf("GE") != -1) {
                __atomic_number = 32;
                __vdw_radius = 292;
                __atomic_mass = 72.59;
                __element = "GE";
            } else if (__element.indexOf("HE") != -1) {
                __vdw_radius = 400;
                __atomic_number = 2;
                __atomic_mass = 4.003;
                __element = "HE";
            } else if (__element.indexOf("HF") != -1) {
                __atomic_number = 72;
                __vdw_radius = 392;
                __atomic_mass = 178.5;
                __element = "HF";
            } else if (__element.indexOf("HG") != -1) {
                __atomic_number = 80;
                __vdw_radius = 425;
                __atomic_mass = 200.6;
                __element = "HG";
            } else if (__element.indexOf("HO") != -1) {
                __atomic_number = 67;
                __vdw_radius = 435;
                __atomic_mass = 164.9;
                __element = "HO";
            } else if (__element.indexOf("IN") != -1) {
                __atomic_number = 49;
                __vdw_radius = 408;
                __atomic_mass = 114.8;
                __element = "IN";
            } else if (__element.indexOf("IR") != -1) {
                __atomic_number = 77;
                __vdw_radius = 330;
                __atomic_mass = 192.2;
                __element = "IR";
            } else if (__element.indexOf("KR") != -1) {
                __atomic_number = 36;
                __vdw_radius = 400;
                __atomic_mass = 83.8;
                __element = "KR";
            } else if (__element.indexOf("LA") != -1) {
                __atomic_number = 57;
                __vdw_radius = 468;
                __atomic_mass = 138.9;
                __element = "LA";
            } else if (__element.indexOf("LI") != -1) {
                __vdw_radius = 170;
                __atomic_number = 3;
                __atomic_mass = 6.941;
                __element = "LI";
            } else if (__element.indexOf("LR") != -1) {
                __atomic_number = 103;
                __vdw_radius = 210;
                __atomic_mass = 257;
                __element = "LR";
            } else if (__element.indexOf("LU") != -1) {
                __atomic_number = 71;
                __vdw_radius = 430;
                __atomic_mass = 175;
                __element = "LU";
            } else if (__element.indexOf("MD") != -1) {
                __atomic_number = 101;
                __vdw_radius = 215;
                __atomic_mass = 256;
                __element = "MD";
            } else if (__element.indexOf("MG") != -1) {
                __vdw_radius = 275;
                __atomic_number = 12;
                __atomic_mass = 24.31;
                __element = "MG";
            } else if (__element.indexOf("MN") != -1) {
                __vdw_radius = 338;
                __atomic_number = 25;
                __atomic_mass = 54.94;
                __element = "MN";
            } else if (__element.indexOf("MO") != -1) {
                __atomic_number = 42;
                __vdw_radius = 368;
                __atomic_mass = 95.94;
                __element = "MO";
            } else if (__element.indexOf("NA") != -1) {
                __atomic_number = 11;
                __vdw_radius = 243;
                __atomic_mass = 22.99;
                __element = "NA";
            } else if (__element.indexOf("NB") != -1) {
                __atomic_number = 41;
                __vdw_radius = 370;
                __atomic_mass = 92.91;
                __element = "NB";
            } else if (__element.indexOf("ND") != -1) {
                __atomic_number = 60;
                __vdw_radius = 452;
                __atomic_mass = 144.2;
                __element = "ND";
            } else if (__element.indexOf("NE") != -1) {
                __atomic_number = 10;
                __vdw_radius = 280;
                __atomic_mass = 20.18;
                __element = "NE";
            } else if (__element.indexOf("NI") != -1) {
                __vdw_radius = 405;
                __atomic_number = 28;
                __atomic_mass = 58.69;
                __element = "NI";
            } else if (__element.indexOf("NO") != -1) {
                __atomic_number = 102;
                __vdw_radius = 212;
                __atomic_mass = 253;
                __element = "NO";
            } else if (__element.indexOf("NP") != -1) {
                __atomic_number = 93;
                __vdw_radius = 238;
                __atomic_mass = 237;
                __element = "NP";
            } else if (__element.indexOf("OS") != -1) {
                __atomic_number = 76;
                __vdw_radius = 342;
                __atomic_mass = 190.2;
                __element = "OS";
            } else if (__element.indexOf("PA") != -1) {
                __atomic_number = 91;
                __vdw_radius = 222;
                __atomic_mass = 231;
                __element = "PA";
            } else if (__element.indexOf("PB") != -1) {
                __atomic_number = 82;
                __vdw_radius = 385;
                __atomic_mass = 207.2;
                __element = "PB";
            } else if (__element.indexOf("PD") != -1) {
                __atomic_number = 46;
                __vdw_radius = 375;
                __atomic_mass = 106.4;
                __element = "PD";
            } else if (__element.indexOf("PM") != -1) {
                __atomic_number = 61;
                __vdw_radius = 450;
                __atomic_mass = 147;
                __element = "PM";
            } else if (__element.indexOf("PO") != -1) {
                __atomic_number = 84;
                __vdw_radius = 420;
                __atomic_mass = 210;
                __element = "PO";
            } else if (__element.indexOf("PR") != -1) {
                __atomic_number = 59;
                __vdw_radius = 455;
                __atomic_mass = 140.9;
                __element = "PR";
            } else if (__element.indexOf("PT") != -1) {
                __atomic_number = 78;
                __vdw_radius = 375;
                __atomic_mass = 195.1;
                __element = "PT";
            } else if (__element.indexOf("PU") != -1) {
                __atomic_number = 94;
                __vdw_radius = 232;
                __atomic_mass = 242;
                __element = "PU";
            } else if (__element.indexOf("RA") != -1) {
                __atomic_number = 88;
                __vdw_radius = 358;
                __atomic_mass = 226;
                __element = "RA";
            } else if (__element.indexOf("RB") != -1) {
                __atomic_number = 37;
                __vdw_radius = 368;
                __atomic_mass = 85.47;
                __element = "RB";
            } else if (__element.indexOf("RE") != -1) {
                __atomic_number = 75;
                __vdw_radius = 338;
                __atomic_mass = 186.2;
                __element = "RE";
            } else if (__element.indexOf("RH") != -1) {
                __atomic_number = 45;
                __vdw_radius = 362;
                __atomic_mass = 102.9;
                __element = "RH";
            } else if (__element.indexOf("RN") != -1) {
                __atomic_number = 88;
                __vdw_radius = 475;
                __atomic_mass = 222;
                __element = "RN";
            } else if (__element.indexOf("RU") != -1) {
                __atomic_number = 44;
                __vdw_radius = 350;
                __atomic_mass = 101.1;
                __element = "RU";
            } else if (__element.indexOf("SB") != -1) {
                __atomic_number = 51;
                __vdw_radius = 365;
                __atomic_mass = 121.8;
                __element = "SB";
            } else if (__element.indexOf("SC") != -1) {
                __vdw_radius = 360;
                __atomic_number = 21;
                __atomic_mass = 44.96;
                __element = "SC";
            } else if (__element.indexOf("SE") != -1) {
                __atomic_number = 34;
                __vdw_radius = 305;
                __atomic_mass = 78.96;
                __element = "SE";
            } else if (__element.indexOf("SI") != -1) {
                __vdw_radius = 300;
                __atomic_number = 14;
                __atomic_mass = 28.09;
                __element = "SI";
            } else if (__element.indexOf("SM") != -1) {
                __atomic_number = 62;
                __vdw_radius = 450;
                __atomic_mass = 150.4;
                __element = "SM";
            } else if (__element.indexOf("SN") != -1) {
                __atomic_number = 50;
                __vdw_radius = 365;
                __atomic_mass = 118.7;
                __element = "SN";
            } else if (__element.indexOf("SR") != -1) {
                __atomic_number = 38;
                __vdw_radius = 280;
                __atomic_mass = 87.62;
                __element = "SR";
            } else if (__element.indexOf("TA") != -1) {
                __atomic_number = 73;
                __vdw_radius = 358;
                __atomic_mass = 180.9;
                __element = "TA";
            } else if (__element.indexOf("TB") != -1) {
                __atomic_number = 65;
                __vdw_radius = 440;
                __atomic_mass = 158.9;
                __element = "TB";
            } else if (__element.indexOf("TC") != -1) {
                __atomic_number = 43;
                __vdw_radius = 338;
                __atomic_mass = 99;
                __element = "TC";
            } else if (__element.indexOf("TE") != -1) {
                __atomic_number = 52;
                __vdw_radius = 368;
                __atomic_mass = 127.6;
                __element = "TE";
            } else if (__element.indexOf("TH") != -1) {
                __atomic_number = 90;
                __vdw_radius = 255;
                __atomic_mass = 232;
                __element = "TH";
            } else if (__element.indexOf("TI") != -1) {
                __vdw_radius = 368;
                __atomic_number = 22;
                __atomic_mass = 47.88;
                __element = "TI";
            } else if (__element.indexOf("TL") != -1) {
                __atomic_number = 81;
                __vdw_radius = 388;
                __atomic_mass = 204.4;
                __element = "TL";
            } else if (__element.indexOf("TM") != -1) {
                __atomic_number = 69;
                __vdw_radius = 430;
                __atomic_mass = 168.9;
                __element = "TM";
            } else if (__element.indexOf("UNE") != -1) {
                __atomic_number = 109;
                __atomic_mass = 266;
                __element = "UNE";
            } else if (__element.indexOf("UNH") != -1) {
                __atomic_number = 106;
                __atomic_mass = 263;
                __element = "UNH";
            } else if (__element.indexOf("UNO") != -1) {
                __atomic_number = 108;
                __atomic_mass = 265;
                __element = "UNO";
            } else if (__element.indexOf("UNP") != -1) {
                __atomic_number = 105;
                __atomic_mass = 260;
                __element = "UNP";
            } else if (__element.indexOf("UNQ") != -1) {
                __atomic_number = 104;
                __atomic_mass = 257;
                __element = "UNQ";
            } else if (__element.indexOf("UNS") != -1) {
                __atomic_number = 107;
                __atomic_mass = 262;
                __element = "UNS";
            } else if (__element.indexOf("XE") != -1) {
                __atomic_number = 54;
                __vdw_radius = 425;
                __atomic_mass = 131.3;
                __element = "XE";
            } else if (__element.indexOf("YB") != -1) {
                __atomic_number = 70;
                __vdw_radius = 485;
                __atomic_mass = 173;
                __element = "YB";
            } else if (__element.indexOf("ZN") != -1) {
                __vdw_radius = 362;
                __atomic_number = 30;
                __atomic_mass = 65.39;
                __element = "ZN";
            } else if (__element.indexOf("ZR") != -1) {
                __atomic_number = 40;
                __vdw_radius = 390;
                __atomic_mass = 91.22;
                __element = "ZR";
            } //The radii for C, H, N, O, P, S were changed to those used by the Richardsons' PROBE;
            //The old radii are commented out to the side
            else if (__element.toUpperCase().indexOf("C") == 0) {
                if (isCarbonylC()) { // means this atom is a carbonyl C
                    __vdw_radius = 165; //__vdw_radius = 180;
                    __atomic_number = 6;
                    __atomic_mass = 12.01;
                    __element = "C";
                } else {
                    __vdw_radius = 175; //__vdw_radius = 180;
                    __atomic_number = 6;
                    __atomic_mass = 12.01;
                    __element = "C";
                }
            } else if (__element.toUpperCase().indexOf("H") == 0) {
                if (isAromaticH() || isPolarH()) {
                    __vdw_radius = 100; //__vdw_radius = 80;
                    __atomic_number = 1;
                    __atomic_mass = 1;
                    __element = "H";
                } else {
                    __vdw_radius = 117; //__vdw_radius = 80;
                    __atomic_number = 1;
                    __atomic_mass = 1;
                    __element = "H";
                }
            } else if (__element.toUpperCase().indexOf("N") == 0) {
                __vdw_radius = 155; //__vdw_radius = 170;
                __atomic_number = 7;
                __atomic_mass = 14.01;
                __element = "N";
            } else if (__element.toUpperCase().indexOf("O") == 0) {
                __vdw_radius = 140; //__vdw_radius = 170;
                __atomic_number = 8;
                __atomic_mass = 16;
                __element = "O";
            } else if (__element.toUpperCase().indexOf("B") == 0) {
                __vdw_radius = 208;
                __atomic_number = 5;
                __atomic_mass = 10.81;
                __element = "B";
            } else if (__element.toUpperCase().indexOf("I") == 0) {
                __vdw_radius = 350;
                __atomic_number = 53;
                __atomic_mass = 126.9;
                __element = "I";
            } else if (__element.toUpperCase().indexOf("F") == 0) {
                __vdw_radius = 160;
                __atomic_number = 9;
                __atomic_mass = 19.0;
                __element = "F";
            } else if (__element.toUpperCase().indexOf("P") == 0) {
                __vdw_radius = 180; //__vdw_radius = 259;
                __atomic_number = 15;
                __atomic_mass = 30.97;
                __element = "P";
            } else if (__element.toUpperCase().indexOf("K") == 0) {
                __vdw_radius = 332;
                __atomic_number = 19;
                __atomic_mass = 39.1;
                __element = "K";
            } else if (__element.toUpperCase().indexOf("S") == 0) {
                __vdw_radius = 180; //__vdw_radius = 255;
                __atomic_number = 16;
                __atomic_mass = 32.07;
                __element = "S";
            } else if (__element.toUpperCase().indexOf("U") == 0) {
                __vdw_radius = 242;
                __atomic_number = 92;
                __atomic_mass = 238;
                __element = "U";
            } else if (__element.toUpperCase().indexOf("V") == 0) {
                __vdw_radius = 332;
                __atomic_number = 23;
                __atomic_mass = 50.94;
                __element = "V";
            } else if (__element.toUpperCase().indexOf("W") == 0) {
                __vdw_radius = 342;
                __atomic_number = 74;
                __atomic_mass = 183.9;
                __element = "W";
            } else if (__element.toUpperCase().indexOf("Y") == 0) {
                __vdw_radius = 445;
                __atomic_number = 39;
                __atomic_mass = 88.91;
                __element = "Y";
            } else {  //unrecognized atom type
                __vdw_radius = 10000;
                __element = myMiscConstants.illegal;
            }
        }
    }

    //Checks if this is a backbone atom (the atom name is one of N, CA, C, O, OXT, H, H1, H2, H3)
    //TODO: check new PDB format Changes. Check if 'HA' and 'CB' are to be considered at a BB atom.
    /**
     * Return true if the atom is a backbone atom.
     *
     * @return true if the atom is a backbone atom
     */
    boolean isBackboneAtom() {
        if (__atom_name.equalsIgnoreCase("N") || __atom_name.equalsIgnoreCase("CA") ||
                __atom_name.equalsIgnoreCase("C") || __atom_name.equalsIgnoreCase("O") ||
                __atom_name.equalsIgnoreCase("OXT") || __atom_name.equalsIgnoreCase("H") ||
                __atom_name.equalsIgnoreCase("HN") || __atom_name.equalsIgnoreCase("H1") ||
                __atom_name.equalsIgnoreCase("H2") || __atom_name.equalsIgnoreCase("H3")) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Return true if the oxygen atom is a carbonyl oxygen atom.
     *
     * @return true if the oxygen atom is a carbonyl oxygen atom
     */
    private boolean isCarbonylO() {
        if (__atom_name.toUpperCase().indexOf("O") == 0 && numberOfBonds() == 1) {
            myAtom carbonylC = bondIterator().next();
            if (carbonylC.numberOfBonds() <= 3) { // Actually it should be equal to 3, but we used <= to accomodate the case when the C terminus is missing.
                return true;
            }
        }
        return false;
    }

    /**
     * Return true if the carbon atom is a carbonyl carbon atom.
     * 
     * @return true if the carbon atom is a carbonyl carbon atom
     */
    private boolean isCarbonylC() {
        if (__atom_name.toUpperCase().indexOf("C") == 0 && numberOfBonds() <= 3) {
            BondIterator iter = bondIterator();
            for (; iter.hasNext();) {
                myAtom carbonylO = iter.next();
                if (carbonylO.getAtomName().toUpperCase().indexOf("O") == 0 && carbonylO.numberOfBonds() == 1) {
                    //System.out.println("Atom Name: " + getAtomName() + " Residue Number: " + getResidueNumber() + " vdw radius: " + getVdwRadius());
                    //System.out.print("^^^^^^^^^#bonds: " + carbonylO.numberOfBonds() + "^^^^^^^^^bonds: ");
                    //carbonylO.printBonds();
                    //carbonylO.print();
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * Return true if the proton is the part of a polar bond.
     * @return true if the proton is the part of a polar bond
     */
    private boolean isPolarH() { // We only consider H attached to N and O
        if (__atom_name.toUpperCase().indexOf("H") == 0 && numberOfBonds() == 1) {
            myAtom moreElectronNegativeAtom = bondIterator().next();
            if (moreElectronNegativeAtom.getAtomName().startsWith("N") || moreElectronNegativeAtom.getAtomName().startsWith("O")) {
                return true;
            }
        }
        return false;
    }

    /**
     * Return true if the proton is attached to an aromatic ring.
     *
     * @return true if the proton is attached to an aromatic ring
     */
    private boolean isAromaticH() { // We only consider H attached to N and O
        if (getResidueName().equalsIgnoreCase("PHE")) {
            if (__atom_name.equalsIgnoreCase("HD1") || __atom_name.equalsIgnoreCase("HD2") ||
                    __atom_name.equalsIgnoreCase("HE1") || __atom_name.equalsIgnoreCase("HE2") ||
                    __atom_name.equalsIgnoreCase("HZ")) {
                return true;
            }
            return false;
        } else if (getResidueName().equalsIgnoreCase("TRP")) {
            if (__atom_name.equalsIgnoreCase("HH2") || __atom_name.equalsIgnoreCase("HZ2") ||
                    __atom_name.equalsIgnoreCase("HE3") || __atom_name.equalsIgnoreCase("HZ3")) {
                return true;
            }
            return false;
        } else if (getResidueName().equalsIgnoreCase("TYR")) {
            if (__atom_name.equalsIgnoreCase("HD1") || __atom_name.equalsIgnoreCase("HD2") ||
                    __atom_name.equalsIgnoreCase("HE1") || __atom_name.equalsIgnoreCase("HE2") ||
                    __atom_name.equalsIgnoreCase("HH")) {
                return true;
            }
            return false;
        } else {
            return false;
        }
    }

    /**
     * Return the distance of this atom from the atom in the argument.
     *
     * @param atom the atom whose distance from this object is being sought
     * @return the distance
     */
    public double distance(myAtom atom) {
        return myPoint.distance(this.__coordinates, atom.__coordinates);
    }

    /**
     * Return the distance between two atoms in the argument.
     * @param atom1 the first atom
     * @param atom2 the second atom
     * @return the distance between the two atoms
     */
    public static double distance(myAtom atom1, myAtom atom2) {
        return myPoint.distance(atom1.__coordinates, atom2.__coordinates);
    }
}
