/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2009 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, 01 December, 2009
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.0, Nov 18, 2009
 * @author        Chittaranjan Tripathy (2007-2009)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */
import java.util.*;
import java.io.*;

/**
 * Description of the class
 */
public class myConstantsForProteinGeometry {

    /**
     * Default constructor which creates an instance of this class. This
     * constructor is private to enforce the singleton property of the class in
     * future.
     */
    private myConstantsForProteinGeometry() {
    }

    /**
     * This class parses the canonical residual dipolar coupling and chemical
     * shift anistropy parameters from a file and stores them as constants.
     */
    private static class proteinGeometryParameterParser {

        /**
         * The bond parameters read from the parameter file is stored in this map.
         */
        private Map<String, Double> __bond_parameter_map = new TreeMap<String, Double>();
        /**
         * The angle parameters read from the parameter file is stored in this map.
         */
        private Map<String, Double> __angle_parameter_map = new TreeMap<String, Double>();
        /**
         * The dihedral parameters read from the parameter file is stored in this map.
         */
        private Map<String, Double> __dihedral_parameter_map = new TreeMap<String, Double>();

        /**
         * This is the default constructor which reads from a file the protein
         * geometry parameters from a file and stores them as constants.
         */
        public proteinGeometryParameterParser() {
            String proteinGeometryParameterFile = System.getProperty("user.dir") + System.getProperty("file.separator") +
                    "mytoppar" + System.getProperty("file.separator") + "protein_geometry.txt";
            try {
                Scanner scanner = new Scanner(new File(proteinGeometryParameterFile));
                scanner.useDelimiter(System.getProperty("line.separator"));
                while (scanner.hasNextLine()) {
                    String line = myMiscUtilities.stripLineComment(scanner.nextLine()).trim();
                    if (line.length() != 0) {
                        String[] words = line.split("\\s+");
                        //System.out.println("words: " + Arrays.toString(words));
                        if (words[0].equalsIgnoreCase("bond")) {
                            __bond_parameter_map.put(words[0] + '_' + words[1] + '_' + words[2], Double.valueOf(words[3]));
                        }
                        if (words[0].equalsIgnoreCase("angle")) {
                            __angle_parameter_map.put(words[0] + '_' + words[1] + '_' + words[2] + '_' + words[3], Math.toRadians(Double.valueOf(words[4])));
                        }
                        if (words[0].equalsIgnoreCase("dihedral")) {
                            __dihedral_parameter_map.put(words[0] + '_' + words[1] + '_' + words[2] + '_' + words[3] + '_' + words[4], Math.toRadians(Double.valueOf(words[5])));
                        }
                    }
                }
                scanner.close();
            } catch (FileNotFoundException e) {
                System.out.println("Error: Input file " + proteinGeometryParameterFile + " not found");
                e.printStackTrace();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }
    /**
     * This object when constructed contains the set of constants related to the
     * protein geometry.
     */
    private static proteinGeometryParameterParser __parser = new proteinGeometryParameterParser();

    // bond lengths
    static final double dN_CA = __parser.__bond_parameter_map.get("bond_N_CA");
    static final double dCA_C = __parser.__bond_parameter_map.get("bond_CA_C");
    static final double dC_N = __parser.__bond_parameter_map.get("bond_C_N");
    static final double dC_O = __parser.__bond_parameter_map.get("bond_C_O");
    static final double dN_HN = __parser.__bond_parameter_map.get("bond_N_HN");
    static final double dCA_HA = __parser.__bond_parameter_map.get("bond_CA_HA");
    static final double dCA_CB = __parser.__bond_parameter_map.get("bond_CA_CB");
    
    // bond angles
    static final double angC_N_CA = __parser.__angle_parameter_map.get("angle_C_N_CA");
    static final double angN_CA_C = __parser.__angle_parameter_map.get("angle_N_CA_C");
    static final double angC_CA_CB = __parser.__angle_parameter_map.get("angle_C_CA_CB");
    static final double angCA_C_O = __parser.__angle_parameter_map.get("angle_CA_C_O");
    static final double angCA_C_N = __parser.__angle_parameter_map.get("angle_CA_C_N");
    static final double angN_C_O = __parser.__angle_parameter_map.get("angle_N_C_O");
    static final double angN_CA_CB = __parser.__angle_parameter_map.get("angle_N_CA_CB");

    static final double angHN_N_CA = __parser.__angle_parameter_map.get("angle_HN_N_CA");
    static final double angC_N_HN = __parser.__angle_parameter_map.get("angle_C_N_HN");
    static final double angN_CA_HA = __parser.__angle_parameter_map.get("angle_N_CA_HA");
    
    // dihedral angles

    static final double dihC_N_CA_HN = __parser.__dihedral_parameter_map.get("dihedral_C_N_CA_HN"); //C(i-1), magic angle
    static final double dihCA_C_N_O = __parser.__dihedral_parameter_map.get("dihedral_CA_C_N_O");
    static final double dihCA_C_N_CA = __parser.__dihedral_parameter_map.get("dihedral_CA_C_N_CA");

    static final double dihC_N_CA_HA = __parser.__dihedral_parameter_map.get("dihedral_C_N_CA_HA");
    static final double dihC_N_CA_CB = __parser.__dihedral_parameter_map.get("dihedral_C_N_CA_CB");
    static final double dihHA_N_CA_CB = __parser.__dihedral_parameter_map.get("dihedral_HA_N_CA_CB");
    


    public static void main(String... args) {
        //myNewConst ct = new myNewConst();
        //System.out.println("aa: " + getInstance().aa);
        System.out.println(myConstantsForProteinGeometry.dN_CA);
        System.out.println(myConstantsForProteinGeometry.angC_N_CA);

        System.out.println(__parser.__bond_parameter_map.size());
        System.out.println(__parser.__angle_parameter_map.size());
        System.out.println(__parser.__dihedral_parameter_map.size());

        System.out.println("Printing the bond parameter map");
        for (Map.Entry<String, Double> entry : __parser.__bond_parameter_map.entrySet()) {
            System.out.println(entry.getKey() + "  " + entry.getValue());
        }

        System.out.println("Printing the angle parameter map");
        for (Map.Entry<String, Double> entry : __parser.__angle_parameter_map.entrySet()) {
            System.out.println(entry.getKey() + "  " + entry.getValue());
        }

        System.out.println("Printing the dihedral parameter map");
        for (Map.Entry<String, Double> entry : __parser.__dihedral_parameter_map.entrySet()) {
            System.out.println(entry.getKey() + "  " + entry.getValue());
        }
    }
}
