/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2009 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, 01 December, 2009
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.0, Nov 18, 2009
 * @author        Chittaranjan Tripathy (2007-2009)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */
import java.io.*;
import java.util.*;

/**
 * Description of the class
 */
public class myDipolarCoupling {
    private String __first_atom_name = myMiscConstants.illegal;
    private int __first_atom_residue_number = 0;
    private String __second_atom_name = myMiscConstants.illegal;
    private int __second_atom_residue_number = 0;
    private double __rdc = 0;
    private double __error = 0;
    Type __type = null;
    
    /**
     * Residual dipolar coupling type.
     */
    public enum Type {
        N_HN, CA_HA, CA_C, CA_CB, C_N, C_HN, CB_HB, C_RCSA
    }

    public static boolean isPhiTypeRdc(Type type) {
        return type.equals(Type.CA_HA) || type.equals(Type.CA_C) || type.equals(Type.CA_CB);
    }

    public static boolean isPsiTypeRdc(Type type) {
        return type.equals(Type.N_HN) || type.equals(Type.C_HN) || type.equals(Type.C_N) || type.equals(Type.C_RCSA);
    }

    /**
     * Default constructor that constructs a default dipolar coupling object.
     */
    public myDipolarCoupling() {
    }

    public myDipolarCoupling(Type type) {
        String firstAtomName = myAtomLabel.__INVALID;
        String secondAtomName = myAtomLabel.__INVALID;
        int diff = 0; // diff = secondAtomResidueNumber - firstAtomResidueNumber

        switch (type) {
            case N_HN:
                firstAtomName = myAtomLabel.__N;
                secondAtomName = myAtomLabel.__H;
                diff = 0;
                break;
            case CA_HA:
                firstAtomName = myAtomLabel.__CA;
                secondAtomName = myAtomLabel.__HA;
                diff = 0;
                break;
            case CA_C:
                firstAtomName = myAtomLabel.__CA;
                secondAtomName = myAtomLabel.__C;
                diff = 0;
                break;
            case CA_CB:
                firstAtomName = myAtomLabel.__CA;
                secondAtomName = myAtomLabel.__CB;
                diff = 0;
                break;
            case C_N:
                firstAtomName = myAtomLabel.__C;
                secondAtomName = myAtomLabel.__N;
                diff = 1;
                break;
            case C_HN:
                firstAtomName = myAtomLabel.__C;
                secondAtomName = myAtomLabel.__H;
                diff = 1;
                break;
            case CB_HB:
                firstAtomName = myAtomLabel.__CB;
                secondAtomName = myAtomLabel.__HB;
                diff = 0;
                break;
            case C_RCSA:
                firstAtomName = myAtomLabel.__C_CSAv1;
                secondAtomName = myAtomLabel.__C_CSAv2;
                diff = 0;
                break;
            default:
                System.out.println("Error: Illegal RDC type");
                System.exit(1);
        }
        __first_atom_name = firstAtomName;
        __first_atom_residue_number = 1;
        __second_atom_name = secondAtomName;
        __second_atom_residue_number = __first_atom_residue_number + diff;
        __rdc = 0.0;
        __error = 0.0;

        myDipolarCouplingConstructorHelper();
    }

    public myDipolarCoupling(Type type, int residueNumber) {
        String firstAtomName = myAtomLabel.__INVALID;
        String secondAtomName = myAtomLabel.__INVALID;
        int diff = 0; // diff = secondAtomResidueNumber - firstAtomResidueNumber

        switch (type) {
            case N_HN:
                firstAtomName = myAtomLabel.__N;
                secondAtomName = myAtomLabel.__H;
                diff = 0;
                break;
            case CA_HA:
                firstAtomName = myAtomLabel.__CA;
                secondAtomName = myAtomLabel.__HA;
                diff = 0;
                break;
            case CA_C:
                firstAtomName = myAtomLabel.__CA;
                secondAtomName = myAtomLabel.__C;
                diff = 0;
                break;
            case CA_CB:
                firstAtomName = myAtomLabel.__CA;
                secondAtomName = myAtomLabel.__CB;
                diff = 0;
                break;
            case C_N:
                firstAtomName = myAtomLabel.__C;
                secondAtomName = myAtomLabel.__N;
                diff = 1;
                break;
            case C_HN:
                firstAtomName = myAtomLabel.__C;
                secondAtomName = myAtomLabel.__H;
                diff = 1;
                break;
            case CB_HB:
                firstAtomName = myAtomLabel.__CB;
                secondAtomName = myAtomLabel.__HB;
                diff = 0;
                break;
            case C_RCSA:
                firstAtomName = myAtomLabel.__C_CSAv1;
                secondAtomName = myAtomLabel.__C_CSAv2;
                diff = 0;
                break;
            default:
                System.out.println("Error: Illegal RDC type");
                System.exit(1);
        }
        __first_atom_name = firstAtomName;
        __second_atom_name = secondAtomName;
        __first_atom_residue_number = residueNumber - diff;
        __second_atom_residue_number = residueNumber;
        __rdc = 0.0;
        __error = 0.0;

        myDipolarCouplingConstructorHelper();
    }

    /**
     * Constructor that constructs a dipolar coupling object from the arguments.
     *
     * @param firstAtomName the name of the first atom
     * @param firstAtomResidueNumber the residue number of the first atom
     * @param secondAtomName the name of the second atom
     * @param secondAtomResidueNumber the residue number of the second atom
     * @param rdc value of the rdc
     * @param error error in the rdc measurement
     */
    public myDipolarCoupling(String firstAtomName, int firstAtomResidueNumber, String secondAtomName, int secondAtomResidueNumber, double rdc, double error) {
        __first_atom_name = firstAtomName.trim();
        __first_atom_residue_number = firstAtomResidueNumber;
        __second_atom_name = secondAtomName.trim();
        __second_atom_residue_number = secondAtomResidueNumber;
        __rdc = rdc;
        __error = error;

        myDipolarCouplingConstructorHelper();
    }

    private void myDipolarCouplingConstructorHelper() {

        if (__first_atom_name.equalsIgnoreCase("H")) {
            __first_atom_name = "HN";
        }
        if (__second_atom_name.equalsIgnoreCase("H")) {
            __second_atom_name = "HN";
        }

        // Swap the atom names and atom residue numbers so that the first atom is always N or C
        if (__first_atom_name.matches("[H].*") && __second_atom_name.matches("[NC].*")) {
            String temp = __first_atom_name;
            __first_atom_name = __second_atom_name;
            __second_atom_name = temp;
            int temp2 = __first_atom_residue_number;
            __first_atom_residue_number = __second_atom_residue_number;
            __second_atom_residue_number = temp2;
        }

        // Swap the atom names so that the first atom is C and the second atom is N
        if (__first_atom_name.equalsIgnoreCase("N") && __second_atom_name.equalsIgnoreCase("C")) {
            String temp = __first_atom_name;
            __first_atom_name = __second_atom_name;
            __second_atom_name = temp;
            int temp2 = __first_atom_residue_number;
            __first_atom_residue_number = __second_atom_residue_number;
            __second_atom_residue_number = temp2;
        }

        if (__first_atom_name.equalsIgnoreCase("C") && __second_atom_name.equalsIgnoreCase("CA")) {
            String temp = __first_atom_name;
            __first_atom_name = __second_atom_name;
            __second_atom_name = temp;
        }

        // Set the dipolar coupling type
        if (__first_atom_name.equalsIgnoreCase("N") && __second_atom_name.startsWith("HN")) {
            __type = Type.N_HN;
        } else if (__first_atom_name.equalsIgnoreCase("CA") && __second_atom_name.startsWith("HA")) { // GLY can have HA2 and HA3
            __type = Type.CA_HA;
        } else if (__first_atom_name.startsWith("CA") && __second_atom_name.equalsIgnoreCase("C")) {
            __type = Type.CA_C;
        } else if (__first_atom_name.startsWith("CA") && __second_atom_name.equalsIgnoreCase("CB")) {
            __type = Type.CA_CB;
        } else if (__first_atom_name.equalsIgnoreCase("C") && __second_atom_name.equalsIgnoreCase("N")) {
            __type = Type.C_N;
        } else if (__first_atom_name.equalsIgnoreCase("C") && __second_atom_name.startsWith("HN")) {
            __type = Type.C_HN;
        } else if (__first_atom_name.equalsIgnoreCase("CB") && __second_atom_name.startsWith("HB")) {
            __type = Type.CB_HB;
        } else if (__first_atom_name.startsWith("C") && __second_atom_name.startsWith("O")) {
            __type = Type.C_RCSA;
        } else {
            System.out.println("Error: Unable to recognize the dipolar coupling type while constructing the dipolar coupling object");
            System.exit(1);
        }
    }


    /**
     * A copy constructor which takes an instance of this class and constructs
     * another instance with all fields copied appropriately.
     *
     * @param dc the object which is to be cloned
     */
    public myDipolarCoupling(final myDipolarCoupling dc) {
        __first_atom_name = dc.__first_atom_name;
        __first_atom_residue_number = dc.__first_atom_residue_number;
        __second_atom_name = dc.__second_atom_name;
        __second_atom_residue_number = dc.__second_atom_residue_number;
        __rdc = dc.__rdc;
        __error = dc.__error;
        __type = dc.__type;
    }

    /**
     * Return the first atom residue number.
     *
     * @return the first atom residue number
     */
    public int getFirstAtomResidueNumber() {
        return __first_atom_residue_number;
    }

    /**
     * Return the second atom residue number.
     *
     * @return the second atom residue number
     */
    public int getSecondAtomResidueNumber() {
        return __second_atom_residue_number;
    }

    /**
     * Return the first atom residue name.
     *
     * @return the first atom residue name
     */
    public String getFirstAtomName() {
        if (__first_atom_name.equalsIgnoreCase("HN"))
            return myAtomLabel.__H;
        return __first_atom_name;
    }

    /**
     * Return the second atom residue name.
     *
     * @return the second atom residue name
     */
    public String getSecondAtomName() {
        if (__second_atom_name.equalsIgnoreCase("HN"))
            return myAtomLabel.__H;
        return __second_atom_name;
    }

    /**
     * Return the residue number of the dipolar coupling object. The residue number
     * here is defined to be the larger of residue numbers of the atoms that define
     * the dipolar coupling object.
     *
     * @return the residue number of the dipolar coupling object.
     */
    public int getResidueNumber() {
        if (__first_atom_residue_number != __second_atom_residue_number)
            ;//System.out.println("The dipolar coupling is between atoms in two different residues");
        return Math.max(__first_atom_residue_number, __second_atom_residue_number);
    }

    /**
     * Return the value of the dipolar coupling.
     *
     * @return the value of the dipolar coupling
     */
    public double getRdc() {
        return __rdc;
    }

    /**
     * Modify the value of the dipolar coupling.
     *
     * @param rdc the new value of the dipolar coupling
     */
    public void setRdc(double rdc) {
        __rdc = rdc;
    }

    /**
     * Return the error in the dipolar coupling.
     *
     * @return the error in the dipolar coupling
     */
    public double getError() {
        return __error;
    }

    /**
     * Modify the value of the error in the dipolar coupling.
     *
     * @param error the error in the dipolar coupling
     */
    public void setError(double error) {
        __error = error;
    }

    /**
     * Return the dipolar coupling type.
     *
     * @return the dipolar coupling type
     */
    public Type getType() {
        return __type;
    }

    /**
     * Return the string representation of the dipolar coupling object.
     *
     * @return the string representation of the dipolar coupling object
     */
    @Override public String toString() {
        String myself = String.valueOf(__first_atom_residue_number) + __first_atom_name +
                '-' + String.valueOf(__second_atom_residue_number) + __second_atom_name +
                "    " + String.valueOf(__rdc) + "    " + String.valueOf(__error);
        return myself;
    }

    /**
     * Return the xplor format string representation of the dipolar coupling object.
     *
     * @return the xplor format string representation of the dipolar coupling object
     */
    public String toStringInXplorFormat() {
        StringBuilder sb = new StringBuilder(80);
        Formatter formatter = new Formatter(sb, Locale.US);

        formatter.format("%s", "assign ( resid 500  and name OO  )\n       ( resid 500  and name Z   )\n       ( resid 500  and name X   )\n       ( resid 500  and name Y   )\n");
        formatter.format("%s%-4d%2s%-2s%s\n", "       ( resid ", __first_atom_residue_number, " and name ", __first_atom_name, "  )");
        formatter.format("%s%-4d%2s%-2s%s%8.4f %8.4f\n", "       ( resid ", __second_atom_residue_number, " and name ", __second_atom_name, "  )", __rdc, __error);

        //System.out.println(sb.toString());
        return sb.toString();
    }

    /**
     * Read the dipolar coupling object file in CYANA format.
     *
     * @param rdcFileName the file containing the dipolar coupling information
     * @return a vector of dipolar coupling objects
     */
    public static Vector<myDipolarCoupling> parseRdcFileInCyanaFormat(String rdcFileName) {
        
        Vector<myDipolarCoupling> myRdcVec = new Vector<myDipolarCoupling>();
        String firstAtomName = myMiscConstants.illegal;
        String secondAtomName = myMiscConstants.illegal;
        int firstAtomResidueNumber = 0;
        int secondAtomResidueNumber = 0;
        double rdc = 0;
        double error = 0;
        
        try {
            Scanner scanner = new Scanner(new File(rdcFileName));
            scanner.useDelimiter(System.getProperty("line.separator"));
            while (scanner.hasNextLine()) {
                String line = scanner.nextLine().trim();
                if (line.length() != 0) {
                    String[] words = line.split("\\s+");
                    if (words[0].matches("[0-9]+") && words[1].matches("[HNC].*")) {
                        firstAtomResidueNumber = Integer.parseInt(words[0]);
                        firstAtomName = words[1];
                        secondAtomResidueNumber = Integer.parseInt(words[2]);
                        secondAtomName = words[3];
                        rdc = Double.parseDouble(words[4]);
                        if (words.length >= 6) {
                            error = Double.parseDouble(words[5]);
                        }
                        myDipolarCoupling myDC = new myDipolarCoupling(firstAtomName, firstAtomResidueNumber, secondAtomName, secondAtomResidueNumber, rdc, error);
                        myRdcVec.add(myDC);
                    }
                }
            }
            scanner.close();
        } catch (FileNotFoundException e) {
            System.out.println("Error: Input file " + rdcFileName + " not found");
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }
        
        Collections.sort(myRdcVec, new dipolarCouplingComparator());
        return myRdcVec;
    }

    /**
     * Return true if the file is in Xplor format and false if it is in Cyana format.
     *
     * @param rdcFileName the file containing the dipolar coupling information
     * @return true if the file is in Xplor format and false if it is in Cyana format
     */
    public static boolean testIfXplorFormatOrCyanaFormat(String rdcFileName) {
        boolean xplorFormat = true;
        try {
            Scanner scanner = new Scanner(new File(rdcFileName));
            scanner.useDelimiter(System.getProperty("line.separator"));
            while (scanner.hasNextLine()) {
                String line = scanner.nextLine().trim();
                if (line.length() != 0) {
                    String[] words = line.split("\\s+");
                    //System.out.println("words: " + Arrays.toString(words));
                    if (words[0].matches("[0-9]+") && words[1].matches("[HNC].*") && words[2].matches("[0-9]+") && words[1].matches("[HNC].*")) {
                        xplorFormat = false;
                    }
                    if (words.length >= 7) {
                        if (words[0].matches("[0-9]+") && words[1].length() == 3 && words[2].matches("[HNC].*") &&
                                words[3].matches("[0-9]+") && words[4].length() == 3 && words[5].matches("[HNC].*")) {
                        xplorFormat = false;
                    }
                    }
                }
            }
        } catch (FileNotFoundException e) {
            System.out.println("Error: Input file " + rdcFileName + " not found");
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }
        System.out.println("xplorFormat: " + xplorFormat);
        return xplorFormat;
    }

    /**
     * Read the dipolar coupling object file in Xplor format.
     *
     * @param rdcFileName the file containing the dipolar coupling information
     * @return a vector of dipolar coupling object
     */
    public static Vector<myDipolarCoupling> parseRdcFileInXplorFormat(String rdcFileName) {

        Vector<myDipolarCoupling> myRdcVec = new Vector<myDipolarCoupling>();
        String firstAtomName = myMiscConstants.illegal;
        String secondAtomName = myMiscConstants.illegal;
        int firstAtomResidueNumber = 0;
        int secondAtomResidueNumber = 0;
        double rdc = 0;
        double error = 0;

        try {
            Scanner scanner = new Scanner(new File(rdcFileName));
            scanner.useDelimiter(System.getProperty("line.separator"));
            int flag = 0; // this is used to make sure that we are reading even number of useful lines
            while (scanner.hasNextLine()) {
                String line = scanner.nextLine();
                line = line.replaceAll("segid\\s+[A-Za-z]+\\s+and\\s+", " "); // for format: ( segid A and resid 29   and name HN  ) -23.160 0.2000
                line = line.replaceAll("\\(|\\)", " ").trim();
                //System.out.println(line);
                if (line.length() != 0) {
                    String[] words = line.split("\\s+");
                    //System.out.println("line: " + line);
                    if (words[1].matches("[0-9]+") && words[4].matches("[HNC].*")) {
                        if (words.length == 5) {
                            //System.out.println(line);
                            firstAtomName = words[4];
                            firstAtomResidueNumber = Integer.parseInt(words[1]);
                            flag++;
                        } else if (words.length > 5) {
                            //System.out.println(line);
                            secondAtomName = words[4];
                            secondAtomResidueNumber = Integer.parseInt(words[1]);
                            rdc = Double.parseDouble(words[5]);
                            error = Double.parseDouble(words[6]);
                            myDipolarCoupling myDC = new myDipolarCoupling(firstAtomName, firstAtomResidueNumber, secondAtomName, secondAtomResidueNumber, rdc, error);
                            myRdcVec.add(myDC);
                            if (flag == 1) {
                                flag = 0;
                            } else {
                                System.out.println("Error: Incorrect input format");
                            }
                        }
                    }
                }
            }
            scanner.close();
        } catch (FileNotFoundException e) {
            System.out.println("Error: Input file " + rdcFileName + " not found");
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }

        Collections.sort(myRdcVec, new dipolarCouplingComparator());
        return myRdcVec;
    }

    /**
     * Return a vector of dipolar coupling objects for the secondary structure
     * element specified by reading the file containing the dipolar coupling
     * information.
     * 
     * @param rdcFileName the file containing the dipolar coupling information
     * @param sseId the specification of secondary structure element boundary
     * @return a vector of dipolar coupling objects for the secondary structure element
     */
    @Deprecated public static Vector<myDipolarCoupling> extractRdcForSse(String rdcFileName, final mySseInfo sseId) {
        Vector<myDipolarCoupling> rdcVec = null;
        boolean isXplorFormat = testIfXplorFormatOrCyanaFormat(rdcFileName);

        if (isXplorFormat) {
            rdcVec = parseRdcFileInXplorFormat(rdcFileName);
        } else {
            rdcVec = parseRdcFileInCyanaFormat(rdcFileName);
        }
        
        Vector<myDipolarCoupling> thisSseRdcVec = new Vector<myDipolarCoupling>();

        for (int i = 0; i < sseId.getEndResidueNumber() - sseId.getBeginResidueNumber() + 1; i++) {
            int residueNo = sseId.getBeginResidueNumber() + i;
            int index = Collections.binarySearch(rdcVec, new myDipolarCoupling("", residueNo, "", residueNo, 0, 0), new dipolarCouplingComparator());
            if (index > -1) {
                thisSseRdcVec.add(rdcVec.elementAt(index));
            }
        }
        return thisSseRdcVec;
    }

    public static Vector<myDipolarCoupling> extractRdcs(String rdcFileName) {
        Vector<myDipolarCoupling> rdcVec = null;
        boolean isXplorFormat = testIfXplorFormatOrCyanaFormat(rdcFileName);

        if (isXplorFormat) {
            rdcVec = parseRdcFileInXplorFormat(rdcFileName);
        } else {
            rdcVec = parseRdcFileInCyanaFormat(rdcFileName);
        }

        return rdcVec;
    }

    /**
     * A class to compare two dipolar coupling objects.
     */
    public static class dipolarCouplingComparator implements Comparator<myDipolarCoupling> {
        /**
         * Compare two dipolar coupling objects where the basis of comparison is
         * the residue number.
         * @param dc1 the first dipolar coupling object
         * @param dc2 the second dipolar coupling object
         * @return -1 if the first object's residue number is less than that of
         * the second object, 0 if the residue numbers are equal; otherwise, 1
         */
        public int compare(myDipolarCoupling dc1, myDipolarCoupling dc2) {
//            if (!dc1.__type.equals(dc2.__type)) {
//                System.out.println("Error: the two dipolar couplings cannot be compared as they are of different type");
//                System.exit(1);
//            }
            
            int d1 = dc1.getResidueNumber();
            int d2 = dc2.getResidueNumber();
            if (d1 < d2) {
                return -1;
            } else if (d1 > d2) {
                return 1;
            } else {
                return 0;
            }
        }
    }

    /**
     * The main method tests this class.
     *
     * @param args
     */
    public static void main(String... args) {

        Type t1 = Type.CA_C;

        Type t2 = t1;
        t1 = Type.C_RCSA;

        System.out.println(t1.toString() + t2.toString());

        int medium = 43;
        String mediumName = "medium" + medium;
        System.out.println(mediumName);
        System.exit(1);


        String userDir = System.getProperty("user.dir");
        String fileSeparator = System.getProperty("file.separator");
        String inputDirectory = userDir + fileSeparator + "inputFiles" + fileSeparator;
        String inputFile = inputDirectory + "ubm2_rdc_ha.tbl"; //"trdc.txt";
        System.out.println("inputFile: " + inputFile);

        Vector<myDipolarCoupling> myRdcVec = parseRdcFileInXplorFormat(inputFile);

        for (myDipolarCoupling dc : myRdcVec)
            System.out.println(dc.toString());

        myRdcVec = extractRdcForSse(inputFile, new mySseInfo(78, 92, "H"));

        System.out.println("*************************");
        for (myDipolarCoupling dc : myRdcVec)
            System.out.println(dc.toString());

        // Testing for cyana format
        inputFile = inputDirectory + "n_hn_rdc_gb1_Blackledge.txt"; //"trdc.txt";
        myRdcVec = parseRdcFileInCyanaFormat(inputFile);

        for (myDipolarCoupling dc : myRdcVec)
            System.out.println(dc.toString());

        myRdcVec = extractRdcForSse(inputFile, new mySseInfo(60, 65, "H"));

        System.out.println("*************************");
        for (myDipolarCoupling dc : myRdcVec)
            System.out.println(dc.toString());
    }
}
