/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2009 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, 01 December, 2009
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.0, Nov 18, 2009
 * @author        Chittaranjan Tripathy (2007-2009)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */
import java.util.*;

/**
 * Description of the class
 */
public class myDipolarCouplingTable {

    private Map<String, Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>>> rdcCsaTable = new TreeMap<String, Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>>>();

    myDipolarCouplingTable(String... rdcFiles) {
        int mediumNumber = 1;
        for (String rdcFileName : rdcFiles) {
            // Extract all the RDCs from the file
            Vector<myDipolarCoupling> rdcVec = myDipolarCoupling.extractRdcs(rdcFileName);

            Vector<myDipolarCoupling> NHnRdcVec = new Vector<myDipolarCoupling>();
            Vector<myDipolarCoupling> CaHaRdcVec = new Vector<myDipolarCoupling>();
            Vector<myDipolarCoupling> CaCRdcVec = new Vector<myDipolarCoupling>();
            Vector<myDipolarCoupling> CaCbRdcVec = new Vector<myDipolarCoupling>();
            Vector<myDipolarCoupling> CNRdcVec = new Vector<myDipolarCoupling>();
            Vector<myDipolarCoupling> CHnRdcVec = new Vector<myDipolarCoupling>();
            Vector<myDipolarCoupling> CbHbRdcVec = new Vector<myDipolarCoupling>();
            Vector<myDipolarCoupling> CRcsaVec = new Vector<myDipolarCoupling>();

            // Put the RDCs into different bins depending on their types
            for (myDipolarCoupling dc : rdcVec) {
                myDipolarCoupling.Type type = dc.getType();
                switch (type) {
                    case N_HN:
                        NHnRdcVec.add(dc);
                        break;
                    case CA_HA:
                        CaHaRdcVec.add(dc);
                        break;
                    case CA_C:
                        CaCRdcVec.add(dc);
                        break;
                    case CA_CB:
                        CaCbRdcVec.add(dc);
                        break;
                    case C_N:
                        CNRdcVec.add(dc);
                        break;
                    case C_HN:
                        CHnRdcVec.add(dc);
                        break;
                    case CB_HB:
                        CbHbRdcVec.add(dc);
                        break;
                    case C_RCSA:
                        CRcsaVec.add(dc);
                        break;
                    default:
                }
            }

            // Sort the bins.
            Collections.sort(NHnRdcVec, new myDipolarCoupling.dipolarCouplingComparator());
            Collections.sort(CaHaRdcVec, new myDipolarCoupling.dipolarCouplingComparator());
            Collections.sort(CaCRdcVec, new myDipolarCoupling.dipolarCouplingComparator());
            Collections.sort(CaCbRdcVec, new myDipolarCoupling.dipolarCouplingComparator());
            Collections.sort(CNRdcVec, new myDipolarCoupling.dipolarCouplingComparator());
            Collections.sort(CHnRdcVec, new myDipolarCoupling.dipolarCouplingComparator());
            Collections.sort(CbHbRdcVec, new myDipolarCoupling.dipolarCouplingComparator());
            Collections.sort(CRcsaVec, new myDipolarCoupling.dipolarCouplingComparator());

            // Create a table of RDCs
            Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMedium = new TreeMap<myDipolarCoupling.Type, Vector<myDipolarCoupling>>();

//            rdcCsaTableForThisMedium.put("N_HN", NHnRdcVec);
//            rdcCsaTableForThisMedium.put("CA_HA", CaHaRdcVec);
//            rdcCsaTableForThisMedium.put("CA_C", CaCRdcVec);
//            rdcCsaTableForThisMedium.put("CA_CB", CaCbRdcVec);
//            rdcCsaTableForThisMedium.put("C_N", CNRdcVec);
//            rdcCsaTableForThisMedium.put("C_HN", CHnRdcVec);
//            rdcCsaTableForThisMedium.put("CB_HB", CbHbRdcVec);
//            rdcCsaTableForThisMedium.put("C_RCSA", CRcsaVec);

            rdcCsaTableForThisMedium.put(myDipolarCoupling.Type.N_HN, NHnRdcVec);
            rdcCsaTableForThisMedium.put(myDipolarCoupling.Type.CA_HA, CaHaRdcVec);
            rdcCsaTableForThisMedium.put(myDipolarCoupling.Type.CA_C, CaCRdcVec);
            rdcCsaTableForThisMedium.put(myDipolarCoupling.Type.CA_CB, CaCbRdcVec);
            rdcCsaTableForThisMedium.put(myDipolarCoupling.Type.C_N, CNRdcVec);
            rdcCsaTableForThisMedium.put(myDipolarCoupling.Type.C_HN, CHnRdcVec);
            rdcCsaTableForThisMedium.put(myDipolarCoupling.Type.CB_HB, CbHbRdcVec);
            rdcCsaTableForThisMedium.put(myDipolarCoupling.Type.C_RCSA, CRcsaVec);

            String medium = "medium" + mediumNumber;
            rdcCsaTable.put(medium, rdcCsaTableForThisMedium);
            mediumNumber++;
        }
    }

    Vector<myDipolarCoupling> getRdcVector(String mediumName, myDipolarCoupling.Type rdcType) {
        Vector<myDipolarCoupling> v = rdcCsaTable.get(mediumName).get(rdcType);
        return v;
    }

    myDipolarCoupling getRdc(String mediumName, myDipolarCoupling.Type rdcType, int residueNumber) {
        Vector<myDipolarCoupling> v = getRdcVector(mediumName, rdcType);
        myDipolarCoupling proxy = new myDipolarCoupling(rdcType, residueNumber);

        int index = Collections.binarySearch(v, proxy, new myDipolarCoupling.dipolarCouplingComparator());
        if (index > -1) {
            return v.elementAt(index);
        }
        return null;
    }
    
    Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> getRdcMapForThisMedium(String mediumName) {
        return rdcCsaTable.get(mediumName);
    }

    public void print() {
        System.out.println("************Printing RDC/CSA Table************");
        for (Map.Entry<String, Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>>> entry : rdcCsaTable.entrySet()) {            
            System.out.println("Medium: " + entry.getKey());
            Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMedium = entry.getValue();
            for (Map.Entry<myDipolarCoupling.Type, Vector<myDipolarCoupling>> entry2 : rdcCsaTableForThisMedium.entrySet()) {
                System.out.println("The RDC type is: " + entry2.getKey());
                Vector<myDipolarCoupling> rdcVec = entry2.getValue();
                System.out.println(rdcVec);
            }            
        }
        System.out.println("**********************************************");
    }

    public static void printInXplorFormat(Map<myDipolarCoupling.Type, Vector<myDipolarCoupling>> rdcCsaTableForThisMedium) {
        System.out.println("************Printing RDC/CSA Table (for this medium)************");
        for (Map.Entry<myDipolarCoupling.Type, Vector<myDipolarCoupling>> entry2 : rdcCsaTableForThisMedium.entrySet()) {
            System.out.println("The RDC type is: " + entry2.getKey());
            Vector<myDipolarCoupling> rdcVec = entry2.getValue();
            //System.out.println(rdcVec);
            for (myDipolarCoupling dc : rdcVec) {
                System.out.println(dc.toStringInXplorFormat());
            }
        }
        System.out.println("****************************************************************");

    }
    
}
