/*
 * This file is part of RDC-ANALYTIC.
 *
 * RDC-ANALYTIC Protein Backbone Structure Determination Software Version 1.0
 * Copyright (C) 2001-2009 Bruce Donald Lab, Duke University
 *
 * RDC-ANALYTIC is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RDC-ANALYTIC is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, see:
 *     <http://www.gnu.org/licenses/>.
 *
 * There are additional restrictions imposed on the use and distribution of this
 * open-source code, including: (A) this header must be included in any
 * modification or extension of the code; (B) you are required to cite our
 * papers in any publications that use this code. The citation for the various
 * different modules of our software, together with a complete list of
 * requirements and restrictions are found in the document license.pdf enclosed
 * with this distribution.
 *
 * Contact Info:
 *     Bruce R. Donald
 *     Duke University
 *     Department of Computer Science
 *     Levine Science Research Center (LSRC)
 *     Durham, NC 27708-0129
 *     USA
 *     email: www.cs.duke.edu/brd/
 *
 * <signature of Bruce Donald>, 01 December, 2009
 * Bruce R. Donald, Professor of Computer Science and Biochemistry
 */

/**
 * @version       1.0.0, Nov 18, 2009
 * @author        Chittaranjan Tripathy (2007-2009)
 * @email         chittu@cs.duke.edu
 * @organization  Duke University
 */

/**
 * Package specification
 */
package analytic;

/**
 * Import statement(s)
 */
import java.util.*;
import java.io.Writer;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.FileWriter;
import java.io.File;

/**
 * Description of the class
 */
public class myPhiPsiSolutionTree {

    private myProtein __best_fragment = null;
    //public int __sampled_rdc_set_number = 0;
    private int __counter_for_total_number_of_solutions = 0;
    private double __min_score = Double.MAX_VALUE;
    private double __min_clash_score = Double.MAX_VALUE; // to be used later
    private boolean __verbose = false;
    private int __max_depth = 0;
    
    /**
     * This enum defines the types of dihedrals exist in the protein backbone
     * chain.
     */
    public static enum myDihedral {

        PHI, PSI, OMEGA
    };

    /**
     * This class represents a degree of freedom in protein backbone chain. An
     * instance of this class represent a dihedral (any of phi, psi and omega)
     * along with the residue number.
     */
    public class myDof {

        private int __residue_number;
        private myDihedral __dof;

        public myDof(int resNum, myDihedral dof) {
            __residue_number = resNum;
            __dof = dof;
        }

        /**
         * Returns the residue number of the residue for which this dihedral object
         * is defined.
         *
         * @return the residue number of the residue for which this dihedral object
         * is defined
         */
        public int getResidueNumber() {
            return __residue_number;
        }

        /**
         * Returns the dihedral type for which this dihedral object is defined.
         *
         * @return the dihedral type for which this dihedral object is defined
         */
        public myDihedral getDof() {
            return __dof;
        }

        /**
         * Returns the string representation of the object.
         * 
         * @return the string representation of the object
         */
        @Override
        public String toString() {
            return "Residue: " + getResidueNumber() + "    DOF: " + getDof();
        }
    }

    /**
     * Returns a vector of backbone dihedrals (degrees of freedom) to be computed.
     *
     * @param thisSseId the secondary structure boundary specification
     * @return a vector of backbone dihedrals (degrees of freedom) to be computed
     */
    private Vector<myDof> getDofsToModify(final mySseInfo thisSseId) {
        Vector<myDof> dofs = new Vector<myDof>();
        for (int i = thisSseId.getBeginResidueNumber(); i <= thisSseId.getEndResidueNumber(); i++) {
            dofs.add(new myDof(i, myDihedral.PHI));
            dofs.add(new myDof(i, myDihedral.PSI));
        }
        return dofs;
    }

    /**
     * Returns a copy of the best fragment over a set of DFS searches one per each
     * set of sampled RDCs.
     *
     * @return a copy of the best fragment
     */
    public myProtein getBestFragment() {
        if (__best_fragment == null)
            return null;
        else return new myProtein(__best_fragment);
    }

    /**
     * Returns the maximum search depth achieved so far while searching different
     * DFS trees (one tree for each set of sampled RDCs).
     *
     * @return the maximum search depth
     */
    public int getMaxSearchDepth() {
        return __max_depth;
    }

    /**
     * Returns the number of solutions found thus far among all searches of the
     * DFS trees (one tree for each set of sampled RDCs).
     * 
     * @return the number of solutions found thus far
     */
    public int getSolutionCounter() {
        return __counter_for_total_number_of_solutions;
    }

    /**
     * This method invokes the recursive depth-first search based fragment
     * computation from the given set of sampled RDCs and input parameters.
     *
     * @param thisSseId the secondary structure element boundary specification
     * @param phiTypeRdcSampled the sampled phi-defining RDC set
     * @param psiTypeRdcSampled the sampled psi-defining RDC set
     * @param phiTypeRdcVecWithMissingRdcsFilled the sampled phi-defining RDC set with missing RDCs filled
     * @param psiTypeRdcVecWithMissingRdcsFilled the sampled psi-defining RDC set with missing RDCs filled
     * @param Rg the rotation matrix that specifies the ralative rotation of the first peptide plane (of the fragment) with respect to the principal order frame
     * @param Syy the diagonilized alignment tensor component
     * @param Szz the diagonilized alignment tensor component
     * @param firstPeptidePlane the first peptide plane of the fragment
     * @param phiPsiSolutionFile the file which stores the set of computed phi and psi sequences
     * @param weightInScoringFunction the relative weight used in scoring function
     * @param rightHand
     */
    public void phiPsiSolutionTreeRootCaller(final mySseInfo thisSseId,
            final Vector<myDipolarCoupling> phiTypeRdcSampled, final Vector<myDipolarCoupling> psiTypeRdcSampled,
            final Vector<myDipolarCoupling> phiTypeRdcVecWithMissingRdcsFilled, final Vector<myDipolarCoupling> psiTypeRdcVecWithMissingRdcsFilled,
            Matrix Rg, double Syy, double Szz, final myPeptidePlane firstPeptidePlane,
            Writer phiPsiSolutionFile, final double weightInScoringFunction, final boolean rightHand) {

        Vector<myDof> dofsToBeComputed = getDofsToModify(thisSseId);
        // With phiType and psiType RDCs the psi angle of the very last residue
        // cannot be computed as the psiType RDC data is not available for it.
        // Therefore, we remove the last psi from the dofsToBeComputed.
        dofsToBeComputed.remove(dofsToBeComputed.size() - 1);

        double[] ppS = new double[dofsToBeComputed.size()];
        int initialSearchDepth = 0;
        int maxSearchDepth = dofsToBeComputed.size();

        recursiveDepthFirstSearchOfPhiPsiSolutionTree(thisSseId, phiTypeRdcSampled, psiTypeRdcSampled, phiTypeRdcVecWithMissingRdcsFilled, psiTypeRdcVecWithMissingRdcsFilled,
            Rg, Syy, Szz, initialSearchDepth, maxSearchDepth, ppS, firstPeptidePlane, phiPsiSolutionFile, weightInScoringFunction, rightHand, dofsToBeComputed);
    }

    /**
     * Do depth-first search of the analytic solution tree to compute the set of all
     * possible phi, psi sequences from which a set of possible fragments can be
     * constructed.
     *
     * @param thisSseId the secondary structure element boundary specification
     * @param phiTypeRdcSampled the sampled phi-defining RDC set
     * @param psiTypeRdcSampled the sampled psi-defining RDC set
     * @param phiTypeRdcVecWithMissingRdcsFilled the sampled phi-defining RDC set with missing RDCs filled
     * @param psiTypeRdcVecWithMissingRdcsFilled the sampled psi-defining RDC set with missing RDCs filled
     * @param Rg the rotation matrix that specifies the ralative rotation of the first peptide plane (of the fragment) with respect to the principal order frame
     * @param Syy the diagonilized alignment tensor component
     * @param Szz the diagonilized alignment tensor component
     * @param currSearchDepth current search depth in the depth-first search tree
     * @param maxSearchDepth maximum depth of the depth-first search tree
     * @param ppS an array to store a solution (a sequence of phi and psi DOFs), i.e., a path from the root to a leaf of the depth-first search tree
     * @param firstPeptidePlane the first peptide plane of the fragment
     * @param phiPsiSolutionFile the file which stores the set of computed phi and psi sequences
     * @param weightInScoringFunction the relative weight used in scoring function
     * @param rightHand
     * @param dofsToBeModified the vector of dihedrals (DOFs) to be computed
     */
    public void recursiveDepthFirstSearchOfPhiPsiSolutionTree(final mySseInfo thisSseId, final Vector<myDipolarCoupling> phiTypeRdcSampled, final Vector<myDipolarCoupling> psiTypeRdcSampled,
            final Vector<myDipolarCoupling> phiTypeRdcVecWithMissingRdcsFilled, final Vector<myDipolarCoupling> psiTypeRdcVecWithMissingRdcsFilled,
            Matrix Rg, double Syy, double Szz, final int currSearchDepth, final int maxSearchDepth, double[] ppS, final myPeptidePlane firstPeptidePlane,
            Writer phiPsiSolutionFile, final double weightInScoringFunction, final boolean rightHand, final Vector<myDof> dofsToBeModified) {

        Vector<Double> phiVec = new Vector<Double>();
        Vector<Double> psiVec = new Vector<Double>();

        if (currSearchDepth == maxSearchDepth) { // We hit the basis of the recursion, i.e., we reached a leaf (NIL) node after computing all phi and psi values
            analyzeThisSolution(thisSseId, phiTypeRdcSampled, psiTypeRdcSampled, phiTypeRdcVecWithMissingRdcsFilled, psiTypeRdcVecWithMissingRdcsFilled,
                    ppS, firstPeptidePlane, phiPsiSolutionFile, weightInScoringFunction, dofsToBeModified, Syy, Szz);
        } else { //currSearchDepth < maxSearchDepth. So recurse down the tree.
            if (dofsToBeModified.elementAt(currSearchDepth).getDof() == myDihedral.PHI) { //this dof is PHI                                             
                phiVec = myForKin.computePhi(Rg, Syy, Szz, phiTypeRdcSampled.elementAt(currSearchDepth / 2), thisSseId.isHelix()); // We need phiType RDC for residue i to solve for Phi_i

                if (phiVec == null || phiVec.isEmpty()) {
                    if (__verbose) {
                        System.out.println("There exists no solution for phi for the sampled rdc " + phiTypeRdcSampled.elementAt(currSearchDepth / 2));
                    }
                }

                for (double thisPhi : phiVec) { //for each phi recurse down the tree searching for the psi
                    ppS[currSearchDepth] = thisPhi;
                    recursiveDepthFirstSearchOfPhiPsiSolutionTree(thisSseId, phiTypeRdcSampled, psiTypeRdcSampled, phiTypeRdcVecWithMissingRdcsFilled, psiTypeRdcVecWithMissingRdcsFilled,
                            Rg, Syy, Szz, currSearchDepth + 1, dofsToBeModified.size(), ppS, firstPeptidePlane, phiPsiSolutionFile, weightInScoringFunction, rightHand, dofsToBeModified);
                }
            } else if (dofsToBeModified.elementAt(currSearchDepth).getDof() == myDihedral.PSI) { //this dof is PSI                
                double phiAtThisResidue = ppS[currSearchDepth - 1];
                psiVec = myForKin.computePsi(Rg, Syy, Szz, psiTypeRdcSampled.elementAt(currSearchDepth / 2 + 1), phiAtThisResidue, thisSseId.isHelix()); // We need psiType rdc for residue (i+1) to solve for Psi_i

                if (psiVec == null || psiVec.isEmpty()) {
                    if (__verbose) {
                        System.out.println("There exists no solution for psi for the sampled rdc " + psiTypeRdcSampled.elementAt(currSearchDepth / 2 + 1));
                    }
                }

                for (double thisPsi : psiVec) { // for each psi recurse down the tree searching for the next phi
                    if (currSearchDepth / 2 + 1 > __max_depth) { // update the the maximum depth (in terms of residues) achieved
                        __max_depth = currSearchDepth / 2 + 1;
                    }
                    ppS[currSearchDepth] = thisPsi;
                    double thisPhi = ppS[currSearchDepth - 1];
                    //myPhiPsi pp = new myPhiPsi();

                    //System.out.println("thisPhi: " + thisPhi + "    thisPsi: " + thisPsi); 
                    Matrix newRg = myForKin.newRG__(thisPhi, thisPsi, Rg, rightHand); //compute a new rotation matrix rg
                    recursiveDepthFirstSearchOfPhiPsiSolutionTree(thisSseId, phiTypeRdcSampled, psiTypeRdcSampled, phiTypeRdcVecWithMissingRdcsFilled, psiTypeRdcVecWithMissingRdcsFilled,
                            newRg, Syy, Szz, currSearchDepth + 1, dofsToBeModified.size(), ppS, firstPeptidePlane, phiPsiSolutionFile, weightInScoringFunction, rightHand, dofsToBeModified);
                }
            } else {
                System.out.println("Error: Illegal DOF");
                System.exit(1);
            }
        }
    }

    /**
     * Analyze the solution wrt. a scoring function. Save the best solution,
     * increment the number of solutions found thus far. Print into files the best
     * solution fragment, the most recent fragment, and the solution phi psi sequence.
     * Currently, the analysis of a solution is implemented to minimize the
     * scoring function, and doing the steric check.
     *
     * @param thisSseId the secondary structure element boundary specification
     * @param phiTypeRdcSampled the sampled phi-defining RDC set
     * @param psiTypeRdcSampled the sampled psi-defining RDC set
     * @param phiTypeRdcVecWithMissingRdcsFilled the sampled phi-defining RDC set with missing RDCs filled
     * @param psiTypeRdcVecWithMissingRdcsFilled the sampled psi-defining RDC set with missing RDCs filled
     * @param ppS an array to store a solution (a sequence of phi and psi DOFs), i.e., a path from the root to a leaf of the depth-first search tree
     * @param firstPeptidePlane the first peptide plane of the fragment
     * @param phiPsiSolutionFile the file which stores the set of computed phi and psi sequences
     * @param weightInScoringFunction the relative weight used in scoring function
     * @param dofsToBeModified the vector of dihedrals (DOFs) to be computed
     * @param Syy the diagonilized alignment tensor component
     * @param Szz the diagonilized alignment tensor component
     */
    public void analyzeThisSolution(final mySseInfo thisSseId, final Vector<myDipolarCoupling> phiTypeRdcSampled, final Vector<myDipolarCoupling> psiTypeRdcSampled,
            final Vector<myDipolarCoupling> phiTypeRdcVecWithMissingRdcsFilled, final Vector<myDipolarCoupling> psiTypeRdcVecWithMissingRdcsFilled,
            double[] ppS, final myPeptidePlane firstPeptidePlane, Writer phiPsiSolutionFile, final double weightInScoringFunction, final Vector<myDof> dofsToBeModified,
            double Syy, double Szz) {
        // Note: (1) The parameter dofsToBeModified is not currently used. Perhaps it can be used afterwards.
        //       (2) ppS contains a solution
        if (ppS.length % 2 != 1) {
            System.out.println("Error: the length of the phi-psi chain should be odd");
            System.exit(1);
        }

        // construct a vector of phi and psi from the solution
        Vector<myPhiPsi> thisSolutionVec = new Vector<myPhiPsi>(ppS.length / 2 + 1); // initial capacity
        for (int i = 0; i < ppS.length - 1; i += 2) {
            thisSolutionVec.add(new myPhiPsi(thisSseId.getBeginResidueNumber() + i / 2, "ALA", ppS[i], ppS[i + 1]));
        }
        if (thisSseId.isHelix()) {
            thisSolutionVec.add(new myPhiPsi(thisSseId.getBeginResidueNumber() + (ppS.length - 1) / 2, "ALA", ppS[ppS.length - 1], Const.psiAveHelix));
        } else { // this SSE is a strand
            thisSolutionVec.add(new myPhiPsi(thisSseId.getBeginResidueNumber() + (ppS.length - 1) / 2, "ALA", ppS[ppS.length - 1], Const.psiAveBeta));
        }

        System.out.println("~~~~~~~~~~~~~ Begin Analysis of this Solution ~~~~~~~~~~~~~");
        try {
            // do all the finalization things
            __counter_for_total_number_of_solutions++;
            phiPsiSolutionFile.write("CounterForTotalNumberOfSolutions: " + __counter_for_total_number_of_solutions + "\n<Phi,Psi>: ");
            System.out.print("CounterForTotalNumberOfSolutions: " + __counter_for_total_number_of_solutions + "\n<Phi,Psi>: ");
            phiPsiSolutionFile.write(Arrays.toString(ppS) + '\n');
            System.out.println(Arrays.toString(ppS));

            myProtein thisFragment = myProtein.buildProteinBackboneFromDihedrals(thisSolutionVec, firstPeptidePlane);
            thisFragment.print("mostRecentFragment.pdb", false); // false indicates that this file is being overwritten instead of being appended
            System.out.println("Printing the solution PDB fragment");
            thisFragment.print();

            double phiRmsd = 0.0, psiRmsd = 0.0;

            // Compute phi rmsd and psi rmsd
            if (thisSseId.isHelix()) {
                for (myPhiPsi thisPP : thisSolutionVec) {
                    phiRmsd += (thisPP.getPhi() - Const.phiAveHelix) * (thisPP.getPhi() - Const.phiAveHelix);
                    psiRmsd += (thisPP.getPsi() - Const.psiAveHelix) * (thisPP.getPsi() - Const.psiAveHelix);
                }
            } else if (thisSseId.isStrand()) {
                for (myPhiPsi thisPP : thisSolutionVec) {
                    phiRmsd += (thisPP.getPhi() - Const.phiAveBeta) * (thisPP.getPhi() - Const.phiAveBeta);
                    psiRmsd += (thisPP.getPsi() - Const.psiAveBeta) * (thisPP.getPsi() - Const.psiAveBeta);
                }
            }

            phiRmsd = Math.sqrt(phiRmsd / thisSolutionVec.size());
            psiRmsd = Math.sqrt(psiRmsd / thisSolutionVec.size());
            System.out.println("phiRmsd: " + phiRmsd + "    psiRmsd: " + psiRmsd);

            /*
            // Compute the phiType rdc rmsd
            System.out.println(phiTypeRdcSampled.elementAt(0).getType() + " RDC: ");
            double phiTypeRdcRmsd = 0.0;
            for (int i = 0; i < thisSolutionVec.size(); i++) {
            //System.out.println((thisSseId.getBeginResidueNumber() + i) + ": " + phiTypeRdcSampled.elementAt(i).getRdc() + "  " + phiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc() + "  " + Math.abs(phiTypeRdcSampled.elementAt(i).getRdc() - phiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc()));
            System.out.println(phiTypeRdcSampled.elementAt(i).getResidueNumber() + ": " + phiTypeRdcSampled.elementAt(i).getRdc() + "  " + phiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc() + "  " + Math.abs(phiTypeRdcSampled.elementAt(i).getRdc() - phiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc()));
            phiTypeRdcRmsd += (phiTypeRdcSampled.elementAt(i).getRdc() - phiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc()) * (phiTypeRdcSampled.elementAt(i).getRdc() - phiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc());
            }
            phiTypeRdcRmsd = Math.sqrt(phiTypeRdcRmsd / thisSolutionVec.size());
            System.out.println(phiTypeRdcSampled.elementAt(0).getType() + " RDC RMSD: " + phiTypeRdcRmsd);

            // Compute the psiType rdc rmsd
            System.out.println(psiTypeRdcSampled.elementAt(0).getType() + " RDC: ");
            double psiTypeRdcRmsd = 0.0;
            for (int i = 0; i < thisSolutionVec.size(); i++) {
            System.out.println(psiTypeRdcSampled.elementAt(i).getResidueNumber() + ": " + psiTypeRdcSampled.elementAt(i).getRdc() + "  " + psiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc() + "  " + Math.abs(psiTypeRdcSampled.elementAt(i).getRdc() - psiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc()));
            psiTypeRdcRmsd += (psiTypeRdcSampled.elementAt(i).getRdc() - psiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc()) * (psiTypeRdcSampled.elementAt(i).getRdc() - psiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc());
            }
            psiTypeRdcRmsd = Math.sqrt(psiTypeRdcRmsd / thisSolutionVec.size());
            System.out.println(psiTypeRdcSampled.elementAt(0).getType() + " RDC RMSD: " + psiTypeRdcRmsd);
             */
            myDipolarCoupling.Type phiRdcType = phiTypeRdcVecWithMissingRdcsFilled.elementAt(0).getType();
            myDipolarCoupling.Type psiRdcType = psiTypeRdcVecWithMissingRdcsFilled.elementAt(0).getType();

            Matrix rotToPOF = Matrix.identity(3, 3);
            System.out.println(phiRdcType + " RDC:");
            double phiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(thisFragment, phiTypeRdcVecWithMissingRdcsFilled,
                    rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(phiRdcType.toString()));
            System.out.println(phiRdcType + " RDC rmsd: " + phiTypeRdcRmsd);

            myProtein thisFragmentFirstRemoved = new myProtein(thisFragment);
            //thisFragmentFirstRemoved.removeResidue(thisSseId.getBeginResidueNumber());

            System.out.println(psiRdcType + " RDC:");
            double psiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(thisFragmentFirstRemoved, psiTypeRdcVecWithMissingRdcsFilled,
                    rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(psiRdcType.toString()));
            System.out.println(psiRdcType + " RDC rmsd: " + psiTypeRdcRmsd);

            System.out.println("Phi/Psi: ");
            for (myPhiPsi thisPP : thisSolutionVec) {
                System.out.println(thisPP.getResidueNumber() + "   " + Math.toDegrees(thisPP.getPhi()) + "  " + Math.toDegrees(thisPP.getPsi()));
            }
/*
            double scoreTermFromRdcs = (phiTypeRdcRmsd / myConstantsForRdcs.DmaxScaled.get(phiRdcType.toString()) +
                    psiTypeRdcRmsd / myConstantsForRdcs.DmaxScaled.get(psiRdcType.toString())) + weightInScoringFunction * (phiRmsd + psiRmsd);
*/
            myPair<Double, Integer> squaredPhiRdcDev = myAlignmentTensorEstimator.computeRdcSumOfSquaredDeviation(thisFragment, phiTypeRdcVecWithMissingRdcsFilled,
                    rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(phiRdcType.toString()));

            myPair<Double, Integer> squaredPsiRdcDev = myAlignmentTensorEstimator.computeRdcSumOfSquaredDeviation(thisFragmentFirstRemoved, psiTypeRdcVecWithMissingRdcsFilled,
                    rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(psiRdcType.toString()));

            double phiPreFactor = myConstantsForRdcs.DmaxScaled.get(phiRdcType.toString());
            double psiPreFactor = myConstantsForRdcs.DmaxScaled.get(psiRdcType.toString());
            double scoreTermFromRdcs = squaredPhiRdcDev.first() / (phiPreFactor * phiPreFactor) + squaredPsiRdcDev.first() / (psiPreFactor * psiPreFactor);
            double totalNumberOfRdcs = squaredPhiRdcDev.second() + squaredPsiRdcDev.second();
            scoreTermFromRdcs = Math.sqrt(scoreTermFromRdcs / totalNumberOfRdcs);
            System.out.println("scoreTermFromRdcs: " + scoreTermFromRdcs);
            double scoreTermFromRdcsSaved = scoreTermFromRdcs;
            scoreTermFromRdcs += weightInScoringFunction * (phiRmsd + psiRmsd);           
            System.out.println("scoreTermFromRdcsAndPhiPsi: " + scoreTermFromRdcs);

            double clashScore = Double.MAX_VALUE;
            // Check for steric clash and return the clash score            
            myStericChecker sc = new myStericChecker(1.0, 0.8, 0.4);
            clashScore = sc.checkStericClash(thisFragment);            
            System.out.println("clashScore: " + clashScore);

            // combine the scores
            double jointScore = scoreTermFromRdcs + clashScore / 20.0;
            System.out.println("jointScore: " + jointScore);

            if (jointScore < __min_score ) { //we got a better fragment save it!                
                __min_score = jointScore;
                //__min_clash_score = clashScore;
                __best_fragment = new myProtein(thisFragment);

                Writer outputBestFragment = null;
                File bestFragmentFile = new File("bestFragment.pdb");
                if (bestFragmentFile.exists()) {
                    bestFragmentFile.delete();
                }
                outputBestFragment = new BufferedWriter(new FileWriter(bestFragmentFile));//, true)); //true for append mode

                outputBestFragment.write("REMARK 100 phiRmsd: " + phiRmsd + "    psiRmsd: " + psiRmsd + '\n');
                outputBestFragment.write("REMARK 100 " + phiTypeRdcSampled.elementAt(0).getType() + " RDC RMSD (including missing RDCs): " + phiTypeRdcRmsd + '\n');
                outputBestFragment.write("REMARK 100 " + psiTypeRdcSampled.elementAt(0).getType() + " RDC RMSD (including missing RDCs): " + psiTypeRdcRmsd + '\n');                
                outputBestFragment.write("REMARK 100 scoreTermFromRdcs: " + scoreTermFromRdcsSaved + '\n');
                outputBestFragment.write("REMARK 100 scoreTermFromRdcsAndPhiPsiDeviation: " + scoreTermFromRdcs + '\n');
                outputBestFragment.write("REMARK 100 clashScore: " + clashScore + '\n');
                outputBestFragment.write("REMARK 100 jointScore: " + jointScore + '\n');
                outputBestFragment.write("REMARK 100 Printing the phi and psi values below\n");
                for (myPhiPsi thisPP : thisSolutionVec) {
                    outputBestFragment.write("REMARK 100 " + thisPP.getResidueNumber() + "   " + Math.toDegrees(thisPP.getPhi()) + "  " + Math.toDegrees(thisPP.getPsi()) + '\n');
                }
                outputBestFragment.write("REMARK 100 Printing the corresponding PDB fragment\n");
                thisFragment.print(outputBestFragment);
                outputBestFragment.close();
            }
        } catch (IOException e) {
            System.out.println("An IOException is thrown while writing to the outputfile");
            e.printStackTrace();
            System.exit(1);
        }
        System.out.println("~~~~~~~~~~~~~~ End Analysis of this Solution ~~~~~~~~~~~~~~");
    }

    /**
     * This method invokes the recursive depth-first search based fragment
     * computation from the given set of sampled RDCs and input parameters.
     *
     * @param thisSseId the secondary structure element boundary specification
     * @param phiTypeRdcSampled the sampled phi-defining RDC set
     * @param psiTypeRdcSampled the sampled psi-defining RDC set
     * @param phiTypeRdcVecWithMissingRdcsFilled the sampled phi-defining RDC set with missing RDCs filled
     * @param psiTypeRdcVecWithMissingRdcsFilled the sampled psi-defining RDC set with missing RDCs filled
     * @param Rg the rotation matrix that specifies the ralative rotation of the first peptide plane (of the fragment) with respect to the principal order frame
     * @param Syy the diagonilized alignment tensor component
     * @param Szz the diagonilized alignment tensor component
     * @param firstPeptidePlane the first peptide plane of the fragment
     * @param phiPsiSolutionFile the file which stores the set of computed phi and psi sequences
     * @param weightInScoringFunction the relative weight used in scoring function
     * @param rightHand
     * @param hBondWeight
     * @param noeVec the set of NOEs used for packing the beta-sheet
     * @param partialSheetThusFar the partial sheet computed thus far
     */
    public void phiPsiSolutionTreeRootCallerForBetaStrand(final mySseInfo thisSseId,
            final Vector<myDipolarCoupling> phiTypeRdcSampled, final Vector<myDipolarCoupling> psiTypeRdcSampled,
            final Vector<myDipolarCoupling> phiTypeRdcVecWithMissingRdcsFilled, final Vector<myDipolarCoupling> psiTypeRdcVecWithMissingRdcsFilled,
            Matrix Rg, double Syy, double Szz, final myPeptidePlane firstPeptidePlane,
            Writer phiPsiSolutionFile, final double weightInScoringFunction, final boolean rightHand,
            final double hBondWeight, Vector<myNoe> noeVec, myProtein partialSheetThusFar) {

        Vector<myDof> dofsToBeComputed = getDofsToModify(thisSseId);
        // With phiType and psiType RDCs the psi angle of the very last residue
        // cannot be computed as the psiType RDC data is not available for it.
        // Therefore, we remove the last psi from the dofsToBeComputed.
        dofsToBeComputed.remove(dofsToBeComputed.size() - 1);

        double[] ppS = new double[dofsToBeComputed.size()];
        int initialSearchDepth = 0;
        int maxSearchDepth = dofsToBeComputed.size();

        recursiveDepthFirstSearchOfPhiPsiSolutionTreeForBetaStrand(thisSseId, phiTypeRdcSampled, psiTypeRdcSampled, phiTypeRdcVecWithMissingRdcsFilled, psiTypeRdcVecWithMissingRdcsFilled,
            Rg, Syy, Szz, initialSearchDepth, maxSearchDepth, ppS, firstPeptidePlane, phiPsiSolutionFile, weightInScoringFunction, rightHand, dofsToBeComputed,
            hBondWeight, noeVec, partialSheetThusFar);
    }

    /**
     * Do depth-first search of the analytic solution tree to compute the set of all
     * possible phi, psi sequences from which a set of possible fragments can be
     * constructed.
     *
     * @param thisSseId the secondary structure element boundary specification
     * @param phiTypeRdcSampled the sampled phi-defining RDC set
     * @param psiTypeRdcSampled the sampled psi-defining RDC set
     * @param phiTypeRdcVecWithMissingRdcsFilled the sampled phi-defining RDC set with missing RDCs filled
     * @param psiTypeRdcVecWithMissingRdcsFilled the sampled psi-defining RDC set with missing RDCs filled
     * @param Rg the rotation matrix that specifies the ralative rotation of the first peptide plane (of the fragment) with respect to the principal order frame
     * @param Syy the diagonilized alignment tensor component
     * @param Szz the diagonilized alignment tensor component
     * @param currSearchDepth current search depth in the depth-first search tree
     * @param maxSearchDepth maximum depth of the depth-first search tree
     * @param ppS an array to store a solution (a sequence of phi and psi DOFs), i.e., a path from the root to a leaf of the depth-first search tree
     * @param firstPeptidePlane the first peptide plane of the fragment
     * @param phiPsiSolutionFile the file which stores the set of computed phi and psi sequences
     * @param weightInScoringFunction the relative weight used in scoring function
     * @param rightHand
     * @param dofsToBeModified the vector of dihedrals (DOFs) to be computed
     * @param hBondWeight
     * @param noeVec the set of NOEs used for packing the beta-sheet
     * @param partialSheetThusFar the partial sheet computed thus far
     */
    public void recursiveDepthFirstSearchOfPhiPsiSolutionTreeForBetaStrand(final mySseInfo thisSseId, final Vector<myDipolarCoupling> phiTypeRdcSampled, final Vector<myDipolarCoupling> psiTypeRdcSampled,
            final Vector<myDipolarCoupling> phiTypeRdcVecWithMissingRdcsFilled, final Vector<myDipolarCoupling> psiTypeRdcVecWithMissingRdcsFilled,
            Matrix Rg, double Syy, double Szz, final int currSearchDepth, final int maxSearchDepth, double[] ppS, final myPeptidePlane firstPeptidePlane,
            Writer phiPsiSolutionFile, final double weightInScoringFunction, final boolean rightHand, final Vector<myDof> dofsToBeModified,
            final double hBondWeight, Vector<myNoe> noeVec, myProtein partialSheetThusFar) {

        Vector<Double> phiVec = new Vector<Double>();
        Vector<Double> psiVec = new Vector<Double>();

        if (currSearchDepth == maxSearchDepth) { // We hit the basis of the recursion, i.e., we reached a leaf (NIL) node after computing all phi and psi values
            analyzeThisSolutionForBetaStrand(thisSseId, phiTypeRdcSampled, psiTypeRdcSampled, phiTypeRdcVecWithMissingRdcsFilled, psiTypeRdcVecWithMissingRdcsFilled,
                    ppS, firstPeptidePlane, phiPsiSolutionFile, weightInScoringFunction, dofsToBeModified, Syy, Szz,
                    hBondWeight, noeVec, partialSheetThusFar);
        } else { //currSearchDepth < maxSearchDepth. So recurse down the tree.
            if (dofsToBeModified.elementAt(currSearchDepth).getDof() == myDihedral.PHI) { //this dof is PHI
                phiVec = myForKin.computePhi(Rg, Syy, Szz, phiTypeRdcSampled.elementAt(currSearchDepth / 2), thisSseId.isHelix()); // We need phiType RDC for residue i to solve for Phi_i

                if (phiVec == null || phiVec.isEmpty()) {
                    if (__verbose) {
                        System.out.println("There exists no solution for phi for the sampled rdc " + phiTypeRdcSampled.elementAt(currSearchDepth / 2));
                    }
                }

                for (double thisPhi : phiVec) { //for each phi recurse down the tree searching for the psi
                    ppS[currSearchDepth] = thisPhi;
                    recursiveDepthFirstSearchOfPhiPsiSolutionTreeForBetaStrand(thisSseId, phiTypeRdcSampled, psiTypeRdcSampled, phiTypeRdcVecWithMissingRdcsFilled, psiTypeRdcVecWithMissingRdcsFilled,
                            Rg, Syy, Szz, currSearchDepth + 1, dofsToBeModified.size(), ppS, firstPeptidePlane, phiPsiSolutionFile, weightInScoringFunction, rightHand, dofsToBeModified,
                            hBondWeight, noeVec, partialSheetThusFar);
                }
            } else if (dofsToBeModified.elementAt(currSearchDepth).getDof() == myDihedral.PSI) { //this dof is PSI
                double phiAtThisResidue = ppS[currSearchDepth - 1];
                psiVec = myForKin.computePsi(Rg, Syy, Szz, psiTypeRdcSampled.elementAt(currSearchDepth / 2 + 1), phiAtThisResidue, thisSseId.isHelix()); // We need psiType rdc for residue (i+1) to solve for Psi_i

                if (psiVec == null || psiVec.isEmpty()) {
                    if (__verbose) {
                        System.out.println("There exists no solution for psi for the sampled rdc " + psiTypeRdcSampled.elementAt(currSearchDepth / 2 + 1));
                    }
                }

                for (double thisPsi : psiVec) { // for each psi recurse down the tree searching for the next phi
                    if (currSearchDepth / 2 + 1 > __max_depth) { // update the the maximum depth (in terms of residues) achieved
                        __max_depth = currSearchDepth / 2 + 1;
                    }
                    ppS[currSearchDepth] = thisPsi;
                    double thisPhi = ppS[currSearchDepth - 1];
                    //myPhiPsi pp = new myPhiPsi();

                    //System.out.println("thisPhi: " + thisPhi + "    thisPsi: " + thisPsi); 
                    Matrix newRg = myForKin.newRG__(thisPhi, thisPsi, Rg, rightHand); //compute a new rotation matrix rg
                    recursiveDepthFirstSearchOfPhiPsiSolutionTreeForBetaStrand(thisSseId, phiTypeRdcSampled, psiTypeRdcSampled, phiTypeRdcVecWithMissingRdcsFilled, psiTypeRdcVecWithMissingRdcsFilled,
                            newRg, Syy, Szz, currSearchDepth + 1, dofsToBeModified.size(), ppS, firstPeptidePlane, phiPsiSolutionFile, weightInScoringFunction, rightHand, dofsToBeModified,
                            hBondWeight, noeVec, partialSheetThusFar);
                }
            } else {
                System.out.println("Error: Illegal DOF");
                System.exit(1);
            }
        }
    }

    /**
     * Analyze the solution wrt. a scoring function. Save the best solution,
     * increment the number of solutions found thus far. Print into files the best
     * solution fragment, the most recent fragment, and the solution phi psi sequence.
     * Currently, the analysis of a solution is implemented to minimize the
     * scoring function, and doing the steric check.
     *
     * @param thisSseId the secondary structure element boundary specification
     * @param phiTypeRdcSampled the sampled phi-defining RDC set
     * @param psiTypeRdcSampled the sampled psi-defining RDC set
     * @param phiTypeRdcVecWithMissingRdcsFilled the sampled phi-defining RDC set with missing RDCs filled
     * @param psiTypeRdcVecWithMissingRdcsFilled the sampled psi-defining RDC set with missing RDCs filled
     * @param ppS an array to store a solution (a sequence of phi and psi DOFs), i.e., a path from the root to a leaf of the depth-first search tree
     * @param firstPeptidePlane the first peptide plane of the fragment
     * @param phiPsiSolutionFile the file which stores the set of computed phi and psi sequences
     * @param weightInScoringFunction the relative weight used in scoring function
     * @param dofsToBeModified the vector of dihedrals (DOFs) to be computed
     * @param Syy the diagonilized alignment tensor component
     * @param Szz the diagonilized alignment tensor component
     * @param hBondWeight
     * @param noeVec the set of NOEs used for packing the beta-sheet
     * @param partialSheetThusFar the partial sheet computed thus far
     */
    public void analyzeThisSolutionForBetaStrand(final mySseInfo thisSseId, final Vector<myDipolarCoupling> phiTypeRdcSampled, final Vector<myDipolarCoupling> psiTypeRdcSampled,
            final Vector<myDipolarCoupling> phiTypeRdcVecWithMissingRdcsFilled, final Vector<myDipolarCoupling> psiTypeRdcVecWithMissingRdcsFilled,
            double[] ppS, final myPeptidePlane firstPeptidePlane, Writer phiPsiSolutionFile, final double weightInScoringFunction, final Vector<myDof> dofsToBeModified,
            double Syy, double Szz,
            final double hBondWeight, Vector<myNoe> noeVec, myProtein partialSheetThusFar) {
        // Note: (1) The parameter dofsToBeModified is not currently used. Perhaps it can be used afterwards.
        //       (2) ppS contains a solution
        if (ppS.length % 2 != 1) {
            System.out.println("Error: the length of the phi-psi chain should be odd");
            System.exit(1);
        }

        // construct a vector of phi and psi from the solution
        Vector<myPhiPsi> thisSolutionVec = new Vector<myPhiPsi>(ppS.length / 2 + 1); // initial capacity
        for (int i = 0; i < ppS.length - 1; i += 2) {
            thisSolutionVec.add(new myPhiPsi(thisSseId.getBeginResidueNumber() + i / 2, "ALA", ppS[i], ppS[i + 1]));
        }
        if (thisSseId.isHelix()) {
            thisSolutionVec.add(new myPhiPsi(thisSseId.getBeginResidueNumber() + (ppS.length - 1) / 2, "ALA", ppS[ppS.length - 1], Const.psiAveHelix));
        } else { // this SSE is a strand
            thisSolutionVec.add(new myPhiPsi(thisSseId.getBeginResidueNumber() + (ppS.length - 1) / 2, "ALA", ppS[ppS.length - 1], Const.psiAveBeta));
        }

        System.out.println("~~~~~~~~~~~~~ Begin Analysis of this Solution ~~~~~~~~~~~~~");
        try {
            // do all the finalization things
            __counter_for_total_number_of_solutions++;
            phiPsiSolutionFile.write("CounterForTotalNumberOfSolutions: " + __counter_for_total_number_of_solutions + "\n<Phi,Psi>: ");
            System.out.print("CounterForTotalNumberOfSolutions: " + __counter_for_total_number_of_solutions + "\n<Phi,Psi>: ");
            phiPsiSolutionFile.write(Arrays.toString(ppS) + '\n');
            System.out.println(Arrays.toString(ppS));

            myProtein thisFragment = myProtein.buildProteinBackboneFromDihedrals(thisSolutionVec, firstPeptidePlane);
            thisFragment.print("mostRecentFragment.pdb", false); // false indicates that this file is being overwritten instead of being appended
            System.out.println("Printing the solution PDB fragment");
            thisFragment.print();

            double phiRmsd = 0.0, psiRmsd = 0.0;

            // Compute phi rmsd and psi rmsd
            if (thisSseId.isHelix()) {
                for (myPhiPsi thisPP : thisSolutionVec) {
                    phiRmsd += (thisPP.getPhi() - Const.phiAveHelix) * (thisPP.getPhi() - Const.phiAveHelix);
                    psiRmsd += (thisPP.getPsi() - Const.psiAveHelix) * (thisPP.getPsi() - Const.psiAveHelix);
                }
            } else if (thisSseId.isStrand()) {
                for (myPhiPsi thisPP : thisSolutionVec) {
                    phiRmsd += (thisPP.getPhi() - Const.phiAveBeta) * (thisPP.getPhi() - Const.phiAveBeta);
                    psiRmsd += (thisPP.getPsi() - Const.psiAveBeta) * (thisPP.getPsi() - Const.psiAveBeta);
                }
            }

            phiRmsd = Math.sqrt(phiRmsd / thisSolutionVec.size());
            psiRmsd = Math.sqrt(psiRmsd / thisSolutionVec.size());
            System.out.println("phiRmsd: " + phiRmsd + "    psiRmsd: " + psiRmsd);

            /*
            // Compute the phiType rdc rmsd
            System.out.println(phiTypeRdcSampled.elementAt(0).getType() + " RDC: ");
            double phiTypeRdcRmsd = 0.0;
            for (int i = 0; i < thisSolutionVec.size(); i++) {
            //System.out.println((thisSseId.getBeginResidueNumber() + i) + ": " + phiTypeRdcSampled.elementAt(i).getRdc() + "  " + phiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc() + "  " + Math.abs(phiTypeRdcSampled.elementAt(i).getRdc() - phiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc()));
            System.out.println(phiTypeRdcSampled.elementAt(i).getResidueNumber() + ": " + phiTypeRdcSampled.elementAt(i).getRdc() + "  " + phiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc() + "  " + Math.abs(phiTypeRdcSampled.elementAt(i).getRdc() - phiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc()));
            phiTypeRdcRmsd += (phiTypeRdcSampled.elementAt(i).getRdc() - phiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc()) * (phiTypeRdcSampled.elementAt(i).getRdc() - phiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc());
            }
            phiTypeRdcRmsd = Math.sqrt(phiTypeRdcRmsd / thisSolutionVec.size());
            System.out.println(phiTypeRdcSampled.elementAt(0).getType() + " RDC RMSD: " + phiTypeRdcRmsd);

            // Compute the psiType rdc rmsd
            System.out.println(psiTypeRdcSampled.elementAt(0).getType() + " RDC: ");
            double psiTypeRdcRmsd = 0.0;
            for (int i = 0; i < thisSolutionVec.size(); i++) {
            System.out.println(psiTypeRdcSampled.elementAt(i).getResidueNumber() + ": " + psiTypeRdcSampled.elementAt(i).getRdc() + "  " + psiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc() + "  " + Math.abs(psiTypeRdcSampled.elementAt(i).getRdc() - psiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc()));
            psiTypeRdcRmsd += (psiTypeRdcSampled.elementAt(i).getRdc() - psiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc()) * (psiTypeRdcSampled.elementAt(i).getRdc() - psiTypeRdcVecWithMissingRdcsFilled.elementAt(i).getRdc());
            }
            psiTypeRdcRmsd = Math.sqrt(psiTypeRdcRmsd / thisSolutionVec.size());
            System.out.println(psiTypeRdcSampled.elementAt(0).getType() + " RDC RMSD: " + psiTypeRdcRmsd);
             */
            myDipolarCoupling.Type phiRdcType = phiTypeRdcVecWithMissingRdcsFilled.elementAt(0).getType();
            myDipolarCoupling.Type psiRdcType = psiTypeRdcVecWithMissingRdcsFilled.elementAt(0).getType();

            Matrix rotToPOF = Matrix.identity(3, 3);
            System.out.println(phiRdcType + " RDC:");
            double phiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(thisFragment, phiTypeRdcVecWithMissingRdcsFilled,
                    rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(phiRdcType.toString()));
            System.out.println(phiRdcType + " RDC rmsd: " + phiTypeRdcRmsd);

            myProtein thisFragmentFirstRemoved = new myProtein(thisFragment);
            //thisFragmentFirstRemoved.removeResidue(thisSseId.getBeginResidueNumber());

            System.out.println(psiRdcType + " RDC:");
            double psiTypeRdcRmsd = myAlignmentTensorEstimator.computeRdcRmsd(thisFragmentFirstRemoved, psiTypeRdcVecWithMissingRdcsFilled,
                    rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(psiRdcType.toString()));
            System.out.println(psiRdcType + " RDC rmsd: " + psiTypeRdcRmsd);

            System.out.println("Phi/Psi: ");
            for (myPhiPsi thisPP : thisSolutionVec) {
                System.out.println(thisPP.getResidueNumber() + "   " + Math.toDegrees(thisPP.getPhi()) + "  " + Math.toDegrees(thisPP.getPsi()));
            }
/*
            double scoreTermFromRdcs = (phiTypeRdcRmsd / myConstantsForRdcs.DmaxScaled.get(phiRdcType.toString()) +
                    psiTypeRdcRmsd / myConstantsForRdcs.DmaxScaled.get(psiRdcType.toString())) + weightInScoringFunction * (phiRmsd + psiRmsd);
*/            
            myPair<Double, Integer> squaredPhiRdcDev = myAlignmentTensorEstimator.computeRdcSumOfSquaredDeviation(thisFragment, phiTypeRdcVecWithMissingRdcsFilled,
                    rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(phiRdcType.toString()));

            myPair<Double, Integer> squaredPsiRdcDev = myAlignmentTensorEstimator.computeRdcSumOfSquaredDeviation(thisFragmentFirstRemoved, psiTypeRdcVecWithMissingRdcsFilled,
                    rotToPOF, -(Syy + Szz), Syy, Szz, myConstantsForRdcs.DmaxScaled.get(psiRdcType.toString()));

            double phiPreFactor = myConstantsForRdcs.DmaxScaled.get(phiRdcType.toString());
            double psiPreFactor = myConstantsForRdcs.DmaxScaled.get(psiRdcType.toString());
            double scoreTermFromRdcs = squaredPhiRdcDev.first() / (phiPreFactor * phiPreFactor) + squaredPsiRdcDev.first() / (psiPreFactor * psiPreFactor);
            double totalNumberOfRdcs = squaredPhiRdcDev.second() + squaredPsiRdcDev.second();
            scoreTermFromRdcs = Math.sqrt(scoreTermFromRdcs / totalNumberOfRdcs);
            System.out.println("scoreTermFromRdcs: " + scoreTermFromRdcs);
            scoreTermFromRdcs += weightInScoringFunction * (phiRmsd + psiRmsd);
            System.out.println("scoreTermFromRdcsAndPhiPsi: " + scoreTermFromRdcs);            

            myNewPacker pk = new myNewPacker();
            pk.setGlobalBestOrBestFirst(true);
            myTriple<myProtein, myProtein, Double> scoreFromCenterFit = pk.__packUsingNoes__(partialSheetThusFar, thisFragment, noeVec, null);//pk.centerFit__(partialSheetThusFar, thisFragment, noeVec);

            if (scoreFromCenterFit.first() == null || scoreFromCenterFit.second() == null) {
                System.out.println("Warning: In AnalyzeSolutionForBetaStrand() the fragment is null null coz score is >= 5");
                System.out.println("Therefore, discarding this solution and returning from analyzeThisSolutionForBetaStrand");
                System.out.println("~~~~~~~~~~~~~~ End Analysis of this Solution ~~~~~~~~~~~~~~");
                return;
            }

            double scoreTermFromSheetPacking = scoreFromCenterFit.third();
            System.out.println("scoreTermFromSheetPacking: " + scoreTermFromSheetPacking);

            double jointScore = scoreTermFromRdcs + scoreTermFromSheetPacking;
            System.out.println("jointScore: " + jointScore);

            if (jointScore < __min_score) { //we got a better fragment save it!                
                __min_score = jointScore;
                //__min_clash_score = clashScore;
                __best_fragment = new myProtein(scoreFromCenterFit.second());

                Writer outputBestFragment = null;
                File bestFragmentFile = new File("bestFragment.pdb");
                if (bestFragmentFile.exists()) {
                    bestFragmentFile.delete();
                }
                outputBestFragment = new BufferedWriter(new FileWriter(bestFragmentFile));//, true)); //true for append mode

                outputBestFragment.write("REMARK 100 phiRmsd: " + phiRmsd + "    psiRmsd: " + psiRmsd + '\n');
                outputBestFragment.write("REMARK 100 " + phiTypeRdcSampled.elementAt(0).getType() + " RDC RMSD (including missing RDCs): " + phiTypeRdcRmsd + '\n');
                outputBestFragment.write("REMARK 100 " + psiTypeRdcSampled.elementAt(0).getType() + " RDC RMSD (including missing RDCs): " + psiTypeRdcRmsd + '\n');
                outputBestFragment.write("REMARK 100 scoreTermFromSheetPacking: " + scoreTermFromSheetPacking + '\n');
                outputBestFragment.write("REMARK 100 scoreTermFromRdcsAndPhiPsiDeviation: " + scoreTermFromRdcs + '\n');
                outputBestFragment.write("REMARK 100 jointScore: " + jointScore + '\n');               
                outputBestFragment.write("REMARK 100 Printing the phi and psi values below\n");
                for (myPhiPsi thisPP : thisSolutionVec) {
                    outputBestFragment.write("REMARK 100 " + thisPP.getResidueNumber() + "   " + Math.toDegrees(thisPP.getPhi()) + "  " + Math.toDegrees(thisPP.getPsi()) + '\n');
                }
                outputBestFragment.write("REMARK 100 Printing the corresponding PDB fragment\n");
                //thisFragment.print(outputBestFragment);
                myProtein mergedProtein = myProtein.cloneAndMergeTwoProteinFragments(scoreFromCenterFit.first(), scoreFromCenterFit.second());
                mergedProtein.print(outputBestFragment);
                outputBestFragment.close();
            }
        } catch (IOException e) {
            System.out.println("An IOException is thrown while writing to the outputfile");
            e.printStackTrace();
            System.exit(1);
        }
        System.out.println("~~~~~~~~~~~~~~ End Analysis of this Solution ~~~~~~~~~~~~~~");
    }
}
